#!/usr/bin/perl

package GenomeData;
sub new {
	my($class, %options) = @_;
	## Singleton
	if ($THE_DATA) {
		return $THE_DATA;
	}
	$THE_DATA = {};
	if ($options{datapath}) {
		$THE_DATA->{datapath} = $options{datapath};
	} else {
		$THE_DATA->{datapath} = "../data";
	}
	return bless $THE_DATA, $class;
}
sub getInstance {
	## Do not create even if $THE_DATA is not defined
	return $THE_DATA;
}
sub read {
	my($class, $genomes, %options) = @_;
	my($this) = $class->new(%options);
	my(@species, @out_species);
	open(G, $genomes) || die "Can't open $genomes\n";
	while(<G>){
		chomp;
		my($sp,$name,$length,$order,$origin) = split;
		$this->{length}->{$sp} = $length;
		$this->{name}->{$sp} = $name;
		$this->{order}->{$sp} = $order;
		$this->{origin}->{$sp} = $origin;
		if (! $options{genome_only}) {
			$this->read_genetab($sp);
		}
		push(@species, $sp);
		push(@out_species, $sp) if ($order);
	}
	close(G);
	@out_species = sort {$this->{order}->{$a}<=>
				$this->{order}->{$b}} @out_species;
	$this->{species} = \@species;
	$this->{spnum} = scalar(@species);
	$this->{out_species} = \@out_species;
	$this;
}
sub set_species {
	my($this, $species, $out_species) = @_;
	$this->{species} = $species;
	$this->{spnum} = scalar(@{$species});
	if ($out_species) {
		$this->{out_species} = $out_species;
	} else {
		$this->{out_species} = $species;
	}
}
sub read_genetab {
	my($this, @SPLIST) = @_;
#	my($GENEDIR) = "$this->{datapath}/gene";
	my($GENEDIR) = "$this->{datapath}";
	my($ln);
	@SPLIST = split(/,/, $::orgs) if (! @SPLIST);
	foreach my $sp0 (@SPLIST) {
		open(F, "$GENEDIR/$sp0.txt") || die "Can't open $GENEDIR/$sp0.txt";
		while (<F>) {
			chomp;
			next if ($ln++==0);
			next if (/^$/);
			my ($sp,$name,$gene,$from,$to,$dir,$type,$descr) = split(/\t/);
			next if ($type ne 'CDS');
			my $pos = $this->pos_calc($sp,$from,$to);
			$this->{data}->{$sp}->{$name} = {
				from=>$from, to=>$to, dir=>$dir,
				pos=>$pos, gene=>$gene, chrid=>$chrid,
			};
		}
		close(F);
	}
}

sub pos_calc {
	my($this, $sp, $from, $to, $unit) = @_;
	my $pos = ($from + $to) /2;
	die "$0: Error: zero length: $sp\n" if ($this->{length}->{$sp} == 0);
	$pos = ($pos - $this->{origin}->{$sp}) % $this->{length}->{$sp};
	if ($unit eq 'degree') {
		## $unit = 'degree';
		if ($pos < $this->{length}->{$sp} / 2) {
			$pos = $pos / $this->{length}->{$sp} * 360;
		} else {
			$pos = ($pos / $this->{length}->{$sp} - 1) * 360;
		}
	} else {
		if ($pos >= $this->{length}->{$sp} / 2) {
			$pos = $pos - $this->{length}->{$sp};
		}
		$pos /= 1000;
	}
	$pos = sprintf("%.1f", $pos);
	return $pos;
}
sub get {
	my($this, @opt) = @_;
	$this->getGene(@opt);
}
sub spnum {
	my($this) = @_;
	$this->{spnum} ? $this->{spnum} : scalar(@{ $this->{species} });
}

sub getGene {
	my($this, $sp, $name) = @_;
	return $this->{data}->{$sp}->{$name};
}
sub setSpGroup {
	my($this, $spGroup) = @_;
	foreach my $splist (split(/,/, $spGroup)) {
		my @spnames = split(/:/, $splist);
		foreach $sp (@spnames) {
			$this->{SpGroup}->{$sp} = $spnames[0];
		}
	}
}
sub getSpGroup {
	my($this, $sp) = @_;
	return $this->{SpGroup}->{$sp} ? $this->{SpGroup}->{$sp} : $sp;
}
sub setSpIndex {
	my($this) = @_;
	my(%SpIdx, @SpList, $i);
	my($sp, $sp0);
	$i = 0;
	foreach $sp (@{ $this->{species} }) {
		if ($this->getSpGroup($sp) eq $sp) {
			$SpIdx{$sp} = $i;
			$SpList[$i] = $sp;
			$i++;
		}
	}
	$this->{spnum} = $i;
	foreach $sp (@{ $this->{species} }) {
		if (($sp0 = $this->getSpGroup($sp)) ne $sp) {
			$SpIdx{$sp} = $SpIdx{ $sp0 };
		}
	}
	$this->{SpIdx} = \%SpIdx;
	$this->{SpList} = \@SpList;
}
sub getSpIndex {
	my($this, $sp) = @_;
	$this->{SpIdx}->{$sp};
}
sub getPhyloPat {
	my($this, $species, $opt) = @_;
	my $vect = BitVector->new(0,$this->spnum);
	foreach $sp (@{$species}) {
		$vect->add( $this->getSpIndex($sp) );
	}
	$vect->toString;
}
sub countSpecies {
	my($this, $species, %opt) = @_;
	my(%tmpFoundIn, %tmpFoundOut);
	foreach $sp (@{$species}) {
		if ($this->isOutGroup($sp)) {
			$tmpFoundOut{$this->getSpGroup($sp)} = 1;
		} else {
			$tmpFoundIn{$this->getSpGroup($sp)} = 1;
		}
	}
	if ($opt{separate_outgrp}){
		return (scalar(keys %tmpFoundIn), scalar(keys %tmpFoundOut));
	} elsif ($opt{without_outgrp}){
		return scalar(keys %tmpFoundIn);
	} else {
		return scalar(keys %tmpFoundIn) + scalar(keys %tmpFoundOut);
	}
}
sub setOutGroup {
	my($this, $outGroup) = @_;
	foreach $sp (split(/,/, $outGroup)) {
		$this->{OutGroup}->{$sp} = 1;
	}
}
sub isOutGroup {
	my($this, $sp) = @_;
	return $this->{OutGroup}->{$sp};
}
sub isOutGroupIndex {
	my($this, $idx) = @_;
	return $this->isOutGroup( $this->{SpList}->[$idx] );
}
###########################################################
package BitVector;
sub new {
	my($class, $vect, $length) = @_;
	my($this) = {};
	$this->{vect} = $vect;
	$this->{length} = $length;
	return bless $this, $class;
}
sub add {
	my($this, $idx) = @_;
	my($add_vect)= (1<<$idx);
	if ( ($this->{vect} & $add_vect) == 0 ) {
		$this->{cnt}++;
		$this->{vect} |= $add_vect;
	}
}
sub and {
	my($this, $other) = @_;
	BitVector->new($this->{vect} & $other->{vect});
}
sub or {
	my($this, $other) = @_;
	BitVector->new($this->{vect} | $other->{vect});
}
sub contain {
	my($this, $other) = @_;
	my $or_vect = ($this->{vect} | $other->{vect});
	if ($or_vect == $this->{vect}) {
		return 1;
	} elsif ($or_vect == $other->{vect}) {
		return -1;
	}
}
sub toString {
	my($this) = @_;
	my $vect = $this->{vect};
	my $bitstr;
	my $i;
	while ($vect) {
		$c = $vect & 1;
		$bitstr .= "$c";
		$vect >>= 1;
		last if (++$i > $this->{length});
	}
	while ($i++ < $this->{length}) {
		$bitstr .= "0";
	}
	$bitstr;
}
sub print {
	my($this) = @_;
	print $this->toString, "\n";
}

###########################################################
package FuncCat;
$FuncCat::GeneFuncFname = "geneFunction";
$FuncCat::FuncInfoFname = "function";
$FuncCat::FuncColorFname = "colorTab";
sub new {
	my($class, $dir, $db, %opt) = @_;
	my($this) = {};
	$this->{dir} = $dir;
	$this->{db} = $db;
	bless $this, $class;
	if ($db) {
		$this->readFuncInfo($sp, $db);
		$this->readFuncColor($sp, $db);
	}
	if ($opt{default_color}) {
		$this->{default_color} = $opt{default_color};
	}
	return $this;
}
sub readGeneFunc {
	my($this, $sp, $db) = @_;
	$db = $this->{db} if (! $db);
	if (! open(F, "$this->{dir}/${FuncCat::GeneFuncFname}.$sp.$db") ) {
		warn "Can't open funcfile for $sp.$db\n";
		return 0;
	}
	while(<F>) {
		chomp;
		($sp,$name,$db,$func) = split(/\t/);
		push(@{$this->{Func}->{$db}->{$sp}->{$name}}, $func);
	}
	close(F);
}
sub getGeneFunc {
	my($this, $name, $db, $idx) = @_;
	my($sp, $gene) = split(/:/, $name);
	$db = $this->{db} if (! $db);
	if (! $this->{Func}->{$db}->{$sp}) {
		$this->readGeneFunc($sp, $db);
	}
	if (! $this->{Func}->{$db}->{$sp}->{$gene}) {
		$this->{Func}->{$db}->{$sp}->{$gene}
				= $this->getDefaultFunc($db);
	}
	if (defined $idx) {
		return $this->{Func}->{$db}->{$sp}->{$gene}->[$idx];
	} else {
		## arrayref
		return $this->{Func}->{$db}->{$sp}->{$gene};
	}
}
sub getGeneFunc1 {
	my($this, $name, $db) = @_;
	return $this->getGeneFunc($name,$db, 0);
}
sub readFuncInfo {
	my($this, $db) = @_;
	$db = $this->{db} if (! $db);
	my $funcfile = "$this->{dir}/${FuncCat::FuncInfoFname}.$db";
	if (! open(F, $funcfile) ) {
		warn "Can't open funcinfo file of $db ($funcfile)\n";
		return 0;
	}
	while(<F>) {
		chomp;
		if (/^#(\S+)\s+(\S+)/) {
			$var = $1; $val = $2;
			if ($var =~ /^default$/) {
				$this->{Variable}->{$db}->{$var} = $val;
			}
		}
		($db,$func,$name,$code) = split(/\t/);
		$this->{FuncInfo}->{$db}->{$func}->{name} = $name;
		$this->{FuncInfo}->{$db}->{$func}->{code} = $code;
	}
	close(F);
}
sub getFuncInfo {
	my($this, $func, $infotype, $db) = @_;
	$db = $this->{db} if (! $db);
	if (! $this->{FuncInfo}->{$db}) {
		$this->readFuncInfo($sp, $db);
		$this->readFuncColor($sp, $db);
	}
	if ($infotype) {
		return $this->{FuncInfo}->{$db}->{$func}->{$infotype};
	} else {
		return $this->{FuncInfo}->{$db}->{$func};
	}
}
sub listColorDefinedFuncID {
	my($this, $db) = @_;
	push(@{$this->{DefColors}}, $func);
}
sub listFuncID {
	my($this, $db, %opt) = @_;
	$db = $this->{db} if (! $db);
	my @funcIDs =  (sort keys %{ $this->{FuncInfo}->{$db} });
	if ($opt{colordef}) {
		my @newFuncIDs;
		foreach $func (@funcIDs) {
			if (defined $this->{FuncInfo}->{$db}->{$func}->{color}){
				push(@newFuncIDs, $func);
			}
		}
		return @newFuncIDs;
	} else {
		return@funcIDs;
	}
}
sub getDefaultFunc {
	my($this, $db) = @_;
	$db = $this->{db} if (! $db);
	$this->{Variable}->{$db}->{default};
}
sub readFuncColor {
	my($this, $db) = @_;
	$db = $this->{db} if (! $db);
	my $funcfile="$this->{dir}/${FuncCat::FuncColorFname}.$db";
	if (! open(F, $funcfile) ) {
		warn "Can't open funccolor file of $db ($funcfile)\n";
		return 0;
	}
	while(<F>) {
		chomp;
		($db,$func,$color) = split(/\t/);
		$this->{FuncInfo}->{$db}->{$func}->{color} = $color;
	}
	close(F);
}
sub getFuncColor {
	my($this, $func, $db) = @_;
	$db = $this->{db} if (! $db);
	if (! $this->{FuncInfo}->{$db}) {
		$this->readFuncColor($sp, $db);
	}
	if ( $this->{FuncInfo}->{$db}->{$func}->{color} ) {
		return $this->{FuncInfo}->{$db}->{$func}->{color};
	} else {
		@func = split(/\./, $func);
		for ($n = $#func - 1; $n >= 0; $n--) {
			$func = join(".",@func[0..$n]);
print STDERR ">>>$func\n";
			if ( $this->{FuncInfo}->{$db}->{$func}->{color} ) {
				return $this->{FuncInfo}->{$db}->{$func}->{color};
			}
			
		}
		if ($this->{default_color}) {
			$this->{default_color};
		} else {
			$func = $this->getDefaultFunc;
			return $this->{FuncInfo}->{$db}->{$func}->{color};
		}
	}
}
###########################################################
package main;
if (__FILE__ eq $0) {
	$g = GenomeData->read("../data.test/genomes");
	$d = $g->get("bsu","BSU00110"),"\n";
	print "$d,$d->{from},$d->{to}\n";
	$funcCat = FuncCat->new("../func", "cog");
	$func = $funcCat->getGeneFunc("eco:B0002");
	$color = $funcCat->getFuncColor($func);
	print "eco:B0002, $func,$color\n";
	$func = $funcCat->getGeneFunc("eco:B0014");
	$color = $funcCat->getFuncColor($func);
	print "eco:B0014, $func,$color\n";

	$b = BitVector->new;
	$c = BitVector->new;
	$b->add(1); $b->add(2); $b->add(5); $b->add(3); $b->add(2);
	$b->print;
	print $b->{cnt},"\n";

	$c->add(3); $c->add(2); $c->add(3);
	$c->print;
	print $c->{cnt},"\n";

	($b->or($c))->print;

	print $b->contain($c),"\n";
	print $c->contain($b),"\n";
}
###########################################################
1;
