/*
 * DomClust: Hierarchical Clustering for Orthologous Domain Classification
 * Copyright (c) 2000-2007, Ikuo Uchiyama
 * All rights reserved.
 */
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "domclust.h"
#include "util.h"
#include "neighbor.h"
#include "spec.h"

#define MAXLINE 4000
#define WITH_POSITION 1

static char head[MAXLINE];
static Node *rootnode;

int outputHier0(Node *node, int lev, signed char dir);
int clearNodeOutMarks(pList *nodelist, NodeFlag flag);
int outputNewick0(TreeNode *node, Dist prevdist);
int printDomInfo0(Domain *dom, char flag);
static pList **clustTab;

#ifdef WITH_NEIGHBOROUT
outputNeighborNode(NodeSet *nodes, Node *node)
{
	Node *nbrnode, *nn, *n;
	Neighbor *nbr;
	signed char dir;
	int flag = 0;
	static int posclustnum, clustnum;
	listIter iter;
	pList *nbrlist;
	int id, cnt;

	if (! isRoot(node) || (Opt.minent > 1 && ! node->child)) {
		return 0;
	}

	nbrlist = nodeNeighbor_gapsearch(nodes, node);
	setListIter(&iter, nbrlist, 1);
	while (n = (Node *) getListIter(&iter)) {
		if (! isRoot(n) || (Opt.minent > 1 && ! n->child)) {
			continue;
		}
		if (! flag) {
			printf("PosCluster %d\n", ++posclustnum);
			clustnum = 0;
			flag = 1;
		}
		rootnode = n;
		printf("Cluster %d\n", ++clustnum);
		outputHier(n);
		printf("//\n");
		/* clear printed node */
		setFlagNode(n, NODE_DELETED);
/*
		if (checkNbrClust(n, dir, &nbrnode, &nbr) == 0) {
			break;
		}
		n = nbrnode;
*/
	}
}
#endif

static struct {
	int hom_clnum;
	int ortho_clnum;
	int count_clusters;
} Counter;

outputClusterInfo(ClusterInfo *cInfo, int clustnum, NodeSet *nodes)
{
	Domain *dom;
	int i, clsize, spcnt;
	int prev_homclust = -1;
	int ortho_clustid;
	char str_clustid[20];
	int subclustid;
	SubClusterInfo *subInfo;
	static char buf[64];

	if (Opt.outstyle == DOMAINOUT) {
		return 0;
	}
	Counter.ortho_clnum = 1;
	if (Opt.outstyle == TREEDETAIL) {
		outputHeader();
	}
	for (i = 0; i < clustnum; i++) {
		if (cInfo[i].clustid < 0) continue;
/***
		clsize = numelemList(cInfo[i].members);
		if (clsize == 0) {
			continue;
		}
		spcnt = (int) spFlagCntW_All(cInfo[i].spflag);
		if (spcnt < Opt.minsp || clsize < Opt.minent) {
			continue;
		}
		f (Opt.outgroupMode && outgroupFlagCheck(cInfo[i].spflag)==2){
			continue;
		}
***/
		ortho_clustid = cInfo[i].clustid;
		str_clustid[0] = '\0';
		if (Opt.homClustOut && cInfo[i].homclust != prev_homclust) {
			++Counter.hom_clnum;
			prev_homclust = cInfo[i].homclust;
			if (Opt.outstyle != TABOUT) {
				int scoreout_flag = 0;
				sprintf(buf, "HomCluster %d", Counter.hom_clnum);
				if (Opt.outstyle == TREEDETAIL) {
					printNodeScoreFmt(cInfo[i].homclustRoot,
						stdout, buf, 2);
				} else {
					printf("%s", buf);
					if (Opt.outputScore &&
						cInfo[i].homclustRoot->child) {
						printNodeScore(
							cInfo[i].homclustRoot,
							Opt.outputScoreFp, buf);
					}
				}
				printf("\n");
			}
		}
		if (Opt.outstyle != TABOUT) {
			sprintf(buf, "Cluster %d", ortho_clustid);
			printf("%s", buf);
			if (Opt.outputScore && cInfo[i].root->node->child) {
				printNodeScore(cInfo[i].root->node, Opt.outputScoreFp, buf);
			}
			printf("\n");
		} else {
			/** Opt.outstyle == TABOUT **/
			if (Opt.homClustOut) {
				sprintf(str_clustid, "%d|", Counter.hom_clnum);
			}
			sprintf(&str_clustid[strlen(str_clustid)],
					"%d", ortho_clustid);
		}
		switch (Opt.outstyle) {
		case TREEOUT:
		case TREEDETAIL:
		case GRAPHOUT:
		case NEWICK:
		case NEWICK_DIST:
		case HIER_DETAILOUT:
			if (Opt.outgroupMode == OutGroupMode && Opt.outputSubClustTree) {
				outputSubClusterInfo(&cInfo[i], clustnum, nodes);
				if (Opt.outstyle != TABOUT) {
					putchar('\n');
				}
			} else {
				outputClusterInfoMulti(
					&cInfo[i], clustnum, nodes);
			}
			if (Opt.outputScoreFp && Opt.outgroupMode) {
				subclustid = 1;
				while (subInfo = (SubClusterInfo *)
						shiftList(cInfo[i].ingroups)) {
					sprintf(buf, "SubCluster %d", subclustid);
					printNodeScore(subInfo->root->node,
						Opt.outputScoreFp, buf);
					subclustid++;
				}
			}
			break;
		default:
			/** default output **/
			/** output members **/
			if (! Opt.outgroupMode || Opt.outputSubClustTree) {
				outputClustInfoLines(cInfo[i].members, str_clustid, 0, NULL);

			} else {    /** outgroupMode **/
				subclustid = 1;
				while (subInfo = (SubClusterInfo *)
					shiftList(cInfo[i].ingroups)) {
					outputClustInfoLines(subInfo->members,
							str_clustid, subclustid,
							subInfo->root);
					subclustid++;
				}
				/** OutGroup **/
				if (Opt.outgroupMode==OutGroupMode) {
					outputClustInfoLines(
						cInfo[i].outgroup->members,
						str_clustid, -1,
						cInfo[i].outgroup->root);
				}
			}
			break;
		}
		if (Opt.taxMapOut == 1) {
			printf("======\n");
			mapTaxInfo_Cinfo(&cInfo[i]);
			printf("======\n\n");
		}
	}
}
outputClustInfoLines(pList *cInfoList, char *str_clustid, int subclustid,
		TreeNode *root)
{
	Domain *dom;
	static char buf[64];
	if (Opt.outstyle != TABOUT) {
		if (subclustid > 0) {
			sprintf(buf, "SubCluster %d", subclustid);
			printf("%s",buf);
			if (Opt.outputScore && root && root->node->child) {
				printNodeScore(root->node,Opt.outputScoreFp,buf);
			}
			putchar('\n');
		} else if (subclustid < 0) {
			printf("OutGroup\n");
			sprintf(&str_clustid[strlen(str_clustid)], ".0");
		}
	}
	while (dom = (Domain *) shiftList(cInfoList)) {
		if (Opt.outstyle == TABOUT) {
			if (subclustid > 0) {
				printf("%s.%d ", str_clustid, subclustid);
			} else if (subclustid < 0) {
				printf("%s.0 ", str_clustid);
			} else {
				printf("%s ", str_clustid);
			}
		} else {
		}
		printDomInfo(dom);
		putchar('\n');
	}
	if (Opt.outstyle != TABOUT) {
		printf("\n");
	}
}
outputClusterInfoMulti(ClusterInfo *cInfo, int clustnum, NodeSet *nodes)
{
	listIter iter;
	TreeNode *tnode;
	int tn = 0;
	pList *nodelist;
	int testflag = 1;
	int clnum;

	clnum = numelemList(cInfo->clusters);

	setListIter(&iter, cInfo->clusters, 1);
	if (testflag) {
		nodelist = create_pList();
		while (tnode = (TreeNode *) getListIter(&iter)) {
			testTreeOverlap(tnode, nodelist);
		}
	}
	setListIter(&iter, cInfo->clusters, 1);
	while (tnode = (TreeNode *) getListIter(&iter)) {
		outputCluster_sub(nodes, tnode, cInfo->root, 0);
		if (Opt.outstyle != TABOUT) {
			putchar('\n');
		}
		if (++tn < clnum){
			if (Opt.outstyle != NEWICK &&
					Opt.outstyle != NEWICK_DIST) {
				printf("--");
			}
			printf("\n");
		}
	}
	if (Opt.outstyle == TREEDETAIL) {
		printf("//\n");
	}
	if (testflag) {
		clearNodeOutMarks(nodelist, NODE_TMPMARK);
		clearNodeOutMarks(nodelist, NODE_DUPMARK);
		free_pList(nodelist);
	}
}
outputSubClusterInfo(ClusterInfo *cInfo, NodeSet *nodes)
{
	SubClusterInfo *subInfo;
	while (subInfo = (SubClusterInfo *) shiftList(cInfo->ingroups)) {
		printf("SubCluster %d\n", subInfo->clustid);
		outputCluster_sub(nodes, subInfo->root, cInfo->root, 0);
		if (Opt.outstyle != TABOUT) {
			putchar('\n');
		}
	}
}

/**
outputClusterData(pList *roots, NodeSet *nodes)
{
	int i;
	TreeNode *tnode;
	listIter iter;
	int spcnt;

	if (Opt.outstyle == DOMAINOUT) {
		return;
	}
	if (Opt.outstyle == CLUSTTAB) {
		outputSpNamesForClustTab();
		alloc_clustTabList();
	}

	setListIter(&iter, roots, 1);
	while (tnode = (TreeNode *) getListIter(&iter)) {
		Node *node = tnode->node;

		spcnt = (int) spFlagCntW_All(node->spflag);
/ *
printf(">spcnt=%d,cnt=%d; %d,%d\n", spcnt, node->cnt,Opt.minsp,Opt.minent);
* /
		if (spcnt < Opt.minsp || node->cnt<Opt.minent) {
			continue;
		}
		if (Opt.outgroupMode && outgroupFlagCheck(node->spflag)==2){
			/ * only outgroup species * /
			continue;
		}
		if (Opt.outstyle != NEIGHBOR) {
			if (Opt.homClustOut && Counter.count_clusters == 0) {
				printf("HomCluster %d\n", ++Counter.hom_clnum);
			}
			if (Opt.outstyle != TABOUT && Opt.outstyle != CLUSTTAB){
				printf("Cluster %d", ++Counter.ortho_clnum);
				putchar('\n');
			}
			++Counter.count_clusters;
		}
		outputCluster_sub(nodes, tnode, tnode, Counter.ortho_clnum);
		if (Opt.outstyle != TABOUT) {
			putchar('\n');
		}
	}
	if (Opt.outstyle == CLUSTTAB) {
		free_clustTabList();
	}
}
**/
clearNodeOutMarks(pList *nodelist, NodeFlag flag)
{
/*
	listIter iter;
	Node *n;
	setListIter(&iter, nodelist, 1);
	while (n = (Node *) getListIter(&iter)) {
		unsetFlagNode(n, flag);
	}
*/
	/* move to graph.c */
	clearAllMarks_sub(nodelist, flag);
}
testTreeOverlap(TreeNode *tnode, pList *nodelist)
{
	Node *child1 = NULL, *child2 = NULL;
	
	if (testFlagNode(tnode->node, NODE_TMPMARK)) {
		/* an overlapping node */
		setFlagNode(tnode->node, NODE_DUPMARK);
	} else {
		setFlagNode(tnode->node, NODE_TMPMARK);
		pushList(nodelist, tnode->node);
	}
	if (tnode->child1) {
		testTreeOverlap(tnode->child1, nodelist);
	}
	if (tnode->child2) {
		testTreeOverlap(tnode->child2, nodelist);
	}
/*
	getChilds(node, &child1, &child2);
	if (child1) {
		testTreeOverlap(child1, nodelist);
	}
	if (child2) {
		testTreeOverlap(child2, nodelist);
	}
*/
}
outputCluster_sub(NodeSet *nodes, TreeNode *tnode, TreeNode *root, int clustnum)
{
	rootnode = root->node;
	switch (Opt.outstyle) {
	case TREEOUT:
		head[0] = '\0';
		outputTree(tnode);
		break;
	case TREEDETAIL:
		outputTreeDetail(tnode);
		break;
	case GRAPHOUT:
		outputGraph(tnode->node);
		break;
#ifdef NEIGHBOROUT
	case NEIGHBOR:
		outputNeighborNode(nodes, tnode->node);
		return 0;
		break;
#endif
	case NEWICK:
	case NEWICK_DIST:
		outputNewick(tnode);
		break;
	case TABOUT:
		outputSimpleTab(tnode, clustnum);
		break;
	case NORMALOUT:
		outputNormal(tnode);
		break;
	case CLUSTTAB:
		outputClustTab(tnode, clustnum);
		break;
	case HIER_DETAILOUT:
/*
	case NORMALOLD:
*/
	default:
		outputHier(tnode->node);
		break;
	}
}

outputNormal(TreeNode *tnode)
{
	if (Opt.outgroupMode == OutGroupMode) {
		outputNormal_outgroup(tnode);
	} else {
		outputNormal_normal(tnode);
	}
}
outputNormal_normal(TreeNode *tnode)
{
	pList *members = create_pList();
	Domain *dom;
	collectCluster(tnode, members);
	while (dom = (Domain *) shiftList(members)) {
		printDomInfo(dom);
		printf("\n");
	}
	freeList(members);
}
outputNormal_outgroup(TreeNode *tnode)
{
	pList *ingroups = create_pList();
	SubClusterInfo *ingroup;
	SubClusterInfo *outgroup = NULL;
	listIter iter;
	Domain *dom;
	int subclustid = 1;
	if (Opt.outgroupMode == OutGroupMode) {
		outgroup = createSubClusterInfo();
	}

	collectCluster_outgroup(tnode, ingroups, outgroup);
	while (ingroup = (SubClusterInfo *) shiftList(ingroups)) {
		printf("SubCluster %d\n", subclustid++);
		while (dom = (Domain *) shiftList(ingroup->members)) {
			printDomInfo(dom); printf("\n");
		}
		printf("\n");
	}
	if (Opt.outgroupMode == OutGroupMode) {
		printf("OutGroup\n");
		setListIter(&iter, outgroup->members, 1);

		while (dom = (Domain *) getListIter(&iter)) {
			printDomInfo(dom); printf("\n");
		}
	}
}
alloc_clustTabList()
{
	int i;
	if ( (clustTab = (pList **) malloc(sizeof(pList*) * SPnum))==NULL ){
		allocError("clusters");
	}
	for (i = 0; i < SPnum; i++) {
		if ( (clustTab[i] = create_pList())==NULL ) {
			allocError("clustTab");
		}
	}
}
free_clustTabList()
{
	int i;
	for (i = 0; i < SPnum; i++) {
		free(clustTab[i]);
	}
	free(clustTab);
}
outputSpNamesForClustTab()
{
	int i;
	char *spname;
	printf("#id");
	for (i = 0; i < SPnum; i++) {
		spname = getSPname(i);
		printf("\t%s",spname);
	}
	printf("\n");
}
outputClustTab(TreeNode *tnode, int clustnum)
{
	pList *members = create_pList();
	Domain *dom;
	Node *lnode;
	char spname[SPNAMELEN];
	char *p, *q;
	int i, j, spid;

	collectCluster(tnode, members);
	while (dom = (Domain *) shiftList(members)) {
		lnode = dom->leaf;
		for (p = lnode->name, q = spname; *p && *p != ':'; p++, q++) {
			*q = *p;
		}
		*q = '\0';
		spid = getSPid(spname);
		pushList(clustTab[spid], dom);
	}
	printf("%d",clustnum);
	for (i = 0; i < SPnum; i++) {
		j = 0;
		printf("\t");
		while (dom = (Domain *) shiftList(clustTab[i])) {
			if (j++ > 0) printf(" ");
			printDomName(dom);
		}
	}
	freeList(members);
}
outputSimpleTab(TreeNode *tnode, int clustnum)
{
	pList *members = create_pList();
	Domain *dom;
	collectCluster(tnode, members);
	while (dom = (Domain *) shiftList(members)) {
		printDomInfo(dom);
		printf("\n");
	}
	freeList(members);
}
printNodeInfo(Node *node, Node *root)
{
	Domain *dom;
	int domn;
	if (node->domains) {
		domn = getDomainNoMark(node->domains, root, &dom);
		if (domn) {
			printf("%s", node->name);
			if (numelemList(node->domains) > 1) {
				printf("(%d)", domn);
			}
			printf(" %d %d",(int)dom->from, (int)dom->to);
		} else {
			printf("(%s deleted)",node->name);
/*
			printf(" %d %d",1,(int)node->len);
*/
		}
	} else {
		printf("%s %d %d",node->name,1,(int)node->len);
	}
}
printDomName(Domain *dom)
{
	printDomInfo0(dom, 0);
}
printDomInfo(Domain *dom)
{
	printDomInfo0(dom, WITH_POSITION);
}
printDomInfo0(Domain *dom, char flag)
{
	Node *node = dom->leaf, *root = dom->root;
	int domn;
	printf("%s", node->name);
	if (node->domains) {
		if (dom->num) {
			if (Opt.outstyle == TABOUT) {
				printf(" %d", dom->num);
			} else if (numelemList(node->domains) > 1) {
				printf("(%d)", dom->num);
			}
			if (flag & WITH_POSITION) {
			    printf(" %d %d",(int)dom->from, (int)dom->to);
			}
		} else {
			domn = getDomainNoMark(node->domains, root, &dom);
			if (domn) {
				if (numelemList(node->domains) > 1) {
					printf("(%d)", domn);
				}
				if (flag & WITH_POSITION) {
				    printf(" %d %d",(int)dom->from, (int)dom->to);
				}
			} else if (flag & WITH_POSITION) {
				printf(" %d %d",1,(int)node->len);
			}
		}
	} else if (flag & WITH_POSITION) {
		printf(" %d %d",1,(int)node->len);
	}
}
printNodeScore(Node *node, FILE *fp, char *string)
{
	int format = 0;
	if (fp == NULL) {
		fp = stdout;
		format = 1;
	}
	printNodeScoreFmt(node, fp, string, format);
}
printNodeScoreFmt(Node *node, FILE *fp, char *string, int format)
{
	if (node->child) {
		if (format==0) {
			if (string) {
				fprintf(fp, "%s", string);
			}
			fprintf(fp, ":score=%f:dist=%f\n",
					node->child->score, node->child->dist);
		} else if (format==1) {
			fprintf(fp, " [score=%f dist=%f]",
					node->child->score, node->child->dist);
		} else if (format==2) {
			if (string) {
				fprintf(fp, "%s", string);
			}
			fprintf(fp, " : %f %f",
				node->child->score, node->child->dist);
		}
	}
}

/* collect cluster members by traversing the tree */
collectCluster(TreeNode *root, pList *nodelist)
{
	collectCluster_sub(root, nodelist);
	clearDomainMark(nodelist);
}
collectCluster_sub(TreeNode *tnode, pList *nodelist)
{
	Domain *dom;
	if (tnode == NULL) {
	} else if (ClustTree_isLeaf(tnode)) {
		/* leaf node */
		pushList(nodelist, tnode->dom);
	} else {
		collectCluster_sub(tnode->child1, nodelist);
		collectCluster_sub(tnode->child2, nodelist);
	}
}
collectCluster_outgroup(TreeNode *root, pList *subclstList,
	SubClusterInfo *outgrpSubClust)
{
	listIter iter;

	SubClusterInfo *ingrp;
	collectCluster_outgroup_sub(root, subclstList, outgrpSubClust, 1);
	setListIter(&iter, subclstList, 1);
	while (ingrp = (SubClusterInfo *) getListIter(&iter)) {
		clearDomainMark(ingrp->members);
	}
	if (outgrpSubClust) {
		clearDomainMark(outgrpSubClust->members);
	}
}
collectCluster_outgroup_meta_all(pList *clustRoots, pList **newClustRoots)
{
	
	listIter iter, iter2;
	TreeNode *tnode;
	SubClusterInfo *subInfo;
	pList *subclstList = create_pList();
	char delflag;

	setListIter(&iter, clustRoots, 1);
	*newClustRoots = create_pList();
Node *tmpnode;
	while (tnode = (TreeNode *) getListIter(&iter)) {
		collectCluster_outgroup_meta(tnode, subclstList);
		delflag = 1;
		setListIter(&iter2, subclstList, 1);
		while (subInfo = (SubClusterInfo *) getListIter(&iter2)) {
			pushList(*newClustRoots, subInfo->root);
			if (subInfo->root == tnode) {
				delflag = 0;
				continue;
			}
			setFlagNode(subInfo->root->node, NODE_TMPMARK);
		}
		if (delflag) {
			deleteRootNode(tnode->node);
			phyloCut_without_check(tnode->node, NULL);
		}
		while (subInfo = (SubClusterInfo *) popList(subclstList)) {
			setFlagNode(subInfo->root->node, NODE_TMPMARK);
		}
	}
}
collectCluster_outgroup_meta(TreeNode *root, pList *subclstList)
{
	listIter iter;

	SubClusterInfo *ingrp;
	SubClusterInfo *outgrpSubClust = createSubClusterInfo();
	SubClusterInfo *new_outgrpSubClust = NULL;
	collectCluster_outgroup_meta_sub(root, subclstList, outgrpSubClust, &new_outgrpSubClust, 1);
	setListIter(&iter, subclstList, 1);

	while (ingrp = (SubClusterInfo *) getListIter(&iter)) {
		clearDomainMark(ingrp->members);
	}

	if (numelemList(outgrpSubClust->members) > 0) {
		pushList(subclstList, outgrpSubClust);
	}
}
collectCluster_outgroup_sub(TreeNode *tnode, pList *subclstList,
			SubClusterInfo *outgrpSubClust, int outRootFlag)
{
	Domain *dom;
	int domn;
	int flg;
	OutGrpStat outchk;
	Node *node;


	if (tnode == NULL) {
		return 0;
	}
	node = tnode->node;

	if (! ClustTree_isLeaf(tnode)) {

		outchk = outgroupCheck(node, outRootFlag);

		if (outchk == In_In) {
			/* both are ingroup nodes */
/*`
			if (isAllRoot(node) || ! nodeVisited(node)) {
*/
			if (isAllInRoot(node)) {
				SubClusterInfo *ingrp = createSubClusterInfo();
				ingrp->root = tnode;
				collectCluster_sub(tnode, ingrp->members);
				pushList(subclstList, ingrp);
			} else {
				collectCluster_outgroup_sub(tnode->child1,
					subclstList, outgrpSubClust,
					outRootFlag);
				collectCluster_outgroup_sub(tnode->child2,
					subclstList, outgrpSubClust,
					outRootFlag);
			}
		} else if (outchk == Out_Out) {
			/* both are outgroup nodes */
			collectCluster_sub(tnode->child1, outgrpSubClust->members);
			collectCluster_sub(tnode->child2, outgrpSubClust->members);
		} else if (outchk == Out_Both) {
			/* node1 is outgroup nodes */
			collectCluster_outgroup_sub(tnode->child2, subclstList,
					outgrpSubClust, outRootFlag);
			collectCluster_sub(tnode->child1, outgrpSubClust->members);
		} else if (outchk == Both_Out) {
			/* node2 is outgroup nodes */
			collectCluster_outgroup_sub(tnode->child1, subclstList,
					outgrpSubClust, outRootFlag);
			collectCluster_sub(tnode->child2, outgrpSubClust->members);
		} else if (outchk == Both_Both) {
			/** outgrp spec in a subcluster: possible HGT **/
			/** never come here unless -Ohorizweight is specified*/
			SubClusterInfo *ingrp = createSubClusterInfo();
			ingrp->root = tnode;
			collectCluster_sub(tnode, ingrp->members);
			pushList(subclstList, ingrp);
		}
	} else {
		/* leaf node */
		if (tnode->dom) {
			dom = tnode->dom;

			flg = outgroupFlagCheck(node->spflag);
			if (flg == 0) {
				/* ingroup */
				SubClusterInfo *ingrp = createSubClusterInfo();
				ingrp->root = tnode;
				pushList(ingrp->members, dom);
				pushList(subclstList, ingrp);
			} else {
				/* outgroup */
				pushList(outgrpSubClust->members, dom);
			}
		}
	}
}

/** OBSOLETE!! **/
/* MetaGenomeMode: outgroup == unk-only (meta) node */
collectCluster_outgroup_meta_sub(TreeNode *tnode, pList *subclstList,
			SubClusterInfo *outgrpSubClust,
			SubClusterInfo **new_outgrpSubClust,
			int outRootFlag)
{
	Domain *dom;
	int domn;
	int flg;
	OutGrpStat outchk;
	Node *node;
/*
	pList *new_outgrp = outgrp;
*/

	if (tnode == NULL) {
		return 0;
	}
	node = tnode->node;

	if (! ClustTree_isLeaf(tnode)) {

		outchk = outgroupCheck(node, outRootFlag);

/*
if (node){
  printNode(node);
  if (node->child){
	printf(" %f", node->child->dist);
  }
  printf(": %d\n", outchk);
}
*/

		if (outchk == In_In) {
			/* both are ingroup nodes */
			if (isAllInRoot(node)) {
				SubClusterInfo *ingrp = createSubClusterInfo(tnode);
				collectCluster_sub(tnode, ingrp->members);
				pushList(subclstList, ingrp);
				*new_outgrpSubClust = NULL;
				return 1;
			} else {
				collectCluster_outgroup_meta_sub(tnode->child1,
					subclstList, outgrpSubClust,
					new_outgrpSubClust,
					outRootFlag);
				collectCluster_outgroup_meta_sub(tnode->child2,
					subclstList, outgrpSubClust,
					new_outgrpSubClust,
					outRootFlag);
				*new_outgrpSubClust = NULL;
			}
		} else if (outchk == Out_Out) {
			/* both are outgroup (unk only) nodes */
			if (*new_outgrpSubClust == NULL) {
				*new_outgrpSubClust = createSubClusterInfo(tnode);
				pushList(subclstList, *new_outgrpSubClust);
			}
			collectCluster_sub(tnode->child1, (*new_outgrpSubClust)->members);
			collectCluster_sub(tnode->child2, (*new_outgrpSubClust)->members);
		} else if (outchk == Out_Both) {
			/* node1 is outgroup (unk only) nodes */
			int ret = collectCluster_outgroup_meta_sub(tnode->child2,
					subclstList, outgrpSubClust,
					new_outgrpSubClust,
					outRootFlag);
			if (ret == 1) {
				/* sister is a single OG -- put into this OG */
				*new_outgrpSubClust = (SubClusterInfo*) getListIdx(subclstList, -1);
				(*new_outgrpSubClust)->root = tnode;
			    } else {
				/* multiple OGs -- put into another OG */
				if (*new_outgrpSubClust == NULL) {
					*new_outgrpSubClust = createSubClusterInfo(tnode->child1);
					pushList(subclstList, *new_outgrpSubClust);
				}
			}
			collectCluster_sub(tnode->child1, (*new_outgrpSubClust)->members);
			return ret;
		} else if (outchk == Both_Out) {
			/* node2 is outgroup (meta) nodes */
			int ret = collectCluster_outgroup_meta_sub(tnode->child1,
					subclstList, outgrpSubClust,
					new_outgrpSubClust,
					outRootFlag);
			if (ret == 1) {
				/* sister is a single OG -- put into this OG */
				*new_outgrpSubClust = (SubClusterInfo*) getListIdx(subclstList, -1);
				(*new_outgrpSubClust)->root = tnode;
			    } else {
				/* multiple OGs -- put into another OG */
				if (*new_outgrpSubClust == NULL) {
					*new_outgrpSubClust = createSubClusterInfo(tnode->child2);
					pushList(subclstList, *new_outgrpSubClust);
				}
			}
			collectCluster_sub(tnode->child2, (*new_outgrpSubClust)->members);
			return ret;
		} else if (outchk == Both_Both) {
			/** outgrp spec in a subcluster: possible HGT **/
			/** never come here unless -Ohorizweight is specified*/
			SubClusterInfo *ingrp = createSubClusterInfo(tnode);
			collectCluster_sub(tnode, ingrp->members);
			pushList(subclstList, ingrp);
		}
	} else {
		/* leaf node */

		if (tnode->dom) {
			dom = tnode->dom;

			flg = outgroupFlagCheck(node->spflag);
			if (flg == 0) {
				/* ingroup */
				SubClusterInfo *ingrp = createSubClusterInfo(tnode);
				pushList(ingrp->members, dom);
				pushList(subclstList, ingrp);
				return 1;
			} else {
				/* outgroup */
				pushList(outgrpSubClust->members, dom);
			}
		}
	}
	return 0;
}

outputHier(Node *node)
{
	outputHier0(node, 1, (signed char) 1);
}
outputHier0(Node *node, int lev, signed char dir)
{
	int i;
	Edge *child;
	char indent[200];
	Node *maxnode;
	int maxcnt;
	Domain *dom;

	indent[0] = '\0';
	if (Opt.outstyle == HIER_DETAILOUT) {
		for (i = 0; i < lev; i++) {
			strcat(indent, "  ");
		}
	}
	if (isFlankNode1(node)) {
		if (Opt.outstyle == HIER_DETAILOUT) {
			printf("%s - (%s,%d) %d/%d (flanking node)\n",
				indent, node->name,node->id,
				nodeLen(node),node->len);
		}
		outputHier0(node->child->node1, lev, dir);
	} else if (isFlankNode2(node)) {
		if (Opt.outstyle == HIER_DETAILOUT) {
			printf("%s - (%s,%d) %d/%d (flanking node)\n",
				indent, node->name,node->id,
				nodeLen(node),node->len);
		}
#ifdef LARGE
		dir *= node->child->dir;
#endif
		outputHier0(node->child->node2, lev, dir);
	} else if (isIntNode(node)) {
		child = node->child;
		if (Opt.outstyle == HIER_DETAILOUT) {
			ConnCount conn = 0;
#ifdef EXPERIMENTAL
			conn = child->connect;
#endif
			printf("%s", indent);
			printf("[%d](%s,%d) [%d:%d] %d/%d %.2f %d  ",
				lev,node->name,node->id,
				node->newreg.from,node->newreg.to,
				nodeLen(node), node->len,
				MEASURE(child),conn);
			print_specFlag(node->spflag);

			if (node->left) {
				printf("%s", indent);
				printf("    Left: ");
/*
				printList(node->left, printNode);
				printNeighborList(node->left, node);
*/
				maxcnt = getMaxNeighbor(node->left,node,1,
					&maxnode,NULL);
				if (maxcnt && maxcnt >=
					(double) node->cnt * Opt.nbrConnRatio) {
					printNode(maxnode);
					printf(" [%d]",maxcnt);
				}
				putchar('\n');
			}
			if (node->right) {
				printf("%s", indent);
				printf("    Right: ");
/*
				printList(node->right, printNode);
				printNeighborList(node->right, node);
*/
				maxcnt = getMaxNeighbor(node->right,node,1,
					&maxnode,NULL);
				if (maxcnt && maxcnt >=
					(double) node->cnt * Opt.nbrConnRatio) {
					printNode(maxnode);
					printf(" [%d]",maxcnt);
				}
				putchar('\n');
			}
		}
		outputHier0(child->node1, lev+1, dir);
#ifdef LARGE
		dir *= node->child->dir;
#endif
		outputHier0(child->node2, lev+1, dir);
	} else {
		printf("%s", indent);
		printNodeInfo(node,rootnode);
		if (Opt.revMatch) {
			printf(" %d", dir);
		}
		putchar('\n');
	}
}
outputGraph(Node *node)
{
	Node *maxnode = NULL;
	enum {Leaf,NonLeaf} type = NonLeaf;
	int maxcnt;

	if (isFlankNode1(node)) {
		if (isIntNode(node->child->node1)) {
			printf("%d %d\n", node->id, node->child->node1->id);
		} else {
			printf("%d %s\n", node->id, node->child->node1->name);
		}
		outputGraph(node->child->node1);
	} else if (isFlankNode2(node)) {
		if (isIntNode(node->child->node2)) {
			printf("%d %d\n", node->id, node->child->node2->id);
		} else {
			printf("%d %s\n", node->id, node->child->node2->name);
		}
		outputGraph(node->child->node2);
	} else if (isIntNode(node)) {
		if (isIntNode(node->child->node1)) {
			printf("%d %d\n", node->id, node->child->node1->id);
		} else {
			printf("%d %s\n", node->id, node->child->node1->name);
		}
		if (isIntNode(node->child->node2)) {
			printf("%d %d\n", node->id, node->child->node2->id);
		} else {
			printf("%d %s\n", node->id, node->child->node2->name);
		}
		outputGraph(node->child->node1);
		outputGraph(node->child->node2);
	} else {
		type = Leaf;
	}

	if (node->left) {
		maxcnt = getMaxNeighbor(node->left,node,0,&maxnode,NULL);
		if (maxcnt && maxcnt >=
				rint(node->cnt * Opt.nbrConnRatio)) {
			if (type == NonLeaf) {
				printf("%d %d L\n", node->id, maxnode->id);
			} else {
				if (isLeaf(maxnode) && ! isRoot(maxnode)) {
					printf("%s %s L\n",node->name,maxnode->name);
				}
			}
		}
	}
	if (node->right) {
		maxcnt = getMaxNeighbor(node->right,node,0,&maxnode,NULL);
		if (maxcnt && maxcnt >=
				rint(node->cnt * Opt.nbrConnRatio)) {
			if (type == NonLeaf) {
				printf("%d %d R\n", node->id, maxnode->id);
			} else {
				if (isLeaf(maxnode) && ! isRoot(maxnode)) {
					printf("%s %s R\n",node->name,maxnode->name);
				}
			}
		}
	}
}

outputTree(TreeNode *tnode)
{
	if (tnode->child1 && ! tnode->child2) {
		tnode = tnode->child1;
	} else if (tnode->child2 && ! tnode->child1) {
		tnode = tnode->child2;
	}
	outputTree_sub(tnode, 0, 1, 0);
}
outputTree_sub(TreeNode *tnode, int lev, int dir)
{

	Edge *child;
	Domain *dom;
	Node *node;
	char markchar;

	if (tnode == NULL) return;

 	node = tnode->node;
	markchar = ( (node->flag & NODE_DUPMARK) ? '*' : '+');
	dom = tnode->dom;

	if (ClustTree_isLeaf(tnode)) {
		/** leaf **/
		printf("%s%c%s", head, markchar, "- ");
		printDomInfo(tnode->dom);

		if (Opt.outgroupMode == OutGroupMode) {
			if (dom) {
			    if (dom->subgrp >= 0) {
				if (dom){
					printf(" [I%d]", dom->subgrp);
					if(outgroupFlagCheck(node->spflag)) {
						printf(" *");
					}
				}
			    } else {
				printf(" [O]");
			    }
			} else {
			}
		}
		putchar('\n');
	} else {
		child = node->child;
		if (child) {
			head[lev*2] = '\0';
			strcat(head, (lev > 0 && dir == -1 ? "| " : "  "));
			outputTree_sub(tnode->child1, lev+1, 1);
			head[lev*2] = '\0';
			printf("%s%c%s", head, markchar, "-| ");
			if (Opt.decinum) {
				printf("%.*f\n", Opt.decinum,
					MEASURE(child));
			} else {
				printf("%.1f\n", MEASURE(child));
			}
	
			strcat(head, (lev > 0 && dir == 1 ? "| " : "  "));
			outputTree_sub(tnode->child2, lev+1, -1);
		} else {
			printf("%s%c%s[%s]", head, markchar,
				"- ", node->name);
			printf(" %d %d",1,(int)node->len);
			putchar('\n');
		}
	}
}
outputTreeDetail(TreeNode *tnode)
{
	if (tnode->child1 && ! tnode->child2) {
		tnode = tnode->child1;
	} else if (tnode->child2 && ! tnode->child1) {
		tnode = tnode->child2;
	}
	outputTreeDetail_sub(tnode, NULL);
}
outputTreeDetail_sub(TreeNode *tnode, TreeNode *parent)
{

	Edge *child;
	Domain *dom;
	Node *node;
	char markchar;
	int pid = -1;

	if (tnode == NULL) return;

 	node = tnode->node;
	markchar = ( (node->flag & NODE_DUPMARK) ? '*' : '+');
	dom = tnode->dom;
	
	if (parent != NULL && parent->node != NULL) {
		pid = parent->node->id;
	}

	if (ClustTree_isLeaf(tnode)) {
		/** leaf **/
		printf("L %d %d %s ", node->id, pid, node->name);
		printf("%d %d ", tnode->dom->from, tnode->dom->to);
		printf("%d", (numelemList(node->domains) > 1) ?  dom->num : 0);

/*
		if (Opt.outgroupMode == OutGroupMode) {
			if (dom) {
			    if (dom->subgrp >= 0) {
				if (dom){
					printf(" [I%d]", dom->subgrp);
					if(outgroupFlagCheck(node->spflag)) {
						printf(" *");
					}
				}
			    } else {
				printf(" [O]");
			    }
			} else {
			}
		}
*/
		putchar('\n');
	} else {
		printf("I %d %d %s ", node->id, pid, node->name);
		child = node->child;
		if (child) {
			if (Opt.decinum) {
				printf("%.*f %.*f", Opt.decinum, child->score,
						Opt.decinum, child->dist);
			} else {
				printf("%.1f %.1f", child->score, child->dist);
			}
			putchar('\n');
	
			outputTreeDetail_sub(tnode->child1, tnode);
			outputTreeDetail_sub(tnode->child2, tnode);
		} else {
/*
			printf("%s%c%s[%s]", head, markchar,
				"- ", node->name);
			printf(" %d %d",1,(int)node->len);
			putchar('\n');
*/
		}
	}
}
outputHeader()
{
	printf("# simtype=%d\n",Opt.sim);
	printf("# cutoff=%f\n",Opt.cutoff);
	if (Opt.sim) { /* similarity */
		printf("# missdist=%f\n",Opt.missscore);
	} else {
		printf("# missdist=%f\n",Opt.missdist);
	}
	printf("# phylocutratio=%f\n",Opt.phylocutratio);
	printf("# distscale=%d\n",Opt.distscale);
}

static char namebuf[NAMELEN];
outputNewick(TreeNode *tnode)
{
	if (tnode->child1 && ! tnode->child2) {
		tnode = tnode->child1;
	} else if (tnode->child2 && ! tnode->child1) {
		tnode = tnode->child2;
	}
	outputNewick0(tnode, ((Dist) -1.0) );
	printf(";");
}
outputNewick0(TreeNode *tnode, Dist prevdist)
{
	Dist dist = 0;
	Dist currdist = 0;

	if (tnode == NULL) {
	} else if (ClustTree_isLeaf(tnode)) {
		if (numelemList(tnode->node->domains) > 1) {
			sprintf(namebuf, "%s#%d", tnode->node->name,
						tnode->dom->num);
		} else {
			strcpy(namebuf, tnode->node->name);
		}
		convname(namebuf);
		if (Opt.outstyle == NEWICK_DIST) {
			printf("%s", namebuf);
			if (! Opt.sim && prevdist >= 0) {
				dist = prevdist;
				if (Opt.decinum) {
					printf(":%.*f ", Opt.decinum,
						(float)dist);
				} else {
					printf(":%.1f ",  (float)dist);
				}
			}
		} else {
			printf("%s", namebuf);
		}
	} else {
		currdist = tnode->node->child->dist / 2;
		printf("(");
		outputNewick0(tnode->child1, currdist);
		printf(", ");
		outputNewick0(tnode->child2, currdist);
		printf(")");
		if (Opt.outstyle == NEWICK_DIST) {
			if (! Opt.sim && prevdist >= 0) {
				dist = prevdist - currdist;
				if (Opt.decinum) {
					printf(":%.*f ", Opt.decinum,
						(float)dist);
				} else {
					printf(":%.1f ",  (float)dist);
				}
			}
		}
	}
}
convname(char *name)
{
	register char *p;
	for (p = name; *p; p++) {
		if (*p == ':') {
			*p = '_';
		}
	}
}
