/*
 * DomClust: Hierarchical Clustering for Orthologous Domain Classification
 * Copyright (c) 2000-2007, Ikuo Uchiyama
 * All rights reserved.
 */

#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "domclust.h"
#include "util.h"
#include "neighbor.h"
#include "spec.h"

cmpr_node_by_size(Node **a, Node **b)
{
	return (*b)->cnt - (*a)->cnt;
}
cmpr_treenode_by_size(TreeNode **a, TreeNode **b)
{
	return ( (*b)->node->cnt == (*a)->node->cnt ) ?
		(*b)->node->id - (*a)->node->id :
		( (*b)->node->cnt - (*a)->node->cnt );
}
cmpr_list_by_size(pList **a, pList **b)
{
	return (*b)->numelem - (*a)->numelem;
}

pList *createRootNodeList(NodeSet *nodes);

outputResults(SimGraph *SimG, int argc, char **argv)
{
	NodeSet *nodes = SimG->nodes;
	EdgeSet *edges = SimG->edges;
	pList *origClustRoots;
	pList *clustRoots, *clustRootsHom;
	ClusterInfo *cInfo;
	int i, clustnum;

	setTotalNodeNum(nodes);

	if (Opt.outstyle == DUMP) {
		dumpGraph(SimG, argc, argv);
		exit(0);
	}

	if (Opt.spmaskStr) {
		checkSkipMaskAll(nodes);
	}

	/* save original roots (homolog clusters) */

	if (Opt.homClustOut) {
		origClustRoots = createRootNodeList(nodes);
		sortList(origClustRoots, cmpr_node_by_size);
	}

	/** phylogenetic tree cutting procedure **/
        phyloCutAll(nodes, edges);

	deleteOutgroupClusters(nodes);


	/** reexamine neighboring clusters **/
	if (Opt.adjInclRatio >= ONE) {
		checkIncludedClustersAll(nodes);
	}

	if (Opt.delete_small) {
		/** delete small clusters before domain definition **/
		checkClusterSize(nodes);
	}
	/** domain definition **/
	checkDomains(nodes);

	if (Opt.homClustOut) {
		convClustTree_fromHom(nodes, origClustRoots, &clustRootsHom, &clustRoots);
		sortList(clustRootsHom, cmpr_list_by_size);
	} else {
		clustRoots = convClustTree(nodes);
	}

/**
	clustRoots = convClustTree(nodes);

print_clustList(clustRoots, "00>>>")

	if (Opt.outgroupMode == MetaGenomeMode && Opt.outputSubClustTree) {
		pList *newClustRoots;
		collectCluster_outgroup_meta_all(clustRoots, &newClustRoots);
		free_pList(clustRoots);
		clustRoots = newClustRoots;

print_clustList(clustRoots, "01>>>")

		checkDomains(nodes);
	}
**/

/*
	if (Opt.homClustOut) {
		convClustTree_fromHom(nodes, origClustRoots, &clustRootsHom, NULL);
		sortList(clustRootsHom, cmpr_list_by_size);
	}
*/

	sortList(clustRoots, cmpr_treenode_by_size);

	if (Opt.adjOvlpRatio || (Opt.mincutcnt > 1) ||
			(Opt.adjInclRatio && (Opt.adjInclRatio < ONE)) ) {
		/* merging adjacent clusters */
		clustnum = checkOverlapClustersAll(clustRoots, &cInfo, nodes);
	} else {
		clustnum = createClusterInfo(clustRoots, &cInfo, NULL);
		assignSubgroups_cinfo_all(cInfo, clustnum);
	}
	assign_clusterid(cInfo, clustnum);

	if (Opt.homClustOut) {
		/** assign and merge homologous groups **/
		mergeHomgroups_cinfo(cInfo, clustnum, clustRootsHom);
	}
	if (Opt.taxMapOut != 3) {
		outputClusterInfo(cInfo, clustnum, nodes);
	}
	if (Opt.taxMapOut > 1) {
		mapTaxInfoAll(cInfo, clustnum);
	}
}
/** for debug **/
print_clustList(pList *clustList, char *string)
{
	listIter iter;
	TreeNode *tn;
	setListIter(&iter, clustList, 1);
	while (tn = (TreeNode*)getListIter(&iter)) {
		if (string) printf(string);
		printNode(tn->node); putchar('\n');
	}
}

assign_clusterid(ClusterInfo *cInfo, int clustnum)
{
	int i, clsize;
	double spcnt;
	int clustid = 0;
	for (i = 0; i < clustnum; i++) {
		clsize = numelemList(cInfo[i].members);
		spcnt = spFlagCntW_All(cInfo[i].spflag);
		if ( (clsize == 0) ||
			(spcnt < Opt.minsp || clsize < Opt.minent) ||
			/* small clusters */
			(Opt.outgroupMode &&
				outgroupFlagCheck(cInfo[i].spflag)==2) ) {
			/* only outgroup species */

			cInfo[i].clustid = -1;
			continue;
		}
		cInfo[i].clustid = ++clustid;
	}
}


#ifdef EXPERIMENTAL
checkConnect(Node *node) {
	float cutcnt;
	if (Opt.chkConnect > 1) {
		/** count **/
		cutcnt = Opt.chkConnect;
	} else {
		/** ratio **/
		cutcnt = ((float)node->child->node1->cnt +
			node->child->node2->cnt) * Opt.chkConnect;
	}
	if (node->child->connect < cutcnt) {
		/* cut */
		return 0;
	} else {
		return 1;
	}
}
#endif

#define DEBUG_ON(x) if(x){ tmpflag=1; } else { tmpflag=0; }
int tmpflag;

phyloCutAll(NodeSet *nodes, EdgeSet *edges)
{
	int i;
	Node *node;
	NodeSetIter iter;

	/* decreasing order:
		note that new root nodes may be created after the cut */
	getRootInit(&iter, nodes, -1, 0);
/*
	while (node = getRootNext(&iter)) {
*/
	while (node = getRootNext_all(&iter)) {

#ifdef TMPDBG	/* for DEBUG */
DEBUG_ON(node->id == 18574)
#endif

		/** cut non-recursively
		   further cut should be done in the succeeding loops **/
		if ( checkUnvisitedParent(node) == 0) {
			continue;
		}
		phyloCut(node);
	}
}

phyloCut_disttest(Node *node)
{
	int cut1 = 0, cut2 = 0;
	Node *childNode;
	if (node->child == NULL) {
		return 0;
	}
	if (isFlankNode(node)) {
		/* never come here? */
		return 0;
	} else {
		if (node->child->node1->child) {
			cut1 = phyloCut_disttest_sub(node, node->child->node1);
		}
		if (node->child->node1->child) {
			cut2 = phyloCut_disttest_sub(node, node->child->node2);
		}
	}
	return (cut1 || cut2);
}
phyloCut_disttest_sub(Node *node, Node *child)
{
	/* test mode */
	while (isFlankNode(child)) {
		if (isFlankNode1(child)) {
			child = child->child->node1;
		} else {
			child = child->child->node2;
		}
	}
	if (! child->child) {
		return 0;
	}
/*
printf(">>node=%s,%d; %d,%d\n", node->name,node->id, node, child);
*/
	double distr = DISTRATIO( MEASURE(child->child),
			MEASURE(node->child) );
/*
printf(">dist=%lf,%lf\n",MEASURE(child->child),MEASURE(node->child));
*/
	if (distr <= Opt.distdiffcut){
		return 1;
	} else {
		return 0;
	}
}
postPhyloCut(Node *node, Node *parent)
{
	int stat;
	int outgroupflag = 0;

if (nodeDeleted0(node)){
	printf("phylocut: ????\n");
	if (node->child) {
		printf("%d\n",isRoot(node));
		printf("%d\n",isRoot(node->child->node1));
		printf("%d\n",isRoot(node->child->node2));
	}
	return 0;
}
	if (! parent) {
		return 0;
	}
	/** When there is a parent, we have entered this node after
		cutting this parent node. So we must cancel
		the domain boundaries created during the cluster
		merging process.
	**/

if (tmpflag) {
printf("******\n");
printNode(node);
printNode(parent);
printf("\n");
printf("(%d,%d);(%d,%d)\n",parent->left,parent->right,node->left,node->right);
}

	if (isOutGrpNode(parent)) {
		setOutGrpNode(node);
		outgroupflag = 1;
	}
	if (! outgroupflag) {
		restoreNeighborList(parent, node);
		restoreBreak(node, parent);
	}

	if ( (stat=checkUnvisitedParent(node)) == 0) {
		/** do not proceed further if there remains an unvisited parent **/
		/** never come here unless the node is an outgroup node **/
		if (outgroupflag) {
			/** set the node as an ingroup root to prevent further cut 
				because the flanking node is in a single cluster **/
/*
printf(">>>outgrpflag\n");
printNode(node);
printf("\n");
*/
			makeRootNode(node);
		} else {
		}
		return 0;
	}


	/* ### There is no unvisited parent */

	/* delete the flanking root node if exist,
		which is probably retained as a long domain */
	if (! outgroupflag) {
		if (node->parentL && isRoot(node->parentL)) {
			deleteNode(node->parentL);
			restoreBreak(node, node->parentL);
		}
		if (node->parentR && isRoot(node->parentR)) {
			deleteNode(node->parentR);
			restoreBreak(node, node->parentR);
		}
	}
/*
if (node->parentL && node->parentL->id==10869) {
printf("L******%d,%d\n",parent->id,outgroupflag);
} else if (node->parentR && node->parentR->id==10869) {
printf("R******%d,%d\n",parent->id,outgroupflag);
}
*/

	/* if the parent is not a root then make this node as a root */
/***
	if (! node->parent || ! isRoot(node->parent)) {
		makeRootNode(node);
	}
***/
/*
printf(">>nroot\n");
printNode(node);
printf("\n");
*/
	makeRootNode(node);

	return 0;
}
phyloCut(Node *node)
{
	int cut = 0;
	OutGrpStat outchk = In_In;
	if (node->child) {
		if (isFlankNode(node)) {
			Node *nbrNode = NULL, *child = NULL;
			/* nbrNode = central node (newnode0) */
			if (isFlankNode1(node)) {
				child = node->child->node1;
			} else if (isFlankNode2(node)) {
				child = node->child->node2;
			} else {
				fprintf(stderr, "????\n");
			}
			if (isOutGrpNode(node)) {
				postPhyloCut(child, node);
				if (isInRoot(node)) {
					deleteNode(node);
				}
				return 0;
			}
			/* find the neighboring (central) node */
			if (node == child->parentL) {
				/* right neighbor */
				nbrNode = (Node *) getNeighborNode(
							node, 1, NULL);
			} else if (node == child->parentR) {
				/* left neighbor */
				nbrNode = (Node *) getNeighborNode(
							node, -1, NULL);
			} else {
				/** ???? never come here **/
				/** node == child->parent || child->parentM */
				fprintf(stderr, "ERROR?? a central node is treated as a flanking node\n");
				assert(0);
				makeRootNode(child);
				deleteRootNode(node);
				return 0;
			}
			if (nbrNode && nbrNode->flag & NODE_DELETED) {
				/* the central node is already cut */
				/* never come here ? */
				fprintf(stderr, "????\n");
				deleteNode(node);
				return postPhyloCut(child, node);
			} else {
				if (checkNodeLen(node) < 2) {
					/* short segment */
					deleteNode(node);
					checkIncludedCluster(node);
				} else {
					/* having sufficient length */
					/* make this node as a root */
					makeRootNode(node);
				}
				return 0;
			}
			/* never come here */
		}

		if (Opt.outgroupMode == OutGroupMode) {
			int cutO;
			cutO = checkOutGrpRoot(node, &outchk);
			if (cutO) {
				cut = 1;
			} else {
				if (outchk==In_In || outchk==In_In_Horiz) {
					if ( cut==0 && Opt.in_cutoff &&
						BETTER(Opt.in_cutoff, MEASURE(node->child)) ) {
						/* cutoff for ingroup */
						cut = 1;
					} else {
					/* in, in (or in,in+out_horiz) */
						cut = duplicationCheck_inGrp(node);
					/* force to check weak condition */
						if (cut==0 && outchk==In_In_Horiz) {
							cut = 2;
						}
					}
				} else {
					cut = duplicationCheck(node);
				}
			}
		} else if (Opt.metagenomeMode) {
			cut = phyloCheck_forMeta(node);
		} else {
	/** Test for the phylogenetic tree cutting procedure **/
			cut = duplicationCheck(node);
		}
/*
printNode(node);
if (node->child->node1) printNode(node->child->node1);
if (node->child->node2) printNode(node->child->node2);
printf("cut=%d\n",cut);
*/
		if (cut == 1) {
			/** immediately cut this node **/
		} else if ( cut == 2 ) {
#ifdef EXPERIMENTAL
			if ((Opt.chkConnect && ! checkConnect(node)) ||
				(Opt.sumcut &&
			  	Opt.sumcut > getEdgeScoreSum(node->child))) {

				/* weak cut condition (connect/sumcut) */
			} else
#endif
			if (Opt.distdiffcut) {
				/* weak cut condition (distdiffcut) */
				int cut1;
				cut1 = phyloCut_disttest(node);
				if (cut1 == 0) {
					cut = 0;
				} else {
					/* cut */
				}
			} else {
				cut = 0;
			}
		}
		if (outchk == Out_Both || outchk == Both_Out) {
			/* out,in+out || in+out,out */
			if (cut == 0) {
				/* possible outgroup root */
				/* force to cut */
				cut = 3;
			}
		}
		if (cut) {
			cutNode(node, cut, outchk);
		}
	}
	return cut;
}
phyloCut_without_check(Node *node, Node *parent)
{
	Edge *child = node->child;
/*
	if (nodeDeleted0(node)) {
		return 0;
	}
*/
	if (testFlagNode(node, NODE_TMPMARK)) {
/*
printf("OK");
printNode(node);
printf("\n");
*/
		return 0;
	}
/*
printf("phylocut:");
printNode(node);
printf("\n");
*/
	if (child) {
		if (isFlankNode(node)) {
			Node *nbrNode = NULL, *childNode = NULL;
			if (isFlankNode1(node)) {
				childNode = node->child->node1;
			} else if (isFlankNode2(node)) {
				childNode = node->child->node2;
			}
			if (node == childNode->parentL) {
				nbrNode = (Node *) getNeighborNode(node, 1, NULL);
			} else if (node == childNode->parentR) {
				nbrNode = (Node *) getNeighborNode(node, -1, NULL);
			}
			if (nbrNode && nbrNode->flag & NODE_DELETED) {
				fprintf(stderr, "????\n");
				deleteNode(node);
			} else {
				if (checkNodeLen(node) < 2) {
					/* short segment */
					deleteNode(node);
					checkIncludedCluster(node);
				} else {
					/* having sufficient length */
					/* make this node as a root */
					makeRootNode(node);
				}
/*
				return 0;
*/
			}
			phyloCut_without_check(childNode, node);
		} else {
/*
			if (parent) {
printf("RESTORE::");
printNode(node);
printf("\n");
				restoreNeighborList(parent, node);
				restoreBreak(node, parent);
			}
*/
			unsetOutGrpNode(node);
			cutNode(node, 1, In_In);

			phyloCut_without_check(child->node1, node);
			phyloCut_without_check(child->node2, node);
		}
	}
}
cutNode(Node *node, int cut, OutGrpStat outchk)
{
	int outroot_flag = 0;
	int skip_child = 0;
	if (Opt.DEBUG & DBG_nbrrestore) {
		printf("Cut: "); printNode(node); putchar('\n');
		print_specFlag(node->child->node1->spflag);
		print_specFlag(node->child->node2->spflag);
	}
	if (! isOutGrpNode(node) && cut == 3) {
	    /** this is an outgroup root: do not delete **/
		outroot_flag = 1;
	    /* temporarily set the flag to propagate it to children */
		setOutGrpNode(node);
	} else {
		deleteRootNode(node);
	}
	if (outchk == Out_Both) {
		skip_child = 1;
	} else if (outchk == Both_Out) {
		skip_child = 2;
	}
	if (skip_child != 1) {
		postPhyloCut(node->child->node1, node);
	}
	if (skip_child != 2 &&
		(node->child->node1 != node->child->node2)) {
		/* NOT self_match */
		postPhyloCut(node->child->node2, node);
	}
	if (outroot_flag) {
		/* reset the flag in the outgroup root */
		unsetOutGrpNode(node);
	}
}

/** metagenomeMode **/
phyloCheck_forMeta(Node *node)
{
	specFlagP spflag1, spflag2;
	int cut = 0;
	if (metagenomeOnlyCluster(node->spflag)) {
		return 0;
	} else if (node->child) {
		if (isFlankNode(node)) {
			if (isFlankNode1(node)) {
				cut = phyloCheck_forMeta(node->child->node1);
			} else if (isFlankNode2(node)) {
				cut = phyloCheck_forMeta(node->child->node2);
			}
		} else {
			spflag1 = node->child->node1->spflag; 
			spflag2 = node->child->node2->spflag; 
			if (metagenomeOnlyCluster(spflag1)) {
				cut = phyloCheck_forMeta(node->child->node2);
			} else if (metagenomeOnlyCluster(spflag2)) {
				cut = phyloCheck_forMeta(node->child->node1);
			} else {
				cut = duplicationCheck(node);
			}
		}
	} else {
		/** leaf **/
		return 0;
	}
	return(cut);
}

checkOutGrpRoot(Node *node, OutGrpStat *ret_outchk) {
	OutGrpStat outchk;
	int cut;
	outchk = outgroupCheckRaw(node, 1);
	if (ret_outchk) {
		*ret_outchk = outchk;
	}
	cut = checkOutGrpCutNode(outchk, node);
	if (cut == 2) {
		/* check children */
		if (node->child) {
			if (isFlankNode1(node)) {
				cut = checkOutGrpRoot(node->child->node1,NULL);
			} else if (isFlankNode2(node)) {
				cut = checkOutGrpRoot(node->child->node2,NULL);
			} else {
				int cut1, cut2;
				cut1 = checkOutGrpRoot(node->child->node1,NULL);
				cut2 = checkOutGrpRoot(node->child->node2,NULL);
				if (cut1 || cut2) {
					cut = 1;
				} else {
					cut = 0;
				}
			}
		} else {
			cut = 0;
		}
	}
	return cut;
}
/* basic rule for cutting outgruoup nodes */
checkOutGrpCutNode(OutGrpStat outchk, Node *node)
{
	if (outchk == Out_Out) {
		/* out,out -> not cut */
		return 0;
/*
	} else if (outchk == In_In || outchk == In_In_Horiz) {
*/
	} else if (outchk == In_In) {
		/* in,in -> not cut */
		return 0;
	} else if (outchk == In_In_Horiz) {
		return duplicationCheck_inGrp(node);
	} else if (outchk == Both_Both) {
		/* in+out,in+out -> cut */
		return 1;
	} else {
		/* outchk == Out_Both || Both_Out */
		/* out,in+out || in+out,out -> need to check children */
		return 2;
	}
}
/**
checkOutRoot(Node *node)
{
	if (node->child) {
		if (isFlankNode(node)) {
		} else {
		}
	}
}
**/

#ifdef OOOO
cancelOutRoot(Node *node)
{
	if (isRoot(node)) {
		/* cancel this root node */
/*
printNode(node);
printf(":CANCEL\n");
*/
		deleteRootNode(node);
	} else {
/*
printNode(node);printf(">>parent:%d,%d,%d\n",node->parentL,node->parent,node->parentR);
if (node->parent){
printNode(node->parent);printf(">>cantest\n");
}
*/
		if (node->parentL) cancelOutRoot(node->parentL);
		if (node->parent) cancelOutRoot(node->parent);
		if (node->parentR) cancelOutRoot(node->parentR);
	}
}
#endif

/* skip nodes that contain only masked species */
checkSkipMaskAll(NodeSet *nodes)
{
	Node *node;
	NodeSetIter iter;

	getRootInit(&iter, nodes, -1, 0);
	while (node = getRootNext(&iter)) {
		if (! nodeDeleted0(node)) {
			checkSkipMask(node);
		}
	}
}
checkSkipMask(Node *node)
{
	Node *child1, *child2;
	int maskchk = spFlagMaskCheck(node->spflag, SPflags.spMask);
#ifdef WITH_DEBUG
/***
printf("Match:%d\n",maskchk);
print_specFlag(node->spflag);
print_specFlag(SPflags.spMask);
printf("===\n");
**/
#endif
	if (maskchk == 2) {
		/** all organisms are matched with specified ones **/
		return 0;
	} else if (maskchk == 0) {
		/** no organism is matched with specified ones **/
		deleteRootNode(node);
		return 0;
	}
#ifdef AA
	if (node->child) {
		child1 = node->child->node1;
		child2 = node->child->node2;
		if (! isFlankNode(node)) {
			if (spFlagMaskCheck(child1->spflag, SPflags.spMask) == 0) {
				/* child1 should be deleted */
				unsetFlagNode(node, NODE_INT1);
			} else if (spFlagMaskCheck(child2->spflag, SPflags.spMask) == 0) {
				/* child2 should be deleted */
				unsetFlagNode(node, NODE_INT2);
			}
		}
	}
#endif
	return 0;
}

/* delete node and recover neighbor relationships */
deleteNode(Node *node)
{
	listIter *iterL, *iterR;
	Neighbor *nbrL, *nbrR;
	Node *nodeL, *nodeR;
	int cnt;
	char status;

	if (nodeDeleted0(node)) {
		return 0;
	}

if (Opt.DEBUG & DBG_nbrrestore) {
	printf("DelFlank: ");
	printNode(node);
	printf("\n");
}

	/***
		 +<-------newlink------->+
		 |     nbrL       nbrR   |
		nodeL --+-- node --+-- nodeR
			   delete
	**/

	deleteRootNode(node);
	if (node->left && node->right) {
		iterL = createListIter(node->left, 1);
		iterR = createListIter(node->right, 1);
		while (nbrL = (Neighbor *)getListIter(iterL)) {
			nodeL = (Node *) (((Node *) nbrL->node1 == node)
					? nbrL->node2 : nbrL->node1);

			setListIter(iterR, node->right, 1);

			while (nbrR = (Neighbor *)getListIter(iterR)) {
				nodeR = (Node *) (
					    ((Node *) nbrR->node1 == node) ?
					    nbrR->node2 : nbrR->node1);

				cnt = (nbrL->count < nbrR->count) ?
					nbrL->count : nbrR->count;
			/***
				   nbrL       nbrR
				L -{1}- curr -{-1}- R
				1         1        -1
			**/
				if (nbrL->status == 0 && nbrR->status == 0){
					status = 0;
				} else if (nbrL->status == 2 || nbrR->status == 2){
					status = 2;
				} else {
					status = 1;
				}

if (Opt.DEBUG & DBG_nbrrestore) {
	printf("newnode: ");
	printNode(nodeL); printNode(nodeR);
	printf(" %d\n", status);
}
				addNeighborRestore(nodeL,nodeR,
					nbrL->dir,nbrR->dir,cnt, status);
			}
		}
	}
	return 1;
}
pList *createRootNodeList(NodeSet *nodes)
{
	NodeSetIter iter;
	Node *node;
	pList *rootNodes = create_pList();

	getRootInit(&iter, nodes, -1, 1);
	while (node = getRootNext(&iter)) {
		pushList(rootNodes, node);
	}
	return rootNodes;
}

deleteOutgroupClusters(NodeSet *nodes)
{
	NodeSetIter iter;
	Node *node;

	getRootInit(&iter, nodes, -1, 0);
	while (node = getRootNext(&iter)) {
		if (outgroupFlagCheck(node->spflag) == 2) {
		/* the set contains only the outgroup organisms */
			deleteRootNode(node);
		}
	}
}
/* Classifying out/in-group patterns of both subtrees */
/* Do not discriminate In_In and In_In_Horiz */
OutGrpStat outgroupCheck(Node *node, int outRootFlag)
{
	OutGrpStat outchk = outgroupCheckRaw(node, outRootFlag);
	if (outchk == In_In_Horiz) {
		return In_In;
	} else {
		return outchk;
	}
}
OutGrpStat outgroupCheckRaw(Node *node, int outRootFlag)
{
	specFlagP spflag1, spflag2;
	int check1, check2;

	if (node->child==NULL) {
		check1 = outgroupFlagCheck(node->spflag);
		if (check1 == 0) {
			return In_In;
		} else {
			return Out_Out;
		}
	}

	spflag1 = node->child->node1->spflag; 
	spflag2 = node->child->node2->spflag; 

	check1 = outgroupFlagCheck(spflag1);
	check2 = outgroupFlagCheck(spflag2);

	if (outRootFlag) {
		/* unknown-only cluster is treated as an outgroup */
/*
		if (metagenomeOnlyCluster(spflag1)) {
			check1 = 2;
		}
		if (metagenomeOnlyCluster(spflag2)) {
			check2 = 2;
		}
*/
	}

	if (check1 == 0 && check2 == 0) {
		/* in, in : check */
		return In_In;
	} else if (check1 == 2 && check2 == 2) {
		/* out, out : ignore */
		return Out_Out;
	} else if (check1 == 2) {
		/* out, in+out : add left, check right */
		return Out_Both;
	} else if (check2 == 2) {
		/* in+out, out : add right, check left */
		return Both_Out;
	} else {
		/* in, in+out or in+out, in+out : cut */

		if (Opt.horizweight) {
			double outcnt1,outcnt2,outcntM, incnt1,incnt2,incntM;
			double in_del, out_del, horiz;
			specFlag spflagM; /* merged set */

			spFlagOR(spflag1,spflag2,spflagM);

			outcnt1 = spFlagANDcntW(spflag1, SPflags.outGroup);
			incnt1 = spFlagANDcntW(spflag1, SPflags.inGroup);
			outcnt2 = spFlagANDcntW(spflag2, SPflags.outGroup);
			incnt2 = spFlagANDcntW(spflag2, SPflags.inGroup);
			outcntM = spFlagANDcntW(spflagM, SPflags.outGroup);
			incntM = spFlagANDcntW(spflagM, SPflags.inGroup);

			in_del = (incntM - incnt1) + (incntM - incnt2);
			out_del = (outcntM - outcnt1) + (outcntM - outcnt2);
			horiz = outcntM;
			if ((in_del + out_del) < horiz / Opt.horizweight) {
				return Both_Both;
			} else {
				return In_In_Horiz;
			}
		}
		return Both_Both;
		
	}
}

/*	0 .. no outgroup species,
	1 .. there is a species belonging to the outgroup,
	2 .. all species are belonging to the outgroup */
outgroupFlagCheck(specFlag spflag)
{
	return spFlagMaskCheck(spflag, SPflags.outGroup);
}
/*	0 .. no ingroup species,
	1 .. there is a species belonging to the ingroup,
	2 .. all species are belonging to the ingroup */
ingroupFlagCheck(specFlag spflag)
{
	return spFlagMaskCheck(spflag, SPflags.inGroup);
}

#ifdef AAA
spFlagMaskCheck(specFlag spflag, specFlag spFlagPat)
{
	specFlag matched;
	int matched_spcnt, spcnt;

	spFlagAND(spflag, spFlagPat, matched);
	matched_spcnt = spFlagCnt(matched);
	if (matched_spcnt) {
		spcnt = spFlagCnt(spflag);
		if (matched_spcnt == spcnt) {
		/* all organisms in the set are matched */
			return 2;
		} else {
		/* there exists at least one matched organism */
			return 1;
		}
	}
	return 0;
}
#endif

/* check the phylogenetic tree cutting criterion */
duplicationCheck(Node *node)
{
	return duplicationCheck0(node->spflag, node->child->node1->spflag,
		node->child->node2->spflag);
}
/* check duplications only among ingroup species */
duplicationCheck_inGrp(Node *node)
{
	specFlag spflag0, spflag1, spflag2;
	spFlagAND(node->spflag, SPflags.inGroup, spflag0);
	spFlagAND(node->child->node1->spflag, SPflags.inGroup, spflag1);
	spFlagAND(node->child->node2->spflag, SPflags.inGroup, spflag2);
	return duplicationCheck0(spflag0, spflag1, spflag2);
}
duplicationCheck0(specFlag spflag0, specFlag spflag1, specFlag spflag2)
{
	int mchcnt, spcnt1, spcnt2, mchcnt2;
	double wt_mchcnt, wt_spcnt1, wt_spcnt2;

	/** Either one is an unknown-only cluster -- cut **/
/**
	if (unknownOnlyCluster(spflag1) || unknownOnlyCluster(spflag2)) {
		return 1;
	}
**/

	mchcnt = spFlagANDcnt(spflag1, spflag2);
	spcnt1 = spFlagCnt(spflag1);
	spcnt2 = spFlagCnt(spflag2);

	if (mchcnt * spcnt1 * spcnt2 == 1) {
		/** Do not cut this node **/
		return 0;
	}

	wt_mchcnt = spFlagANDcntW(spflag1, spflag2);
	wt_spcnt1 = spFlagCntW(spflag1);
	wt_spcnt2 = spFlagCntW(spflag2);
/*
printf(">>%d,%d,%d\n",mchcnt,spcnt1,spcnt2);
printf(">>>%lf,%lf\n",wt_spcnt1*Opt.phylocutratio,wt_spcnt2*Opt.phylocutratio);
*/


	/** cut node when two subgroups share organisms with a ratio of
		more than Opt.phylocutratio: do not cut when the
		ratio is just Opt.phylocutratio */
	if ((wt_mchcnt > (double) wt_spcnt1 * Opt.phylocutratio
		 || wt_mchcnt > (double) wt_spcnt2 * Opt.phylocutratio)
			 && (mchcnt * spcnt1 * spcnt2 != 1) ) {
		/** Immediately cut this node **/
		return 1;
	}
	if (Opt.treecheck) {
		/* Mapping the nodes of gene tree onto the species tree */
		int spnode0, spnode1, spnode2;
		spnode0 = sptree_MatchFlags(spflag0);
		spnode1 = sptree_MatchFlags(spflag1);
		spnode2 = sptree_MatchFlags(spflag2);
		if (spnode0 == spnode1 || spnode0 == spnode2) {
			/* duplication !! */
#ifdef WITH_DEBUG
/*
	printf("==\n");
	print_specFlag(spflag0);
	print_specFlag(spflag1);
	print_specFlag(spflag2);
	printf("%d,%d,%d\n",spnode0,spnode1,spnode2);
*/
#endif
			/* sum of the weights of the duplicated species */
			wt_mchcnt =
			    sptree_MatchFlagsCntW(spflag1,spflag2,spnode0);

			if ((wt_mchcnt > (double) wt_spcnt1 * Opt.phylocutratio
			 || wt_mchcnt > (double) wt_spcnt2 * Opt.phylocutratio)
		 		&& (mchcnt * spcnt1 * spcnt2 != 1) ) {
				/** Cut this node **/
				return 1;
			}
		}
	}
	if ((wt_mchcnt > (double) wt_spcnt1 * Opt.phylocutratio2
		 || wt_mchcnt > (double) wt_spcnt2 * Opt.phylocutratio2)
	) {
		return 2;
	}
	/** Do not cut this node **/
	return 0;
}

restoreBreak(Node *node, Node *parent)
{
	if (! parent) return;

	if (parent == node->parentL) {
		node->brk.from = INFPOS;
/*
		node->parentL = NULL;
*/
	} else if (parent == node->parentR) {
		if (node->parentM) {
			node->brk2.to = SUPPOS;
		} else {
			node->brk.to = SUPPOS;
		}
/*
		node->parentR = NULL;
*/
	} else if (parent == node->parentM) {
		node->brk.to = SUPPOS;
		node->brk2.from = INFPOS;
/*
		node->parentM = NULL;
*/
	} else if (parent == node->parent) {
		if(definedPos(node->brk.from)) {
			node->brk.from = INFPOS;
		} else if (definedPos(node->brk.to)){
			node->brk.to = SUPPOS;
		}
		if (parent->child->node1 == parent->child->node2) {
			if(definedPos(node->brk2.from)) {
				node->brk2.from = INFPOS;
			} else if(definedPos(node->brk2.to)) {
				node->brk2.to = SUPPOS;
			}
		}
/*
		node->parent = NULL;
*/
	} else {
		/*** error **/
		if (node->parent != NULL &&
	node->parentR != NULL && node->parentL != NULL && node->parentM!=NULL) {
		fprintf(stderr, ">>ERROR %d: %d,%d,%d\n",
			parent->id,node->parent,node->parentL,node->parentR);
			printf("??? %s,%d %d,%d,%d,%d,%d\n",node->name,node->id,node->parent,node->parentL,node->parentR,parent,parent->id);
			printf("??? %d %d,%d\n",parent->child->node1, parent->child->node2);
			printNode(node); putchar(' ');
			if (node->parent){
				printNode(node->parent); putchar('\n');
			}
			if (node->parentL){
				putchar('L');
				printNode(node->parentL); putchar('\n');
			}
			if (node->parentR){
				putchar('R');
				printNode(node->parentR); putchar('\n');
			}
		}
	}
}
checkUnvisitedParent(Node *node)
{
	if ( (node->parent == NULL || nodeVisited(node->parent)) &&
		(node->parentL == NULL || nodeVisited(node->parentL)) &&
		(node->parentR == NULL || nodeVisited(node->parentR)) ) {

		/** there remains no unvisited parent **/
		return 1;
	}
/*
if (node->parent != NULL && ! nodeVisited(node->parent)) {printNode(node->parent);printf(":M\n");}
if (node->parentL != NULL && ! nodeVisited(node->parentL)) {printNode(node->parentL);printf(":L\n");}
if (node->parentR != NULL && ! nodeVisited(node->parentR)) {printNode(node->parentR);printf(":R\n");}
printNode(node); printf(": unvisited=yes\n");
*/
	/** there remains unvisited parent **/
	return 0;
}

checkClusterSize(NodeSet *nodes)
{
	NodeSetIter iter;
	Node *node;
	double spcnt;
	int i;

	getRootInit(&iter, nodes, -1, 0);
	while (node = getRootNext(&iter)) {
		if (node->child) {
			spcnt = spFlagCntW_All(node->spflag);

			if (spcnt < Opt.minsp || node->cnt < Opt.minent) {
				deleteNode(node);
			}
		}
	}
}
