#ifndef _BINQUEUE_HPP_
#define _BINQUEUE_HPP_

#include <iostream>
#include <list>
#include <cmath>
#include "HomData.h"
#include "DEBUG.h"

using namespace std;

const int DEF_SCALE = 20;

template <typename T>
class BinQueue {
	int minValue, maxValue, scale;
	bool logscale;
	bool lowCutFlag, highCutFlag;
	int minIdx, maxIdx, indexSize;
	list<T>** data;
	list<T> *data_overflow, *data_underflow;
public:
	enum Best {MAX, MIN} best;
	BinQueue(int min, int max, Best _best=MAX,
			int scl=DEF_SCALE, bool logscl=false) :
		minValue(min), maxValue(max),  scale(scl),
		best(_best), logscale(logscl),
		data(NULL), data_overflow(NULL), data_underflow(NULL) {
		minIdx = maxIdx = 0;
		if (scale == 0) scale = DEF_SCALE;
		if (! logscl) {
			maxIdx = (max - min + 1) * scale;
		}
//cerr << "binqueue:" << maxValue << " " << minValue << " " << maxIdx << endl;
		indexSize = maxIdx + 1;
//cerr << "indexSize=" << indexSize << endl;
		if (indexSize < 0) {
			if (DEBUG::verbose_flag) {
				cerr << "Warning: No valid distance data (indexSize < 0)\n";
			}
			return;
		}
		data = new list<T>*[ indexSize ];
		for (int i = 0; i < indexSize; i++) {
			data[i] = NULL;
		}
		data_overflow = data_underflow = NULL;
	}
	~BinQueue() {
		for(int i = 0; i < indexSize; i++) {
			if (data[i]) delete data[i];
		}
		if (data != NULL) delete[] data;
		if (data_overflow != NULL) delete data_overflow;
		if (data_underflow != NULL) delete data_underflow;
	}
	void add(double key, T datum) {
		int intkey = calcIndex(key);
		if (intkey >= indexSize) {
			// highCutFlag discards overflow data
			if (! highCutFlag) {
				cerr << "BinQueue: index overflow" << endl;
				if (data_overflow==NULL) {
					data_overflow = new list<T>();
				}
				data_overflow->push_back(datum);
			}
		} else if (intkey < 0) {
			// lowCutFlag discards underflow data
			if (! lowCutFlag) {
				cerr << "BinQueue: index underflow "<< key
					<< " " <<minValue<<" "<<intkey<< endl;
				if (data_underflow == NULL) {
					data_underflow = new list<T>();
				}
				data_underflow->push_back(datum);
			}
		} else {
			if (data[intkey] == NULL) {
				data[intkey] = new list<T>();
			}
			data[intkey]->push_back(datum);
		}
	}
	int getBestData(T& bestData) {
		if (best == MAX) {
			return getMaxData(bestData);
		} else {
			return getMinData(bestData);
		}
	}
	int getMaxData(T& bestData) {
		while (maxIdx >= 0 && (data[maxIdx]==NULL
				|| data[maxIdx]->empty())) {
			maxIdx--;
		}
		if (maxIdx < 0) {
			return(-1);
		} else {
			if (data[maxIdx]->empty()) {
				return(-1);
			} else {
				bestData = data[maxIdx]->back();
				data[maxIdx]->pop_back();
			}
			return(0);
		}
	}
	int getMinData(T& bestData) {
		while (minIdx < indexSize && (data[minIdx]==NULL
				|| data[minIdx]->empty())) {
			minIdx++;
		}
		if (minIdx >= indexSize) {
			return(-1);
		} else {
			if (data[minIdx]->empty()) {
			} else {
				bestData = data[minIdx]->front();
				data[minIdx]->pop_front();
			}
			return(0);
		}
	}
	int calcIndex(double value) {
		int idx;
		if (logscale) {
			idx = (int) (log(value/minValue) * scale);
		} else {
			idx = (int) ((value - minValue) * scale);
		}
		return(idx);
	}
	void setLowCutFlag() {lowCutFlag = true;}
	void setHighCutFlag() {highCutFlag = true;}
};
#endif
