#include <stdio.h>
#include <math.h>
#include "mathmat.h"

/** taken from Numerical Recipes in C **/
/** Do not export this routine outside **/

static double sqrarg;
double det_MathMat(MathMat*);
#define SQR(a) ((sqrarg=(a)) == 0.0 ? 0.0 : sqrarg * sqrarg)
#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))
#define TINY 1.0e-20

double pythag(double a, double b)
{
	double absa, absb;
	absa = fabs(a);
	absb = fabs(b);
	if (absa > absb) return absa * sqrt(1.0+SQR(absb/absa));
	else return (absb == 0.0 ? 0.0 : absb * sqrt(1.0+SQR(absa/absb)));
}

#ifdef DEBUG
main(int argc, char **argv)
{
	int n = 3;
	MathMat *mat = create_MathMat(n, n);
	MathMat *newmat, *l_mat, *u_mat, *invmat, *powmat, *powmat2;
	double *d;
	int i, j, k = 0;
/*
	static double aa[] = {1, 2, 7, 4, 5, 2, 5, 3, 5};
*/
	static double aa[] = {1, 2, 7, 2, 5, 3, 7, 3, 4};

	for (i = 0; i < n; i++) {
		for (j = 0; j < n; j++) {
			mat->matrix[i][j] = aa[k++];
		}
	}
	printf_MathMat(mat, "%3.3lf");
	printf("===\n");
	eigen_sym_MathMat(mat, &newmat, &d);
	printf_MathMat(newmat, "%3.3lf");
	for (i = 0; i < n; i++) {
		printf("%d, %lf\n", i, d[i]);
	}
	inv_MathMat(mat, &invmat);
	printf("----\n");
	printf_MathMat(invmat, "%.3f");
	printf("det=%lf\n", det_MathMat(mat));
	pow_MathMat(mat, 5.0, &powmat);
	pow_MathMat(powmat, 0.2, &powmat2);
	printf("----\n");
	printf_MathMat(powmat2, "%.3f");

/*
	lu_decomp_MathMat(mat, &l_mat, &u_mat);
	printf("----\n");
	printf_MathMat(l_mat, "%.3f");
	printf("----\n");
	printf_MathMat(u_mat, "%.3f");
*/
}
#endif

pow_MathMat(MathMat *mat, double pown, MathMat **powered)
{
	MathMat *eigvects, *newmat, *newmat2;
	double *eigvals;
	int i;
	eigen_sym_MathMat(mat, &eigvects, &eigvals);
	for (i = 0; i < mat->row; i++) {
		eigvals[i] = pow(eigvals[i], pown);
	}
	newmat = create_MathMat_diagvect(eigvals, mat->row);
	newmat2 = MathMat_multiply(eigvects, newmat);
	inv_MathMat(eigvects, &newmat);
	*powered = MathMat_multiply(newmat2, newmat);
	free_MathMat(newmat);
	free_MathMat(newmat2);
	free_MathMat(eigvects);
}

eigen_sym_MathMat(MathMat *mat, MathMat **egvects, double **egvals)
{
	double *d, *e;
	MathMat *newmat = copy_MathMat(mat);
	symmetrize_MathMat(newmat);
	if (((d = (double *) malloc(sizeof(double) * mat->row)) == NULL) ||
	    ((e = (double *) malloc(sizeof(double) * mat->row)) == NULL)) {
		fprintf(stderr, "Can't alloc memory\n");
		exit(1);
	}
	tred2(newmat, d, e);
	tqli(d, e, newmat);
	if (egvects) {
		*egvects = newmat;
	}
	if (egvals) {
		*egvals = d;
	}
	free(e);
}

tred2(MathMat *mat, double *d, double *e)
{
	int l, k, j, i;
	double scale, hh, h, g, f;
	int n = mat->row;
	double **a = mat->matrix;
	for (i = n-1; i >= 1; i--) {
		l = i - 1;
		h = scale = 0.0;
		if (l > 0) {
			for (k = 0; k <= l; k++) {
				scale += fabs(a[i][k]);
			}
			if (scale == 0.0) {
				e[i] = a[i][l];
			} else {
				for (k = 0; k <= l; k++) {
					a[i][k] /= scale;
					h += a[i][k] * a[i][k];
				}
				f = a[i][l];
				g = (f >= 0.0 ? -sqrt(h) : sqrt(h));
				e[i] = scale * g;
				h -= f * g;
				a[i][l] = f - g;
				f = 0.0;
				for (j = 0; j <= l; j++) {
					a[j][i] = a[i][j] / h;
					g = 0.0;
					for (k =0; k <= j; k++) {
						g += a[j][k]*a[i][k];
					}
					for (k = j+1; k <= l; k++) {
						g += a[k][j]*a[i][k];
					}
					e[j] = g / h;
					f += e[j] * a[i][j];
				}
				hh = f / (h+h);
				for (j = 0; j <= l; j++) {
					f = a[i][j];
					e[j] = g = e[j] - hh*f;
					for (k = 0; k <= j; k++) {
						a[j][k] -= (f*e[k]+g*a[i][k]);
					}
				}
			}
		} else {
			e[i] = a[i][l];
		}
		d[i] = h;
	}
	e[0] = d[0] = 0.0;
	for (i = 0; i < n; i++) {
		l = i - 1;
		if (d[i]) {
			for (j = 0; j <= l; j++) {
				g = 0.0;
				for (k = 0; k <= l; k++) {
					g += a[i][k] * a[k][j];
				}
				for (k = 0; k <= l; k++) {
					a[k][j] -= g * a[k][i];
				}
			}
		}
		d[i] = a[i][i];
		a[i][i] = 1.0;
		for (j = 0; j <= l; j++) {
			a[j][i] = a[i][j] = 0.0;
		}
	}
}
tqli(double *d, double *e, MathMat *mat)
{
	int m, l, iter, i, k;
	double s, r, p, g, f, dd, c, b;
	int n = mat->row;
	double **z = mat->matrix;

	for (i = 1; i < n; i++) {
		e[i-1] = e[i];
	}
	e[n-1] = 0.0;
	for (l = 0; l < n; l++) {
		iter = 0;
		do {
			for (m = l; m < n-1; m++) {
				dd = fabs(d[m])+fabs(d[m+1]);
				if ((double)(fabs(e[m])+dd) == dd)
					break;
			}
			if (m != l) {
				if (iter++ == 30) {
					fprintf(stderr, "Too many iter\n");
					exit(1);
				}
				g = (d[l+1]-d[l]) / (2.0* e[l]);
				r = pythag(g, 1.0);
				g = d[m] - d[l] + e[l]/(g+SIGN(r,g));
				s = c = 1.0;
				p = 0.0;
				for (i = m-1; i >= l; i--) {
					f = s * e[i];
					b = c * e[i];
					e[i+1] = (r=pythag(f,g));
					if (r == 0.0) {
						d[i+1] -= p;
						e[m] = 0.0;
						break;
					}
					s = f / r;
					c = g / r;
					g = d[i+1] - p;
					r = (d[i] - g) * s + 2.0 * c * b;
					d[i+1] = g + (p = s * r);
					g = c * r - b;
					for (k = 0; k < n; k++) {
						f = z[k][i+1];
						z[k][i+1] = s*z[k][i]+c*f;
						z[k][i] = c*z[k][i]-s*f;
					}
				}
				if (r==0.0 && i>=l)
					continue;
				d[l] -= p;
				e[l] = g;
				e[m] = 0.0;
			}
		} while (m != l);
	}
}

inv_MathMat(MathMat *mat, MathMat **invmat)
{
	MathMat *newmat = copy_MathMat(mat);
	int *indx;
	double det, *col;
	int i, j;

	*invmat = create_MathMat(mat->row, mat->col);
	if ((indx = (int *) malloc(sizeof(int) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	if ((col = (double *) malloc(sizeof(double) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	lu_decomp(newmat, indx, &det);

	for (j = 0; j < newmat->row; j++) {
		for (i = 0; i < newmat->row; i++) {
			col[i] = 0.0;
		}
		col[j] = 1.0;
		lu_backsub(newmat, indx, col);
		for(i = 0; i < newmat->row; i++) {
			(*invmat)->matrix[i][j] = col[i];
		}
	}
	free(col);
	free(indx);
}
double det_MathMat(MathMat *mat)
{
	MathMat *newmat = copy_MathMat(mat);
	int *indx;
	double det;
	int i;
	if ((indx = (int *) malloc(sizeof(int) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	lu_decomp(newmat, indx, &det);
	for(i = 0; i < newmat->row; i++) {
		det *= newmat->matrix[i][i];
	}
	free(indx);
	return det;
}
lu_decomp_MathMat(MathMat *mat, MathMat **mat_l, MathMat **mat_u)
{
	MathMat *newmat = copy_MathMat(mat);
	int *indx;
	double det;
	int i, j;
	if ((indx = (int *) malloc(sizeof(int) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	lu_decomp(newmat, indx, &det);
	*mat_u = newmat;
	*mat_l = create_MathMat_init(0, mat->row, mat->col);
	(*mat_l)->matrix[0][0] = 1.0;
	for (i = 1; i < (*mat_u)->row; i++) {
		for (j = 0; j < i; j++) {
			(*mat_l)->matrix[i][j] = (*mat_u)->matrix[i][j];
			(*mat_u)->matrix[i][j] = 0.0;
		}
		(*mat_l)->matrix[i][i] = 1.0;
	}
}
lu_decomp(MathMat *mat, int *indx, double *d)
{
	int i, imax, j, k;
	double maxval, dum, sum, temp;
	MathVect *vv;
	int n = mat->row;
	double **a = mat->matrix;
	vv = create_MathVect(n);
	*d = 1.0;
	for (i = 0; i < n; i++) {
		maxval = 0.0;
		for (j = 0; j < n; j++) {
			if ((temp = fabs(a[i][j])) > maxval) {
				maxval = temp;
			}
		}
		if (maxval == 0.0){
			fprintf(stderr, "Singular matrix\n");
			exit(1);
		}
		vv->vect[i] = 1.0 / maxval;
	}
	for (j = 0; j < n; j++) {
		for (i = 0; i < j; i++) {
			sum = a[i][j];
			for (k = 0; k < i; k++) {
				sum -= a[i][k] * a[k][j];
			}
			a[i][j] = sum;
		}
		maxval = 0.0;
		for (i = j; i < n; i++) {
			sum = a[i][j];
			for (k = 0; k < j; k++) {
				sum -= a[i][k] * a[k][j];
			}
			a[i][j] = sum;
			if ((dum = vv->vect[i] * fabs(sum)) >= maxval) {
				maxval = dum;
				imax = i;
			}
		}
		if (j != imax) {
			for (k = 0; k < n; k++) {
				dum = a[imax][k];
				a[imax][k] = a[j][k];
				a[j][k] = dum;
			}
			*d = - (*d);
			vv->vect[imax] =  vv->vect[j];
		}
		indx[j] = imax;
		if (a[j][j] == 0.0) {
			a[j][j] = TINY;
		}
		if (j != n-1) {
			dum = 1.0 / a[j][j];
			for (i = j + 1; i < n; i++) {
				a[i][j] *= dum;
			}
		}
	}
	free_MathVect(vv);
}
lu_backsub(MathMat *mat, int *indx, double *b)
{
	int i, ii = -1, ip, j;
	double sum;
	int n = mat->row;
	double **a = mat->matrix;
	for (i = 0; i < n; i++) {
		ip = indx[i];
		sum = b[ip];
		b[ip] = b[i];
		if (ii >= 0) {
			for (j = ii; j <= i-1; j++) {
				sum -= a[i][j] * b[j];
			}
		} else if (sum) {
			ii = i;
		}
		b[i] = sum;
	}
	for (i = n-1; i >= 0; i--) {
		sum = b[i];
		for (j = i+1; j < n; j++) {
			sum -= a[i][j] * b[j];
		}
		b[i] = sum / a[i][i];
	}
}

chol_decomp_MathMat(MathMat *mat, double *b)
{
	MathMat *newmat = copy_MathMat(mat);
	double *p, *x;
	symmetrize_MathMat(newmat);
	if ((p = (double *) malloc(sizeof(double) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	if ((x = (double *) malloc(sizeof(double) * mat->row)) == NULL) {
		fprintf(stderr, "Can't allocate memory\n");
		exit(1);
	}
	if (chol_decomp(newmat, p) == NULL) {
		return;
	}
	if (b){
		chol_solve(newmat, p, b, x);
	}
}
chol_decomp(MathMat *mat, double *p)
{
	int i, j, k;
	int n = mat->row;
	double **a = mat->matrix;
	double sum;
	for (i = 0; i < n; i++) {
		for (j = i; j < n; j++) {
			sum = a[i][j];
			for (k = i - 1; k >= 0; k--) {
				sum -= a[i][k] * a[j][k];
			}
			if (i == j) {
				if (sum <= 0.0) {
					return NULL;
				}
				p[i] = sqrt(sum);
			} else {
				a[j][i] = sum / p[i];
			}
		}
	}
	return 1;
}
chol_solve(MathMat *mat, double *p, double *b, double *x)
{
	int i, k;
	double sum;
	int n = mat->row;
	for (i = 0; i < n; i++) {
		for (sum = b[i], k = i-1; k >= 0; k--) {
			x[i] = sum / p[i];
		}
	}
	for (i = n - 1; i >= 0; i--) {
		for (sum = x[i], k = i+1; k < n; k++) {
			x[i] = sum / p[i];
		}
	}
}
