#!/usr/bin/perl -s
package MbgdUserGenomeCommon;
use strict;
use CGI;
use DirHandle;
use FileHandle;
use File::Path;
use IO::File;
use MBGD;
use MBGD::DB;
use MBGD::WWW;
use GenBank;
use Carp;
require "MBGD_Conf.pl";
require "MBGD_commonPath.pl";
require "libSeq.pl";
require "htblast.pl";

###############################################################################
#
$main::FALSE = 0;
$main::TRUE  = 1;

#
$MbgdUserGenomeCommon::ERR_MSG_COL_TYPE = "Start (column 3) and End (column 4) positions should be numeric.  "
                                        . "Direction (column 5) should be 1 or -1.";

#
$MbgdUserGenomeCommon::ERR_MSG_NO_ID = "Insufficient information or illegal format:"
                                     . "No '%s' column was found at line %d.";

#
$MbgdUserGenomeCommon::ERR_MSG_NO_COLUMN = "Insufficient information or illegal format:"
                                         . "'%s' does not have '%s' column.";

#
$MbgdUserGenomeCommon::ERR_MSG_SAME_ID = "Illegal data:"
                                       . "'%s' exists more than twice.";

#
$MbgdUserGenomeCommon::MYSQL_auto_increment = 399000000;

###############################################################################
#

$MbgdUserGenomeCommon::STA_buildingGenomeDb = 'Building the Genome Database';
$MbgdUserGenomeCommon::STA_blastSearch      = 'BLAST Search';
$MbgdUserGenomeCommon::STA_dpAlignment      = 'DP Alignment';
$MbgdUserGenomeCommon::STA_motifSearch      = 'Domain Search';
$MbgdUserGenomeCommon::STA_buildingHomDb    = 'Building the Homology Database';
$MbgdUserGenomeCommon::STA_postProcessing   = 'Post Processing';
@MbgdUserGenomeCommon::STAlist = (
    $MbgdUserGenomeCommon::STA_buildingGenomeDb,
    $MbgdUserGenomeCommon::STA_blastSearch,
    $MbgdUserGenomeCommon::STA_dpAlignment,
    $MbgdUserGenomeCommon::STA_motifSearch,
    $MbgdUserGenomeCommon::STA_buildingHomDb,
    $MbgdUserGenomeCommon::STA_postProcessing);

$MbgdUserGenomeCommon::MAX_USER_GENOME = 19;
$MbgdUserGenomeCommon::MAX_USER_CHROMOSOME = 99;

@MbgdUserGenomeCommon::LST_KEY_GENOME = ( 'id_user_genome',
                                          'species_name',
                                          'abbreviation_name',
					  'sp',
                                          'strain',
                                          'taxonomy_id',
					  'type' );
@MbgdUserGenomeCommon::LST_KEY_CHROMOSOME = ( 'id_user_chromosome',
                                              'chromosome_name',
                                              'chromosome_shape',
                                              'chromosome_length' );
@MbgdUserGenomeCommon::LST_FIL_CHROMOSOME = ( 'file_gbk',
                                              'file_gene_table',
                                              'file_protein',
                                              'file_dna' );

$MbgdUserGenomeCommon::MAX_email = 48;
@MbgdUserGenomeCommon::LST_KEY_USERINFO = ( 'email',
                                            'name',
                                            'pass1',
                                            'pass2',
                                            'organization',
                                            'receive_email',
                                            'email_clustering',
                                            'update_gene_funccat');

$MbgdUserGenomeCommon::MODE_USERCHR_complete           = 'complete';
$MbgdUserGenomeCommon::MODE_USERCHR_predict            = 'predict';
$MbgdUserGenomeCommon::MODE_USERCHR_metagenome         = 'metagenome';
$MbgdUserGenomeCommon::MODE_USERCHR_genbank            = 'genbank';
$MbgdUserGenomeCommon::MODE_USERCHR_genetab            = 'genetab';
$MbgdUserGenomeCommon::MODE_USERCHR_genetab_proteinseq = 'genetab_proteinseq';
$MbgdUserGenomeCommon::MODE_USERCHR_proteinseq         = 'proteinseq';
$MbgdUserGenomeCommon::MODE_USERCHR_dnaseq             = 'dnaseq';

#
###############################################################################

###############################################################################
#
sub clearLogMessage {
    my($uid) = shift;

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($filename) = "$dirUg/tmp/build.log";

    # ŤХååץ
    unlink("$filename.bak") if (-e "$filename.bak");

    # Хåå
    rename("$filename", "$filename.bak") if (-e "$filename");

    # ե
    new FileHandle(">$filename");

    return;
}

###############################################################################
#
sub writeLogMsg {
    my($uid) = shift;
    my(@msg) = @_;

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($dir) = "$dirUg/tmp";

    mkpath($dir, 0, 0750) if (! -e "$dir");
    my($filename) = "$dir/build.log";
    my($fh) = new FileHandle(">>$filename");
    $fh->print(scalar(localtime()), "\t", join("\n", @msg), "\n");
    $fh->close();

    return;
}

###############################################################################
#
sub saveErrorMessage {
    my($filename) = shift;
    my(@listMessage) = @_;

    #
    my($fh) = new FileHandle(">$filename");
    $fh->print(join("\r\n", @listMessage));
    $fh->close();

    return;
}

###############################################################################
#
sub saveFh2Local {
    my($fhr) = shift;
    my($filename) = shift;
    my($buf);

    #
    my($fhw) = new FileHandle(">$filename");
    while(<$fhr>) {
        $fhw->print($_);
    }
    $fhw->close();

    return;
}

###############################################################################
#
sub saveFh2LocalFasta {
    my($fhr) = shift;
    my($filename) = shift;
    my($buf);

    #
    my($fhw) = new FileHandle(">$filename");
    while(<$fhr>) {
        if (/^\>\s*(.+)$/) {
            my($title) = $1;
            my(@title_list) = split(/\s+/, $title);
            if ($title_list[0] =~ /^gi\|(\d+)/i) {
                $title_list[0] = 'gi_' . $1;
            }
            $_ = '>' . join(' ', @title_list) . "\n";
        }
        $fhw->print($_);
    }
    $fhw->close();

    return;
}

###############################################################################
#
sub getDirBaseUserGenome {
    my($uid) = shift;
    my($opt) = shift;

    if ($uid =~ /^\s*$/) {
        print STDERR "WARNING :: UID is blank!!\n";
        for(my$i = 0; $i <= 3; $i++) {
            print STDERR "CALLER :: ", join(", ", caller($i)), "\n";
        }
    }

    my($mymbgdId) = $uid;
    if (! $opt->{'no_convert'}) {
        $mymbgdId = uid2mymbgdid($uid);
    }
#    my($dirBaseUg) = sprintf("%s/MBGDUserData/%s", $ENV{'MBGD_HOME'}, $mymbgdId);
    my($dirBaseUg) = sprintf("%s", $ENV{'MBGD_HOME'});

    return $dirBaseUg;
}

###############################################################################
#
sub getDirUserGenomeDatabase {
    my($uid) = shift;

#    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($dirUgDb) = sprintf("%s/database", $ENV{'MBGD_HOME'});

    return $dirUgDb;
}

###############################################################################
#
sub getDirUserGenomeSpecies {
    my($uid) = shift;

    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($dirUgSp) = "$dirBaseUg/species";

    return $dirUgSp;
}

###############################################################################
#
sub getDirGenomeSpid {
    my($uid) = shift;
    my($gid) = shift;

    my($dir) = MbgdUserGenomeCommon::getDirUserGenomeSpecies($uid);
    my($spid) = getGenomeSpid($gid);
    my($dirSpid) = "$dir/$spid";

    return $dirSpid;
}

###############################################################################
#
sub getDirGenome {
    my($uid) = shift;
    my($gid) = shift;

    my($dirSpid) = getDirGenomeSpid($uid, $gid);
    my($dirSpidGm) = "$dirSpid/gm";

    return $dirSpidGm;
}

###############################################################################
#
sub getDirUserGenomeSpid {
    my($uid) = shift;
    my($gid) = shift;

    my($dir) = MbgdUserGenomeCommon::getDirUserGenomeSpecies($uid);
    my($spid) = getUserGenomeSpid($gid);
    my($dirSpid) = "$dir/$spid";

    return $dirSpid;
}

###############################################################################
#
sub getDirUserGenome {
    my($uid) = shift;
    my($gid) = shift;

    my($dirSpid) = getDirUserGenomeSpid($uid, $gid);
    my($dirSpidGu) = "$dirSpid/gu";

    return $dirSpidGu;
}
sub getDirUserGenomeReady {
    my($uid, $gid) = @_;
    my($dirSpid) = getDirUserGenomeSpid($uid, $gid);
    return "$dirSpid/gm";
}

###############################################################################
#
sub getGidList {
    my($uid) = shift;
    my(@gid_list);

    my($dirBaseUg) = MbgdUserGenomeCommon::getDirUserGenomeSpecies($uid);
    my($dh) = DirHandle->new("$dirBaseUg") || die("Can not open $dirBaseUg($!)");
    foreach my$file ($dh->read()) {
        next if ($file !~ /^gu(\d+)$/);
        my($gid) = $1;

        my($dir) = "$dirBaseUg/$file";
        if (-e "$dir") {
            push(@gid_list, $gid);
        }
    }

    return @gid_list;
}

###############################################################################
#
sub getDirUserGenomeTmp {
    my($uid) = shift;

    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($dirUgDb) = "$dirBaseUg/tmp";

    return $dirUgDb;
}


###############################################################################
#
sub getDirUserChromosome {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;

    my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid, $idUserGenome);
    my($dirUc) = sprintf("%s/data/chr%02d", $dirUg, $idUserChromosome);

    return $dirUc;
}

###############################################################################
#
sub getUserGenomeIdList {
    my($uid) = shift;
    my($UG_ID);
    my(@List);
    for($UG_ID = 1; $UG_ID <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $UG_ID++){
        if (-d &MbgdUserGenomeCommon::getDirUserGenome($uid, $UG_ID)) {
            push(@List, getUserGenomeSpec($UG_ID));
        }
    }
    return @List;
}

###############################################################################
#
sub getUserGenomeIdListStatusOk {
    my($uid) = shift;
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUser) = "$dirUgDb/bldb";

    #
    my(@spList);
    my($gid);
    for($gid = 1; $gid <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $gid++) {
        my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
        if (! $refUserGenome) {
            next;
        }

        if ($refUserGenome->{'status'} !~ /^Ok$/) {
            next;
        }

        #
        push(@spList, getUserGenomeSpec($gid))
    }

    return @spList;
}

###############################################################################
#
sub saveUgInfo {
    my($filename) = shift;
    my($formOpt) = shift;
    my(@listKey) = @_;
    my($key);

    my($fh) = new FileHandle(">$filename") || return;;
    foreach $key (@listKey) {
        $fh->print($key, "\t", $formOpt->{"$key"}, "\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub loadUgInfo {
    my($filename) = shift;
    my($formOpt) = {};

    my($fh) = new FileHandle("$filename") || return $formOpt;
    while($_ = $fh->getline()) {
        s#[\r\n]*$##;

        my($key, $val) = split(/\t/);
        $formOpt->{"$key"} = $val;
    }
    $fh->close();

    return $formOpt;
}

###############################################################################
#
sub saveUserGenomeInfo {
    my($uid) = shift;
    my($formOpt) = shift;

    #
    my($idUserGenome) = $formOpt->{'id_user_genome'};
    my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid, $idUserGenome);
    mkpath($dirUg, 0, 0750) if (! -e "$dirUg");

    #
    my($filename) = "$dirUg/genome.txt";
    saveUgInfo($filename, $formOpt, @MbgdUserGenomeCommon::LST_KEY_GENOME);

#    my($sta) = MbgdUserGenomeCommon::isReadyUserGenome($uid);
#    if ($sta) {
#        # UserGenome DB νäƤ ===> MySQL ξ򹹿
#        my($db) = MBGD::DB->new("dbi:mysql:$ENV{'MYSQL_DB'}");
#        my($ugSpec) = getUserGenomeSpec($idUserGenome);
#        my($sql) = "update genome_user set "
#                 . "orgname='$formOpt->{'species_name'}',"
#                 . "abbrev='$formOpt->{'abbreviation_name'}',"
#                 . "strain='$formOpt->{'strain'}',"
#                 . "taxid='$formOpt->{'taxonomy_id'}'"
#                 . "where sp='$ugSpec'";
#        $db->execute($sql);
#    }

    return;
}

###############################################################################
#
sub loadUserGenomeInfo {
    my($uid) = shift;
    my($idUserGenome) = shift;

    #
    my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid, $idUserGenome);
    my($filename) = "$dirUg/genome.txt";
    if (! -e $filename) {
        return undef();
    }
    my($refUserGenome) = loadUgInfo($filename);

    # ǽʤηв
    my($fileLastUpdate) = "$dirUg/last_update";
    $refUserGenome->{'last_update'} = -M $fileLastUpdate;

    # Chromosome Ͽ֥å
    $refUserGenome->{'chromosome'} = [];
    my($idUserChromosome);
    my($nStatusOk) = 0;
    my($nStatusNg) = 0;
    for($idUserChromosome = 1; $idUserChromosome <= $MbgdUserGenomeCommon::MAX_USER_CHROMOSOME; $idUserChromosome++) {
        my($refUserChromosome) = loadUserChromosomeInfo($uid,
                                                        $idUserGenome,
                                                        $idUserChromosome);
        push(@{$refUserGenome->{'chromosome'}}, $refUserChromosome);
        if (! $refUserChromosome) {
            next;
        }

        if ($refUserChromosome->{'status'} =~ /^Ok$/) {
            $nStatusOk++;
        }
        else {
            $nStatusNg++;
        }
    }

    #
    $refUserGenome->{'status'} = 'Incomplete';
    if (($nStatusOk != 0) && ($nStatusNg == 0)) {
        $refUserGenome->{'status'} = 'Ok';
    }

    return $refUserGenome;
}

###############################################################################
#
sub getGenomeInfo {
    my($uid) = shift;
    my($gid) = shift;
    my(%spInfo);

    my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
    $spInfo{'sp'} = $refUserGenome->{'sp'};
    $spInfo{'orgname'} = $refUserGenome->{'species_name'};
    $spInfo{'abbrev'}  = $refUserGenome->{'abbreviation_name'};
    $spInfo{'strain'}  = $refUserGenome->{'strain'};
    $spInfo{'taxid'}   = $refUserGenome->{'taxonomy_id'};

#    $spInfo{'sp'}      = getUserGenomeSpec($gid) if ($spInfo{'sp'});

    return %spInfo;
}

###############################################################################
#
sub saveUserChromosomeInfo {
    my($uid) = shift;
    my($formOpt) = shift;

    #
    my($idUserGenome)     = $formOpt->{'id_user_genome'};
    my($idUserChromosome) = $formOpt->{'id_user_chromosome'};
    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    mkpath($dirUc, 0, 0750) if (! -e "$dirUc");

    #
    my($filename) = "$dirUc/chromosome.txt";
    saveUgInfo($filename, $formOpt, @MbgdUserGenomeCommon::LST_KEY_CHROMOSOME);

    my($sta) = MbgdUserGenomeCommon::isReadyUserGenome($uid);
    if ($sta) {
        # UserGenome DB νäƤ ===> MySQL ξ򹹿
        my($db) = MBGD::DB->new("dbi:mysql:$ENV{'MYSQL_DB'}");
        my($ugSpec) = getUserGenomeSpec($idUserGenome);
        my($sql) = "update chromosome_user set "
                 . "name='$formOpt->{'chromosome_name'}',"
                 . "shape='$formOpt->{'chromosome_shape'}'"
                 . "where sp='$ugSpec' "
                 . " and seqno=$idUserChromosome";

        $db->execute($sql);
    }

    #
    my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid, $idUserGenome);
    my($fileLastUpdate) = "$dirUg/last_update";
    new FileHandle(">$fileLastUpdate");

    #
#    setUserGenomeDbStatus($uid, $main::FALSE);
#    clearFileTaxon($uid);
}

###############################################################################
#
sub loadUserChromosomeInfo {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;

    if (! $idUserChromosome) {
        $idUserChromosome = getFirstUserChromosomeId($idUserGenome);
    }

    #
    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    if (! -e $dirUc) {
        return undef();
    }

    #
    my($filename) = "$dirUc/chromosome.txt";
    my($refUserChromosome) = loadUgInfo($filename);

    #
    $refUserChromosome->{'status'} = 'Ok';
    if (! getUserChromosomeStatus($uid, $idUserGenome, $idUserChromosome)) {
        $refUserChromosome->{'status'} = 'Incomplete';
    } elsif (! getUserGenomeConvertingStatus($uid, $idUserGenome, $idUserChromosome)) {
        $refUserChromosome->{'status'} = 'Converting';
    }

    return $refUserChromosome;
}

###############################################################################
#
sub loadUserGeneInfo {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($refUserGene) = {};
    $refUserGene->{'LIST'} = [];

    #
    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    if (! -e $dirUc) {
        return $refUserGene;
    }

    #
    my($refGeneTab) = readGeneTab($uid, $idUserGenome, $idUserChromosome);

    #
    my($fileOrigProtein) = "$dirUc/origProtein";
    if (-e $fileOrigProtein) {
        my($refProtein) = $refGeneTab->{'PROT_SEQ_SET'} = readOrigFasta($fileOrigProtein);
        my($refGene);
        foreach $refGene (@{$refGeneTab->{'LIST'}}) {
            my($id) = $refGene->{'ID'};
            $refGene->{'PROT_SEQ'} = $refProtein->{'HASH'}->{"$id"}->{'SEQ'};
        }
    }

    #
    my($fileOrigDna) = "$dirUc/origDna";
    if (-e $fileOrigDna) {
        my($refDna) = readOrigFasta($fileOrigDna);
	if ($refGeneTab->{WITH_CHR}) {
		$refGeneTab->{'CHR_SEQ_MULTI'} = $refDna->{'LIST'};
	} else {
	        my($ent) = @{$refDna->{'LIST'}};
		$refGeneTab->{'CHR_SEQ'} = $ent->{'SEQ'};
	}
    }

    return $refGeneTab;
}

###############################################################################
#
sub getNewUserGenomeId {
    my($uid) = shift;
    my($idUserGenome);

    #
    for($idUserGenome = 1; $idUserGenome <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $idUserGenome++) {
        my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid,
                                                            $idUserGenome);
        if (! -e $dirUg) {
            return $idUserGenome;
        }
    }

    return undef();
}

###############################################################################
#
sub getNewUserChromosomeId {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome);

    #
    for($idUserChromosome = 1; $idUserChromosome <= $MbgdUserGenomeCommon::MAX_USER_CHROMOSOME; $idUserChromosome++) {
        my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                                $idUserGenome,
                                                                $idUserChromosome);
        if (! -e $dirUc) {
            return $idUserChromosome;
        }
    }

    return undef();
}

###############################################################################
#
sub getFirstUserGenomeId {
    my($uid) = shift;
    my($idUserGenome);

    #
    for($idUserGenome = 1; $idUserGenome <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $idUserGenome++) {
        my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid,
                                                            $idUserGenome);
        if (-e $dirUg) {
            return $idUserGenome;
        }
    }

    return undef();
}

###############################################################################
#
sub getFirstUserChromosomeId {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome);

    #
    for($idUserChromosome = 1; $idUserChromosome <= $MbgdUserGenomeCommon::MAX_USER_CHROMOSOME; $idUserChromosome++) {
        my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                                $idUserGenome,
                                                                $idUserChromosome);
        if (-e $dirUc) {
            return $idUserChromosome;
        }
    }

    return undef();
}

###############################################################################
#
sub clearHomologyRes {
    my($uid) = shift;
    my($sp1) = shift;
    my(@splist) = @_;

    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new("dbi:mysql:$dbname");
    my($sql);
    if ($db->sta_connect()) {
        foreach my$sp2 (@splist) {
            $sql = "delete from homology_user where sp1='$sp1' and sp2='$sp2'";
            $db->execute($sql);
            $sql = "delete from homology_user where sp1='$sp2' and sp2='$sp1'";
            $db->execute($sql);
        }
    }

    return;
}

###############################################################################
#
sub delUserHomologyTables {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($sp) = getUserGenomeSpec($idUserGenome);

    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new("dbi:mysql:$dbname");
    if ($db->sta_connect()) {
        my($sta) = $db->exist_table('cluster_tables');
        if ($sta) {
            my($sql);
            # 饹󥰷̰ꡢ饹
            $sql = "select * from cluster_tables where spec like '%$sp%'";
            my($sth) = $db->execute($sql);
            while (my$ref = $sth->fetchrow_hashref()) {
                my($tabid) = $ref->{'tabid'};
                if ($tabid =~ /(all|default|addspec)/i) {
                    next;
                }
                
                # 饹󥰷̰
                $sql = "delete from cluster_tables where tabid='$tabid'";
                $db->execute($sql);
                
                # 饹󥰷̥ơ֥
                foreach my$tab_head ('cluster', 'clusttab', 'clusttree', 'clustcat') {
                    
                    my($tabname) = $tab_head . '_' . $tabid;
                    $sql = "drop table IF EXISTS $tabname";
                    $db->execute($sql);
                }
                
                # ո̤
                $sql = "delete from protmotif_user where sp like '$sp'";
                $db->execute($sql);
            }
            
            # homology
            $sql = "delete from homology_user where sp1='$sp' or sp2='$sp'";
            $db->execute($sql);
        }
    }

    #
    clearBlastFinishedSpec($uid, $sp);
    clearMotifFinishedSpec($uid, $sp);

    return;
}

###############################################################################
#
sub delUserGenomeTables {
    my($uid) = shift;
    my($idUserGenome) = shift;

    # MySQL ϿƤǡ
    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new("dbi:mysql:$dbname");
    if ($db->sta_connect()) {
        my($sql);
        my($sp) = getUserGenomeSpec($idUserGenome);

        # Genome/Chromosome/Gene
        foreach my$tab ('gene', 'chromosome', 'genome') {
            $sql = "delete from ${tab}_user where sp='$sp'";
            $db->execute($sql);
        }
    }

    # Ϣ homology ̤
    delUserHomologyTables($uid, $idUserGenome);

    return;
}

###############################################################################
#
sub deleteUserGenome {
    my($uid) = shift;
    my($idUserGenome) = shift;

    my($dirUg) = MbgdUserGenomeCommon::getDirUserGenome($uid,
                                                        $idUserGenome);
    rmtree($dirUg);

    # MySQL ϿƤ Genome/Chromosome/Gene ǡ
#    delUserGenomeTables($uid, $idUserGenome);

    #
#    setUserGenomeDbStatus($uid, $main::FALSE);
#    clearFileTaxon($uid);

    return;
}

###############################################################################
#
sub delUserChromosomeTables {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;

    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new("dbi:mysql:$dbname");
    if ($db->sta_connect()) {
        my($sql);
        my($sp) = getUserGenomeSpec($idUserGenome);
        my($seqno) = $idUserChromosome;

        # seqno 򥭡 chromosome/gene 
        $sql = "select * from chromosome_user where sp='$sp' and seqno=$seqno";
        my($sth) = $db->execute($sql); 
        while (my$ref = $sth->fetchrow_hashref()) {
            my($chrid) = $ref->{'id'};

            # chromosome
            $sql = "delete from chromosome_user where id=$chrid";
            $db->execute($sql);

            # gene
            $sql = "delete from gene_user where chrid=$chrid";
            $db->execute($sql);
        }
    }

    # Ϣ homology ̤
    delUserHomologyTables($uid, $idUserGenome);

    return;
}

###############################################################################
#
sub deleteUserChromosome {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;

    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    rmtree($dirUc);

    # MySQL ϿƤ Chromosome/Gene ǡ
    delUserChromosomeTables($uid, $idUserGenome, $idUserChromosome);

    #
    setUserGenomeDbStatus($uid, $main::FALSE);
#    clearFileTaxon($uid);

    return;
}

###############################################################################
#
sub setUserChromosomeStatus {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($status) = shift;

    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    my($fileStatus) = "$dirUc/statusOk";
print STDERR "STAT>>$status>>$fileStatus\n";
    if ($status) {
print STDERR "OPEN\n";
        new FileHandle(">$fileStatus");
    }
    else {
print STDERR "DELETE\n";
        unlink($fileStatus) if (-e $fileStatus);
    }

    return;
}

###############################################################################
#
sub getUserChromosomeStatus {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($status) = shift;

    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);
    my($fileStatus) = "$dirUc/statusOk";
    if (-e $fileStatus) {
        return $main::TRUE;
    }
    else {
        return $main::FALSE;
    }
}
sub getUserGenomeConvertingStatus {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($gdir) = getDirUserGenomeReady($uid, $idUserGenome);
    if (-e "$gdir/data/.status") {
	return $main::FALSE;
    } else {
	return $main::TRUE;
    }
}

###############################################################################
#
sub readNcbiGbk {
    my($fileGbk) = shift;
    my($gid) = shift;
    my($cid) = shift;
    my($refGeneTab) = {};

    #
    my($objGbk) = new GenBank($fileGbk);
    my($data) = $objGbk->read_entry();
    my($refGeneTab) = readGbkFile($data, $gid, $cid);
    return $refGeneTab;
}
sub readNcbiGbk_multi {
    my($fileGbk, $gid, $cid) = @_;
    my($refGeneTab) = {};

    # multi (contig) mode
    $refGeneTab->{WITH_CHR} = 1;
    #
    my($objGbk) = new GenBank($fileGbk);
    while ( my $data = $objGbk->read_entry() ) {
	readGbkFile($data, $gid, $cid, $refGeneTab, 'multi');
    }
    return $refGeneTab;
}
sub readGbkFile {
    my($data, $gid, $cid, $refGeneTab, $mode) = @_;
    my($msg);

    my($chrId) = $data->{accession} ? $data->{accession} : $data->{locus_name};

    if (! $refGeneTab) {
	$refGeneTab = {};

    #
    	$refGeneTab->{'HASH'} = {};
    	$refGeneTab->{'LIST'} = [];
    	$refGeneTab->{'MSG_ERR'} = [];
    }

    if ($mode eq 'multi') {
	# multi (contig) mode for incomplete data: storing chrom seqs ($seqObj) into an array @{ $refGeneTab{CHR_SEQ_MULTI} }
	my($seqObj) = { ID => $chrId, SEQ => $data->{'seq'} };
    	push(@{$refGeneTab->{'CHR_SEQ_MULTI'}}, $seqObj);
    } else {

    #
    	$refGeneTab->{'CHR_SEQ'} = $data->{'seq'};

    #
    	$refGeneTab->{'length'} = $data->{'length'};
    	$refGeneTab->{'shape'}  = $data->{'shape'};
    }

    my($ugSpec) = getUserGenomeSpec($gid);

    #
    my($feat);
    my($idx) = 0;
    if ($mode eq 'multi') {
	$idx = $refGeneTab->{TMPIDX};
    }
    foreach $feat (@{$data->{'features'}}) {
        next if ($feat->{'keyname'} !~ /^cds$/i);

        my($ent) = {};
        $ent->{'ID'}        = $feat->{'attr'}->{'locus_tag'};
        if ($ent->{'ID'} =~ /^\s*$/) {
            $ent->{'ID'} = sprintf("%s_%d_%04d", $ugSpec, $cid, $idx + 1);	# assign ID
        }
        my($id) = $ent->{'ID'};
        $ent->{'GENE_NAME'} = $feat->{'attr'}->{'gene'};
        $ent->{'FROM'}      = $feat->{'location'}->{'from'};
        $ent->{'TO'}        = $feat->{'location'}->{'to'};
        $ent->{'DIR'}       = 1;
        $ent->{'DIR'}       = -1 if ($feat->{'location'}->{'complement'});
        $ent->{'DESC'}      = $feat->{'attr'}->{'product'};
        $ent->{'PROT_SEQ'}       = $feat->{'attr'}->{'translation'};
        $ent->{'PROT_SEQ'} =~ s#\"##g;
	if ($mode eq 'multi') {
		$ent->{'CHRID'} = $chrId;
	}
        if (exists($refGeneTab->{'HASH'}->{"$id"})) {
            # Ʊ ID ΥǡϿ줿
            $msg = sprintf("Found same ID '%s'.", $id);
            push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

            next;
        }

        # OK ǡϡ$id 򥭡 hash ˳Ǽ
        $refGeneTab->{'HASH'}->{"$id"} = $ent;
        push(@{$refGeneTab->{'LIST'}}, $ent);

        $idx++;
    }
    if ($mode eq 'multi') {
	$refGeneTab->{TMPIDX} = $idx;
    }

    return $refGeneTab;
}

###############################################################################
#
sub readOrigGeneTab {
    my($fileGeneTab) = shift;
    my($fileFasta) = shift;
    my($refGeneTab) = {};
    my($msg);

    #
    my($fasta_ref) = readOrigFasta($fileFasta);

    #
    $refGeneTab->{'HASH'} = {};
    $refGeneTab->{'LIST'} = [];
    $refGeneTab->{'MSG_ERR'} = [];

    #
    my($fh) = new FileHandle("$fileGeneTab") || return $refGeneTab;
    my($lineNo) = 0;
    my($header);
    while(my$line = $fh->getline()) {
        foreach $_ (split(/\r/, $line)) {
            $lineNo++;

###            next if (/^\s*#/);
            next if (/^\s*$/);
	    chomp;
            s#[\r\n]*$##;

	    if (/^#/ && ! $header) {
		$header = readTextTabHeader($_);
		if ($header && /chr_acc/) {
			$refGeneTab->{WITH_CHR} = 1;
		}
		next;
	    }


            my($id, $geneName, $from, $to, $dir, $type, $desc, $chr_acc);
	    my($chr);
	    if ($header) {
		my($d) = readTextTabData($_, $header);
            	($id, $geneName, $from, $to, $dir, $type, $desc, $chr_acc) = 
			($d->{locus_tag},$d->{name},
			$d->{from},$d->{to},$d->{dir},$d->{type},
			$d->{descr}, $d->{chr_acc});
	    } else {
            	($id, $geneName, $from, $to, $dir, $desc) = split(/\t/);
	    }
            delete($fasta_ref->{'HASH'}->{"$id"});

            if ($id =~ /^\s*$/) {
                # ID ̤
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_NO_ID", 'ID', $lineNo);
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }

            if ($from =~ /^\s*$/) {
                # from ̤
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_NO_COLUMN", $id, 'From');
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }

            if ($to =~ /^\s*$/) {
                # to ̤
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_NO_COLUMN", $id, 'To');
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }

            if ($dir =~ /^\s*$/) {
                # dir ̤
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_NO_COLUMN", $id, 'Direction');
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }
	    # converting direction strings into {1,-1}
	    if ($dir eq '+' || $dir =~ /^F/i || $dir =~ /^DIR/i) {
		$dir = 1;
	    } elsif ($dir eq '-' || $dir =~ /^R/i || $dir =~ /^INV/i) {
		$dir = -1;
	    }

            my($ent) = {};
            $ent->{'ID'}        = $id;
            $ent->{'GENE_NAME'} = $geneName;
            $ent->{'FROM'}      = $from;
            $ent->{'TO'}        = $to;
            $ent->{'DIR'}       = $dir;
            $ent->{'DESC'}      = $desc;
            $ent->{'CHRID'}      = $chr_acc;
            $ent->{'TYPE'}      = $type;

            # ǡ
            if (($from == -1) && ($to == -1)) {
                # this entry has proteinseq only
            }
            elsif (($from !~ /^\s*\d+\s*$/) &&    # from ǹƤʤ
                ($to   !~ /^\s*\d+\s*$/) &&    # to   ǹƤʤ
                ($dir != -1) && ($dir != 1)
#			&&
#                ($dir ne '-') && ($dir ne '+') &&
#                ($dir !~ /^F/i) && ($dir !~ /^R/i) &&
#                ($dir !~ /^DIR/i) && ($dir !~ /^INV/i)
		) {
                # dir 
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_COL_TYPE");
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }

            if (exists($refGeneTab->{'HASH'}->{"$id"})) {
                # Ʊ ID ΥǡϿ줿
                $msg = sprintf("$MbgdUserGenomeCommon::ERR_MSG_SAME_ID", $id);
                push(@{$refGeneTab->{'MSG_ERR'}}, $msg);

                next;
            }

            # OK ǡϡ$id 򥭡 hash ˳Ǽ
            $refGeneTab->{'HASH'}->{"$id"} = $ent;
            push(@{$refGeneTab->{'LIST'}}, $ent);
        }
    }
    $fh->close();

    #
#    foreach my$id (keys(%{$fasta_ref->{'HASH'}})) {
#        my($ref) = $fasta_ref->{'HASH'}->{"$id"};
    foreach my$id (keys(%{$refGeneTab->{'HASH'}})) {
        my($ref) = $refGeneTab->{'HASH'}->{"$id"};
#        my($ent) = {};
        $ref->{'ID'} = $id;
        $ref->{'TYPE'} = 'CDS' if (! $ref->{'TYPE'});;

#        $refGeneTab->{'HASH'}->{"$id"} = $ent;
#        push(@{$refGeneTab->{'LIST'}}, $ent);
    }

    return $refGeneTab;
}
sub readTextTabHeader {
	my($line) = @_;
	my(@headers, %headers);
	my($i);
	if ($line =~ /^#(.*)$/) {
		@headers = split(/[\s,]/, $1);
		return \@headers;
	}
}
sub readTextTabData {
	my($line, $headers) = @_;
	my(@data) = split(/\t/, $line);
	my($retd);
	foreach my $h (@{$headers}) {
		$retd->{$h} = shift(@data);
	}
	return $retd;
}

###############################################################################
#
sub readOrigFasta {
    my($fileFasta) = shift;
    my($refFasta) = {};
    my($msg);

    $refFasta->{'HASH'} = {};
    $refFasta->{'LIST'} = [];
    $refFasta->{'MSG_ERR'} = [];

    #
    my($fh) = new FileHandle("$fileFasta") || return $refFasta;
    my($lineNo) = 0;
    my($id) = '';
    my($ent) = {};
    while(my$line = $fh->getline()) {
        foreach $_ (split(/\r/, $line)) {
            $lineNo++;

            s#[\r\n]*$##;

            next if (/^\s*#/);
            next if (/^\s*$/);

            if (/^>\s*(\S+)/) {
                my($wkId) = $1;
                if (($id !~ /^\s*$/) && ($ent->{'SEQ'} ne '')) {
                    $refFasta->{'HASH'}->{"$id"} = $ent;
                    push(@{$refFasta->{'LIST'}}, $ent);
                }

                #
                $ent = {};
                $id = $ent->{'ID'} = $wkId;
                $ent->{'SEQ'} = '';

                if (exists($refFasta->{'HASH'}->{"$id"})) {
                    # Ʊ ID ΥǡϿ줿
                    $msg = sprintf("ID '%s' Exists.", $id);
                    push(@{$refFasta->{'MSG_ERR'}}, $msg);
                }
            }
            else {
                $ent->{'SEQ'} .= $_;
            }
        }
    }
    if (($id !~ /^\s*$/) && ($ent->{'SEQ'} ne '')) {
        $refFasta->{'HASH'}->{"$id"} = $ent;
        push(@{$refFasta->{'LIST'}}, $ent);
    }
    $fh->close();

    return $refFasta;
}

###############################################################################
#
sub readGeneTab {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($refGeneTab) = {};

    my($dirUc) = MbgdUserGenomeCommon::getDirUserChromosome($uid,
                                                            $idUserGenome,
                                                            $idUserChromosome);

    #
    my($fileOrigGeneTab) = "$dirUc/origGeneTab";
    if (-e $fileOrigGeneTab) {
        my($fileOrigFasta) = "$dirUc/origFasta";
        return readOrigGeneTab($fileOrigGeneTab, $fileOrigFasta);
    }

    #
    my($fileGbk) = "$dirUc/ncbiGbk";
    if (-e $fileGbk) {
        return readNcbiGbk_multi($fileGbk, $idUserGenome, $idUserChromosome);
    }

    return $refGeneTab;
}

###############################################################################
#
sub clearFileTaxon {
    my($uid) = shift;

    my($dirUgDb) = MbgdUserGenomeCommon::getDirUserGenomeDatabase($uid);
    mkpath($dirUgDb, 0, 0750) if (! -e $dirUgDb);

    my($fileTax) = "$dirUgDb/tax";
    new FileHandle(">$fileTax");

    return;
}

###############################################################################
#
sub createFileTaxon {
    my($uid) = shift;

    my($dirUgDb) = MbgdUserGenomeCommon::getDirUserGenomeDatabase($uid);
    mkpath($dirUgDb, 0, 0750) if (! -e $dirUgDb);

    #
    my($fileTaxMbgd) = "$ENV{'MBGD_HOME'}/database/tax";
    my($fileTax) = "$dirUgDb/tax";
    if (! -e $fileTax) {
        clearFileTaxon($uid);
    }
    elsif (-M $fileTaxMbgd < -M $fileTax) {
        # MBGD ǻѤƤ tax ե뤬
        # ǡԤ줿Ȼפ
        # return  tax ե
    }
    elsif (! -z $fileTax) {
        # tax եϽѤ
        return;
    }

    # ͭ Taxonomy ID Ǥ뤫åΤ taxon.tab ɤ߹
    my($fileTaxonTab) = "$ENV{'MBGD_HOME'}/package/taxonomy/db/taxon.tab";
    my($hashTaxonId) = {};
    my($fhTaxon) = new FileHandle("$fileTaxonTab");
    while($_ = $fhTaxon->getline()) {
        my($taxId) = (/^(\d+)/);
        $hashTaxonId->{"$taxId"} = 1;
    }
    $fhTaxon->close();

    #
    my($fileUg)  = "$dirUgDb/ug";
    my($fh) = new FileHandle(">$fileUg");

    my($idUserGenome);
    for($idUserGenome = 1; $idUserGenome <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $idUserGenome++) {

        my($refUserGenome) = MbgdUserGenomeCommon::loadUserGenomeInfo($uid, $idUserGenome);
        if (! $refUserGenome) {
            next;
        }

        if ($refUserGenome->{'status'} !~ /^ok$/i) {
            next;
        }

        my($taxId) = $refUserGenome->{'taxonomy_id'};
        if (! exists($hashTaxonId->{"$taxId"})) {
            $taxId = 32644;                          # unknown organism
        }
        my($ugSpec) = getUserGenomeSpec($idUserGenome);
        $fh->print(join("\t", $ugSpec,
                              $refUserGenome->{'species_name'},
                              $refUserGenome->{'strain'},
                              $taxId ), "\n");
    }
    $fh->close();

    #
    my($cmdUpdTaxTree) = "$ENV{'MBGD_HOME'}/build/1150UpdateTaxTree.pl";
    my($cmd) = "$cmdUpdTaxTree -user_genome=$fileUg -dbname=$main::DBNAME_MBGD";
writeLogMsg($uid, "LOG :: update $fileTax");
    system("$cmd > $fileTax.$$");
    unlink("$fileTax.bak") if (-e "$fileTax.bak");
    rename("$fileTax", "$fileTax.bak");
    rename("$fileTax.$$", "$fileTax");
}

###############################################################################
#
sub createUserGenomeTableHomologyMerge {
    my($db) = shift;

    #
    my($sql) = "drop table if exists homology";
    $db->do($sql);

    # mbgd.homology  homology_?? ơ֥򥫥
    my(@mbgdHomologyTab);
    my($dbMbgd) = new MBGD::DB($main::DBNAME_MBGD);
    for(my$i = 0; ; $i++) {
        my($tabname) = sprintf("homology_%02d", $i);
        my($sta) = $dbMbgd->exist_table($tabname);
        if (!$sta) {
            last;
        }
        push(@mbgdHomologyTab, "mbgd.$tabname");
    }

    # homology(merge) ơ֥κ
    my($mrgTab) = join(',', @mbgdHomologyTab, 'homology_user');
    my($opt) = {};
    $opt->{'merge_table'} = sprintf("type=merge union=(%s) insert_method=last", $mrgTab);
    my($tab) = new MBGD::DBTable($db, 'Homology');
    $tab->create_table($opt);

    return;
}

###############################################################################
#
sub createUserGenomeTable {
    my($uid) = shift;
    my($tabName);
    my(@tabList) = ('Genome',
                    'DNASeq',
                    'GeneSeq',
                    'ProteinSeq',
                    'Chromosome',
                    'Gene',
                    'Homology',
                    'Motif',
                    'ProtMotif');

    #
    my($dbname) = MbgdUserGenomeCommon::getDbname($uid);
    my($fileOld) = "$ENV{'MBGD_HOME'}/schema/$main::DBNAME_MBGD";
    my($fileNew) = "$ENV{'MBGD_HOME'}/schema/$dbname";
#    symlink("$fileOld", "$fileNew") if (! -e "$fileNew");

    #
    my($db) = new MBGD::DB($dbname);
    foreach $tabName (@tabList) {
        my($opt) = {};
        $opt->{'tablename'} = lc($tabName) . "_user";
        my($tab) = new MBGD::DBTable($db,
                                     $tabName,
                                     $opt);

        # create_table() ᥽åɤȤ $MBGD_HOME/schema ʲ
        # $dbname, user, base ν Schema 
        my($opt_create_tab) = {};
        $opt_create_tab->{'auto_increment'} = $MbgdUserGenomeCommon::MYSQL_auto_increment;
        $tab->create_table($opt_create_tab);
    }

    #
    my($cols);
    my($vals);
    foreach $tabName (@tabList) {
        my($id)   = 2000000000;    # int ξ : 2147483647
        my($spid) = 'gm99999';
        my($sp)   = 'dummy';
        my($name) = 'dummy';
        if ($tabName =~ /^genome$/i) {
            $cols = "id,sp,spid";
            $vals = "$id,'$sp','$spid'";
        }
        elsif ($tabName =~ /^chromosome$/i) {
            $cols = "id,genome,sp,spid";
            $vals = "$id,$id,'$sp','$spid'";
        }
        elsif ($tabName =~ /^gene$/i) {
            $cols = "id,sp,name,chrid";
            $vals = "$id,'$sp','$name',$id";
        }
        elsif ($tabName =~ /^(dna|gene|protein)seq$/i) {
            $cols = "id,chksum,length";
            $vals = "$id,'',0";
        }
        elsif ($tabName =~ /^protmotif$/i) {
            $cols = "id,sp,name,motlib,motid";
            $vals = "$id,'$sp','$name','$name','$name'";
        }
        else {
            next;
        }

        my($tab) = lc($tabName) . '_user';
        my($sql) = "insert IGNORE $tab ($cols) values($vals)";
        $db->execute($sql);
    }

    # merge ơ֥
    my($mrgTabSrc) = $main::DBNAME_MBGD;
    foreach $tabName (@tabList) {
        next if ($tabName =~ /homology/i); # mbgd.homology  MERGE ơ֥Ǥ뤿ᡢۤɺƺ롣

        my($tab) = new MBGD::DBTable($db,
                                     $tabName);

        my($opt) = {};
        $opt->{'merge_table'} = sprintf("type=merge union=(%s.%s,%s_user) insert_method=last", $mrgTabSrc, lc($tabName), lc($tabName));
        $tab->create_table($opt);
    }

    # merge ơ֥ for homology
    createUserGenomeTableHomologyMerge($db);

    return;
}

###############################################################################
#
sub createUserGenomeDatabase {
    my($uid) = shift;
    print STDERR "DBG :: create database for $uid\n";

    #
    my($dbname) = getDbname($uid);
    if ($dbname =~ /^mbgd$/i) {
        print STDERR "WARNING :: dbname is 'mbgd'!! (Use mbgd)\n";
        return;
    }
    if ($dbname =~ /^mbgd_accum$/i) {
        print STDERR "WARNING :: dbname is 'mbgd_accum'!! (Use mbgd_accum)\n";
        return;
    }
    if ($dbname =~ /^mbgd_$/i) {
        print STDERR "WARNING :: dbname is 'mbgd_'!! (NO UID)\n";
        return;
    }

    #
    my($db) = new MBGD::DB('mysql');
    my($sql);

    #
    $sql = sprintf("create database IF NOT EXISTS %s", $dbname);
#	print STDERR "SQL :: $sql\n";
#    $db->do($sql);

    return;
}

###############################################################################
#
# create chromosome and gene data for the given chromsome
sub convUserGenome_ug_chr {
    my($uid) = shift;
    my($gid) = shift;
    my($cid) = shift;
    my($refUserChromosome) = shift;
    my($fhps) = shift;
    my($dirname) = shift;
    my($format) = shift;

    open(O, ">$dirname/.status");
    print O "converting\n";
    close(O);

#    ## clear all files
#    opendir(D, $dirname);
#    while (my $f = readdir(D)) {
#	next if ($f =~ /^\./);
#	unlink("$dirname/$f");
#    }


    # chromosome
    my$filename = sprintf("%s/%s.chromosome.txt", $dirname, $cid);
    my$fh = new FileHandle(">$filename") || return;

    #
    my($spid) = getUserGenomeSpid($gid);
    my($sp)   = getUserGenomeSpec($gid);

    # spec
    $fh->print('spid', "\t", $spid, "\n");
    $fh->print('sp',   "\t", $sp,   "\n");
    # seqno
    $fh->print('seqno', "\t", $cid, "\n");
    # name
    $fh->print('name', "\t", $refUserChromosome->{'chromosome_name'}, "\n");
    # type
    $fh->print('type', "\t", 'chromosome', "\n");
    # shape
    $fh->print('shape', "\t", $refUserChromosome->{'chromosome_shape'}, "\n");
    # accession
    $fh->print('accession', "\t", "$sp-$cid", "\n");
    # gi
    $fh->print('gi', '', "", "\n");

    $fh->close();

    # gene
    my($refUserGene) = loadUserGeneInfo($uid, $gid, $cid);
    my($fh_gene);
    if ($refUserGene->{WITH_CHR}) {
	$format = 'full';
    } else {
##    	my($file_gene) = sprintf("%s/%s.%s", $dirname, $cid, 'gene');
    	my($file_gene) = "$dirname/$cid.gene";
    	$fh_gene = FileHandle->new(">$file_gene") || return;
    }
    my($refGene);
    foreach $refGene (@{$refUserGene->{'LIST'}}) {
        my(@geneInfo);
	if ($format eq 'full') {
		#sp, locustag, gene, chrAcc, from, to, dir, location, type, transl_table, codon_start, protid, gi, geneid, descr
		my($chrid) = "$sp.$cid";
		if ($refGene->{CHRID}) {
			$chrid = $refGene->{CHRID};
		}
		@geneInfo = ($spid, $refGene->{ID}, $refGene->{GENE_NAME}, $chrid,
			$refGene->{FROM}, $refGene->{TO}, $refGene->{DIR},
			'', 'CDS', '', '', '', '', '', $refGene->{DESC});
	} else {
		#locustag, gene, from, to, dir, type, descr
		@geneInfo = ($refGene->{ID}, $refGene->{GENE_NAME},
			$refGene->{FROM}, $refGene->{TO}, $refGene->{DIR}, 'CDS', $refGene->{DESC});
	}
        # spec
##        push(@geneInfo, $sp);
#        push(@geneInfo, $spid);
#        # orf
#        push(@geneInfo, $refGene->{'ID'});
#        # gene
#        push(@geneInfo, $refGene->{'GENE_NAME'});
#        # chrAcc
#        push(@geneInfo, "$spid.$cid");
#        # from
#        push(@geneInfo, $refGene->{'FROM'});
#        # to
#        push(@geneInfo, $refGene->{'TO'});
#        # dir
#        push(@geneInfo, $refGene->{'DIR'});
#        # location
#        push(@geneInfo, "");
#        # type
#        push(@geneInfo, "CDS");
#        # transl_table
#        push(@geneInfo, "");
#        # codon_start
#        push(@geneInfo, "");
#        # protid
#        push(@geneInfo, "");
#        # gi
#        push(@geneInfo, "");
#        # geneid
#        push(@geneInfo, "");
#        # descr
#        push(@geneInfo, $refGene->{'DESC'});

	if ($refUserGene->{WITH_CHR}) {
	    close($fh_gene) if ($fh_gene);
    	    my($file_gene) = "$dirname/$refGene->{CHRID}.gene";
    	    $fh_gene = FileHandle->new(">>$file_gene") || return;
	}
        $fh_gene->print(join("\t", @geneInfo), "\n");
    }
#    $fh_gene->close();

    # protseq
    my($fh_protseq);
    if ($refUserGene->{WITH_CHR}) {
    } else {
    	my($file_protseq) = sprintf("%s/%s.%s", $dirname,
                                            $cid,
                                            'protseq');
    	$fh_protseq = FileHandle->new(">$file_protseq") || return;
    }
    foreach $refGene (@{$refUserGene->{'LIST'}}) {
        my($orf) = $refGene->{'ID'};
        my($seq) = $refGene->{'PROT_SEQ'};
        $seq =~ s#(.{1,60})#$1\n#g;

        next if ($seq =~ /^\s*$/);
	if ($refUserGene->{WITH_CHR}) {
	    close($fh_protseq) if ($fh_protseq);
    	    my($file_protseq) = "$dirname/$refGene->{CHRID}.protseq";
    	    $fh_protseq = FileHandle->new(">>$file_protseq") || return;
	}

        $fh_protseq->print(">$sp:$orf", "\n");
        $fh_protseq->print($seq);

        # UserGenome Ƥ ProteinSeq 򣱥եˤޤȤʥǡѡ
        $fhps->print(">$sp:$orf", "\n");
        $fhps->print($seq);
    }

    #
    foreach my$seq_ref (@{$refUserGene->{'PROT_SEQ_SET'}->{'LIST'}}) {
        my($orf) = $seq_ref->{'ID'};
        if (exists($refUserGene->{'HASH'}->{"$orf"})) {
            next;
        }

        #
        my(@geneInfo);
	if ($format eq 'full') {
		#sp, locustag, gene, chrAcc, from, to, dir, location, type, transl_table, codon_start, protid, gi, geneid, descr
		@geneInfo = ($spid, $orf, '', "$sp.$cid", -1, -1, 1, '', 'CDS', '', '', '', '', '', '');
#        	push(@geneInfo, $sp);
#        	push(@geneInfo, $orf);
#        	push(@geneInfo, '');
#        	push(@geneInfo, "$sp.$cid");
#        	push(@geneInfo, -1);
#        	push(@geneInfo, -1);
#        	push(@geneInfo, 1);
#        	push(@geneInfo, '');
#        	push(@geneInfo, 'CDS');
#        	push(@geneInfo, '');
#        	push(@geneInfo, '');
#        	push(@geneInfo, '');
#        	push(@geneInfo, '');
#        	push(@geneInfo, '');
#        	push(@geneInfo, '');
	} else {
		#locustag, gene, from, to, dir, type, descr
		@geneInfo = ($orf, '', -1, -1, 1, 'CDS', '');
	}
        $fh_gene->print(join("\t", @geneInfo), "\n");

        #
        my($seq) = $seq_ref->{'SEQ'};
        $seq =~ s#(.{1,60})#$1\n#g;

        #
        $fh_protseq->print(">$sp:$orf", "\n");
        $fh_protseq->print($seq);

        # UserGenome Ƥ ProteinSeq 򣱥եˤޤȤʥǡѡ
        $fhps->print(">$sp:$orf", "\n");
        $fhps->print($seq);
    }

    #
    $fh_gene->close();
    $fh_protseq->close();

    # chrseq
    my($fh);
    if ($refUserGene->{WITH_CHR}) {
	foreach my $sqdata (@{$refUserGene->{CHR_SEQ_MULTI}}) {
		my($filename) = "$dirname/$sqdata->{ID}.chrseq";
    		$fh = new FileHandle(">>$filename") || return;
		$fh->print(">$sqdata->{ID}\n");
		my($seq) = $sqdata->{'SEQ'};
    		$seq =~ s#(.{1,60})#$1\n#g;
		$fh->print($seq."\n");
		close($fh);
	}
    } else {
    	$filename = sprintf("%s/%s.%s", $dirname,
                                    $cid,
                                    'chrseq');
    	$fh = new FileHandle(">$filename") || return;
    	$fh->print('>dna sequence', "\n");
    	my($len) = $refUserChromosome->{'chromosome_length'};
    	my($seq);
    	if ($refUserGene->{'CHR_SEQ'} !~ /^\s*$/) {
        	$seq = $refUserGene->{'CHR_SEQ'};
    	} else {
        # ߡ
        	$seq = 'n' x $len;
	}
    	$seq =~ s#(.{1,60})#$1\n#g;
    	$fh->print($seq, "\n");
    	$fh->close();
    }

    # geneseq
    if ($refUserGene->{WITH_CHR}) {
    } else {
    	$filename = sprintf("%s/%s.%s", $dirname,
                                    $cid,
                                    'geneseq');
    	$fh = new FileHandle(">$filename") || return;
    	$fh->close();
    }
    unlink("$dirname/.status");

    return;
}

###############################################################################
#
# create genome.txt
sub convUserGenome_ug_txt {
    my($uid) = shift;
    my($gid) = shift;
    my($filename);
    my($fh);

    #
    my($spid) = getUserGenomeSpid($gid);
##    my($sp)   = getUserGenomeSpec($gid);

    #
    my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
    if (! $refUserGenome) {
        return;
    }

    if ($refUserGenome->{'status'} !~ /^Ok$/) {
#        return;
    }

    #
    my($dirUgDb) = getDirBaseUserGenome($uid);
    my($dirUgDbSp) = "$dirUgDb/species";
    my($dir) = "$dirUgDbSp/$spid/gm";
    my($dirname) = "$dir/data";
    mkpath("$dirname", 0, 0750) if (! -e $dirname);

    #
    my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
    $filename = sprintf("%s/genome.txt", $dir);
    $fh = new FileHandle(">$filename") || return;

    # spec
    $fh->print('spid', "\t", $spid, "\n");
#    $fh->print('sp',   "\t", $sp,   "\n");
    $fh->print('sp',   "\t", $refUserGenome->{'sp'},   "\n");

    # abbreviation
    $fh->print('abbrev', "\t", $refUserGenome->{'abbreviation_name'}, "\n");
    # orgname
    $fh->print('orgname', "\t", $refUserGenome->{'species_name'}, "\n");
    # strain
    $fh->print('strain', "\t", $refUserGenome->{'strain'}, "\n");
    # taxid
    $fh->print('taxid', "\t", $refUserGenome->{'taxonomy_id'}, "\n");
    # type
    $fh->print('type', "\t", $refUserGenome->{'type'}, "\n");
    # source
    $fh->print('source', "\t", '', "\n");
    # institution
    $fh->print('institution', "\t", '', "\n");
    # wwwlink
    $fh->print('wwwlink', "\t", '', "\n");
    # medid
    $fh->print('medid', "\t", '', "\n");
    # journal
    $fh->print('journal', "\t", '', "\n");
    #
    $fh->close();

    return $refUserGenome;
}

###############################################################################
#
# creating "ug" data from the uploaded "gm" data; called from WWW/bin/conv_gm2ug.pl
sub convUserGenome_ug {
    my($uid) = shift;
    my($gid) = shift;

    #
    my($spid) = getUserGenomeSpid($gid);
    my($refUserGenome) = convUserGenome_ug_txt($uid, $gid);
    my($sp)   = getUserGenomeSpec($gid);

    #
    my($dirUgDb) = getDirBaseUserGenome($uid);
    my($dirUgDbSp) = "$dirUgDb/species";
    my($dir) = "$dirUgDbSp/$spid/gm";
    my($dirname) = "$dir/data";
    mkpath("$dirname", 0, 0750) if (! -e $dirname);

    #
    my($fileProtSeq) = sprintf("%s/%s.protseq", $dir, $sp);
    my($fhps) = new FileHandle(">$fileProtSeq");
    my($refUserChromosome);
    my($cid) = 1;

    ## clear all files
    opendir(D, $dirname);
    while (my $f = readdir(D)) {
	next if ($f =~ /^\./);
	unlink("$dirname/$f");
    }

    foreach $refUserChromosome (@{$refUserGenome->{'chromosome'}}) {
        next if (! $refUserChromosome);

        convUserGenome_ug_chr($uid, $gid, $cid, $refUserChromosome, $fhps, $dirname);

        $cid++;
    }
    $fhps->close();

    return $sp;
}

###############################################################################
#
sub convUserGenome {
    my($uid) = shift;
    my(@spUgList);
    my($gid);
    my($filename);
    my($fh);

    #
    for($gid = 1; $gid <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $gid++) {
        my($spUg) = convUserGenome_ug($uid, $gid);

        next if (!$spUg);

        push(@spUgList, $spUg);
    }

    return @spUgList;
}

###############################################################################
#
sub createSptitSymlink {
    my($uid) = shift;

    my($dirMbgdSptit) = "$ENV{'MBGD_HOME'}/database/sptit";

    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUgDbSptit) = "$dirUgDb/sptit";

    # clear OLD sptit data directory
    rmtree($dirUgDbSptit, 0) if (-e $dirUgDbSptit);
    mkpath($dirUgDbSptit, 0, 0750);

    my($dh) = new DirHandle($dirMbgdSptit) || die("Can not open $dirMbgdSptit($!)");
    my($file);
    while($file = $dh->read()) {
        next if ($file !~ /^sptit\./);

        my($fileOld) = sprintf("%s/%s", $dirMbgdSptit, $file);
        my($fileNew) = sprintf("%s/%s", $dirUgDbSptit, $file);
#        unlink($fileNew) if (-e $fileNew);
        symlink($fileOld, $fileNew) if (! -e "$fileNew");
    }
}

###############################################################################
#
sub createSptitForUserGenome {
    my($uid) = shift;

    #
    my(@spList);
    my($gid);
    for($gid = 1; $gid <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $gid++) {
        my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
        if (! $refUserGenome) {
            next;
        }

        if ($refUserGenome->{'status'} !~ /^Ok$/) {
            next;
        }

        push(@spList, getUserGenomeSpec($gid));
    }

    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                      '0330MakeSptit.pl');

writeLogMsg($uid, "LOG :: Make sptit for $uid");
    system("$cmd -usergenome=$uid @spList");
}

###############################################################################
#
sub loadUserGenome {
    my($uid) = shift;
    my($dbname) = getDbname($uid);

    #
    my($fileOld) = "$ENV{'MBGD_HOME'}/schema/mbgd";
    my($fileNew) = sprintf("%s/schema/$dbname", $ENV{'MBGD_HOME'});
#    unlink($fileNew);
#    symlink($fileOld, $fileNew);

    #
    my($cmd) = "$ENV{'MBGD_HOME'}/build/0420UpdateSpecInfo.pl";
    my($opt) = "-usergenome=$uid -dbname=$dbname";

    #
    my($gid);
    for($gid = 1; $gid <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $gid++) {
        my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
        if (! $refUserGenome) {
            next;
        }

        if ($refUserGenome->{'status'} !~ /^Ok$/) {
            next;
        }

        my($sp) = getUserGenomeSpec($gid);
writeLogMsg($uid, "LOG :: Add SPEC($sp) information($dbname)");
        system("$cmd $opt $sp");
    }
}

###############################################################################
#
sub convFastaMd5 {
    my($uid) = shift;
    my($dbname) = getDbname($uid);

    #
    my(@d) = localtime(time());
    my($today) = sprintf("%04d%02d%02d", $d[5]+1900, $d[4]+1, $d[3]);
    my($cmd) = "$ENV{'MBGD_HOME'}/build/0490SelectUpdSeq.pl";
    my($spList) = join( ',', &getUserGenomeIdList($uid) );
    my($opt) = "-dbname=$dbname -SPEC='$spList'";

    #
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($fileFastaMd5All) = "$dirUgDb/fastaMd5.txt";

#print STDERR "EXEC :: $cmd $opt\n";
    system("$cmd $opt > $fileFastaMd5All");
#print STDERR "CMD>>> $cmd $opt > $fileFastaMd5All\n";
    if (-z $fileFastaMd5All) {
	return 0;
    } else {
	return 1;
    }
}

###############################################################################
#
sub clearFinishedSpecAll {
    my($type) = shift;
    my($uid) = shift;

    my($dirUid) = getDirBaseUserGenome($uid);
    mkpath($dirUid, 0, 0750) if (! -e $dirUid);

    # ƥꥢ
    my($filename) = sprintf("$dirUid/%sFinished.spec", $type);
    new FileHandle(">$filename");
}

###############################################################################
#
sub clearBlastFinishedSpecAll {
    my($uid) = shift;
    my($type) = 'blast';

    clearFinishedSpecAll($type, $uid);
}

###############################################################################
#
sub clearMotifFinishedSpecAll {
    my($uid) = shift;
    my($type) = 'motif';

    clearFinishedSpecAll($type, $uid);
}

###############################################################################
#
sub delFinishedSpec {
    my($ref) = shift;
    my($sp0) = shift;

    my($refList) = [];
    foreach my$sp (@{$ref}) {
        push(@{$refList}, $sp) if ($sp !~ /^$sp0$/i);
    }

    return $refList;
}

###############################################################################
#
sub clearFinishedSpec {
    my($type) = shift;
    my($uid) = shift;
    my($sp1) = shift;
    my($sp2) = shift;
    my($fh);

    if (! isUserGenome($sp1)) {
        return;
    }

    my($dirUid) = getDirBaseUserGenome($uid);
    mkpath($dirUid, 0, 0750) if (! -e $dirUid);

    my($ref) = readFinishedSpec($type, $uid);
    if (! $sp2) {
        # ꤵ줿 UserGenome θ̵Ͽ
        delete($ref->{"$sp1"});

        if ($type =~ /^blast$/i) {
            # ¾ UserGenome Ȥ homology ̵Ͽ
            foreach my$sp2 (keys(%{$ref})) {
                $ref->{"$sp2"} = delFinishedSpec($ref->{"$sp2"}, $sp1);
            }
        }
    }
    else {
        # ꤵ줿ʪڥ homology ̵Ͽ
        $ref->{"$sp1"} = delFinishedSpec($ref->{"$sp1"}, $sp2);
        if (isUserGenome($sp2)) {
            $ref->{"$sp2"} = delFinishedSpec($ref->{"$sp2"}, $sp1);
        }
    }
    writeFinishedSpec($type, $uid, $ref);
}

###############################################################################
#
sub clearBlastFinishedSpec {
    my($uid) = shift;
    my($sp1) = shift;
    my($sp2) = shift;
    my($type) = 'blast';

    clearFinishedSpec($type, $uid, $sp1, $sp2);
}

###############################################################################
#
sub clearMotifFinishedSpec {
    my($uid) = shift;
    my($sp1) = shift;
    my($sp2) = shift;
    my($type) = 'motif';

    clearFinishedSpec($type, $uid, $sp1, $sp2);
}

###############################################################################
#
sub readFinishedSpec {
    my($type) = shift;
    my($uid) = shift;
    my($ref) = {};
    my($fh);
    my($line);

    #
    my($dirUid) = getDirBaseUserGenome($uid);
    mkpath($dirUid, 0, 0750) if (! -e $dirUid);
    my($filename) = sprintf("$dirUid/%sFinished.spec", $type);

    # Ѥߤʪɤ߹
    $fh = new FileHandle("$filename");
    if ($fh) {
        while($line = $fh->getline()) {
            $line =~ s#[\r\n]*$##;

            my($spUg, $splist) = split(/\t/, $line);
            next if (! isUserGenome($spUg));

            $ref->{"$spUg"} = [];
            push(@{$ref->{"$spUg"}}, split(/\s/, $splist));
        }
        $fh->close();
    }

    return $ref;
}

###############################################################################
#
sub readBlastFinishedSpec {
    my($uid) = shift;
    my($type) = 'blast';

    return readFinishedSpec($type, $uid);
}

###############################################################################
#
sub readMotifFinishedSpec {
    my($uid) = shift;
    my($type) = 'motif';

    return readFinishedSpec($type, $uid);
}

###############################################################################
#
sub writeFinishedSpec {
    my($type) = shift;
    my($uid) = shift;
    my($ref) = shift;
    my($fh);

    #
    my($dirUid) = getDirBaseUserGenome($uid);
    mkpath($dirUid, 0, 0750) if (! -e $dirUid);
    my($filename) = sprintf("$dirUid/%sFinished.spec", $type);

    # Ѥߤʪν
    $fh = new FileHandle(">$filename");
    foreach my$sp (sort(keys(%{$ref}))) {
        $fh->print($sp, "\t", join(' ', @{$ref->{"$sp"}}), "\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub writeBlastFinishedSpec {
    my($uid) = shift;
    my($ref) = shift;
    my($type) = 'blast';

    writeFinishedSpec($type, $uid, $ref);
}

###############################################################################
#
sub writeMotifFinishedSpec {
    my($uid) = shift;
    my($ref) = shift;
    my($type) = 'motif';

    writeFinishedSpec($type, $uid, $ref);
}

###############################################################################
#
sub addFinishedSpec {
    my($type) = shift;
    my($uid) = shift;
    my($spUgList) = shift;
    my(@spList) = @_;

    #
    my($ref) = readFinishedSpec($type, $uid);

    #
    foreach my$spUg (split(/\,/, $spUgList)) {
        next if (! isUserGenome($spUg));

        if (! exists($ref->{"$spUg"})) {
            $ref->{"$spUg"} = [];
        }

        my(%spHash) = ();
        foreach my$sp (@{$ref->{"$spUg"}}, @spList) {
            $spHash{"$sp"} = 1;
        }
        $ref->{"$spUg"} = [ sort(keys(%spHash)) ];
    }

    #
    writeFinishedSpec($type, $uid, $ref);
}

###############################################################################
#
sub addBlastFinishedSpec {
    my($uid) = shift;
    my($spUgList) = shift;
    my(@spList) = @_;
    my($type) = 'blast';

    addFinishedSpec($type, $uid, $spUgList, @spList);
}

###############################################################################
#
sub addMotifFinishedSpec {
    my($uid) = shift;
    my($spUgList) = shift;
    my(@spList) = @_;
    my($type) = 'motif';

    addFinishedSpec($type, $uid, $spUgList, @spList);
}

###############################################################################
# ۥоݤʪꥹȤ֤
sub getBlastTargetSpecUg {
    my($uid) = shift;
    my($ug) = shift;

    #
    my(@spList_unfinished);

    if (! $ug) {
        return @spList_unfinished;
    }

    #
    my(%spHash_finished);
    my($sp);

    # ѥ᡼եꡢ饹оݤʪ
    my($uInfo) = UserInfo->new($uid);
	my($homParam) = $uInfo->loadHomolParams();
	my($param) = $homParam->asHashRef();
    my(@targetSpecList) = split(/\|/, $param->{'species'});
    my(%targetSpecHash);
    foreach my$sp (@targetSpecList) {
        $targetSpecHash{"$sp"} = 1;
    }
    if (!$targetSpecHash{"$ug"}) {
        return @spList_unfinished;
    }

    # BLAST+dp λƤʪΰ
    my($ref) = readBlastFinishedSpec($uid);
    if (! exists($ref->{"$ug"})) {
        #  ug ϡBLAST+dp ̤¹ԤǤ
        return @targetSpecList;
    }

    #
    foreach my$sp (@{$ref->{"$ug"}}) {
        # BLAST+dp λƤʪ
        $spHash_finished{"$sp"} = 1;
    }

    # BLAST+dp ɬפʪΰ
    foreach my$sp (@targetSpecList) {
        next if ($spHash_finished{"$sp"});      # BLAST+dp λѤߤ SPEC

        push(@spList_unfinished, "$sp");
    }
print STDERR "Unfinished>>>$ug, @spList_unfinished<<<\n";

    return @spList_unfinished;
}

###############################################################################
#
sub getBlastTargetSpec {
    my($uid) = shift;
    my($ug) = shift;

    #
    my(@spList_unfinished);

    #
    my(@spUgList) = ($ug);
    if (! $ug) {
        # Ƥ UserGenome ФĴ٤
        @spUgList = getUserGenomeIdListStatusOk($uid);
    }

    #
    my(%spec_hash);
    foreach my$sp (@spUgList) {
        my(@spec_list) = getBlastTargetSpecUg($uid, $sp);
        foreach my$s (@spec_list) {
            $spec_hash{"$s"} = 1;
        }
    }
    @spList_unfinished = keys(%spec_hash);

    return @spList_unfinished;
}

###############################################################################
#
sub getMotifTargetSpecUg {
    my($uid) = shift;
    my($ug) = shift;

    if (! $ug) {
        return;
    }

    #
    my(%spHash_finished);
    my($sp);

    # ѥ᡼եꡢ饹оݤʪ
    my($uInfo) = UserInfo->new($uid);
	my($homParam) = $uInfo->loadHomolParams();
	my($param) = $homParam->asHashRef();
    my(@targetSpecList) = grep {/^ug\d+$/} split(/\|/, $param->{'species'});
    my(%targetSpecHash);
    foreach my$sp (@targetSpecList) {
        $targetSpecHash{"$sp"} = 1;
    }
    if (!$targetSpecHash{"$ug"}) {
        return;
    }

    # rps-blast λƤʪΰ
    my($ref) = readMotifFinishedSpec($uid);
    if (! exists($ref->{"$ug"})) {
        #  ug ϡrps-blast ̤¹ԤǤ
        return $ug;
    }

    return;
}

###############################################################################
#
sub getMotifTargetSpec {
    my($uid) = shift;
    my($ug) = shift;

    #
    my(@spList_unfinished);

    #
    my(@spUgList) = ($ug);
    if (! $ug) {
        # Ƥ UserGenome ФĴ٤
        @spUgList = getUserGenomeIdListStatusOk($uid);
    }

    #
    my(%spec_hash);
    foreach my$sp (@spUgList) {
        my(@spec_list) = getMotifTargetSpecUg($uid, $sp);
        foreach my$s (@spec_list) {
            $spec_hash{"$s"} = 1;
        }
    }
    @spList_unfinished = keys(%spec_hash);

    return @spList_unfinished;
}

###############################################################################
#
sub isContainUserGenome {
    my($uid) = shift;

    # 饹оݤʪΰ
    my($uInfo) = UserInfo->new($uid);
    my($homParam) = $uInfo->loadHomolParams();
    my($param) = $homParam->asHashRef();
    my(@targetSpecList) = split(/\|/, $param->{'species'});
#print STDERR "PP>$uid, $param->{species}\n";

    my($sp);
    foreach $sp (@targetSpecList) {
        if (isUserGenome($sp)) {
            return 1;
        }
    }
    return 0;
}

###############################################################################
#
#sub getBlastTargetSpList {
#    my($uid) = shift;
#    my($fileDb) = shift;
#    my($n) = 0;
#    my($sql);
#    my($ref);
#    my($sp);
#
#    #
#    my($dbname) = getDbname($uid);
#    my($db) = new MBGD::DB("dbi:mysql:$dbname");
#    my($sth);
#
#    # оݤȤʤʪ
#    my(@spList) = getBlastTargetSpec($uid);
#    if (scalar(@spList) == 0) {
#        # оݤȤʤʪ̵
#        return $n;
#    }
#
#    # SQL ǻѤ뤿 ' ǳ
#    foreach $sp (@spList) {
#        $sp = "'$sp'";
#    }
#
#    #
#    my($splist) = join(',', @spList);
#    $sql = "select * from proteinseq p, gene g "
#         . "where g.sp in($splist) and p.id=g.aaseq";
#    $sth = $db->execute($sql);
#
#    return $n;
#}

###############################################################################
#
sub loadHomSearchRes {
    my($uid) = shift;
    my($fileHomSearchRes) = shift;
    my($dbname) = getDbname($uid);

    my($fileOld) = "$ENV{'MBGD_HOME'}/schema/mbgd";
    my($fileNew) = sprintf("%s/schema/$dbname", $ENV{'MBGD_HOME'});
#    unlink($fileNew);
#    symlink($fileOld, $fileNew);

    #
    convertHomology("$fileHomSearchRes", "$fileHomSearchRes.2");

    #
    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                     '1010LoadHomology.pl');

    #
    my($opt) = "-TYPE=Homology -dbname=$dbname";

    #
writeLogMsg($uid, "LOG :: Load homology results($dbname)");
    system("$cmd $opt $fileHomSearchRes.2");
    unlink("$fileHomSearchRes.2");
}

###############################################################################
#
sub loadRpsBlastRes {
    my($uid) = shift;
    my($fileRpsBlastRes) = shift;
    my($dbname) = getDbname($uid);

    my($fileOld) = "$ENV{'MBGD_HOME'}/schema/mbgd";
    my($fileNew) = sprintf("%s/schema/$dbname", $ENV{'MBGD_HOME'});
#    unlink($fileNew);
#    symlink($fileOld, $fileNew);

    #
    convertProtmotif("$fileRpsBlastRes", "$fileRpsBlastRes.2");

    #
    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                     '1020LoadProtMotif.pl');

    #
    my($opt) = "-TYPE=ProtMotif -dbname=$dbname";

    #
writeLogMsg($uid, "LOG :: Load rpsblast results($dbname)");
    system("$cmd $opt $fileRpsBlastRes.2");
    unlink("$fileRpsBlastRes.2");
}

###############################################################################
#
sub makeUserGenomeTaball {
    my($uid) = shift;

    #
    my($dbname) = getDbname($uid);
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($fileTaball) = "$dirUgDb/taball";
    my($cmd) = "$main::CMD_cp $ENV{'MBGD_HOME'}/database/taball.domclust $fileTaball";
    system("$cmd");

    #
    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                      '1140ConvTaballFromDb.pl');

    #
    my($www) = new MBGD::WWW();
    my($uInfo) = $www->uInfo();
    $uInfo->setUID($uid);
    my($hom) = $uInfo->loadHomolParams();

    # ϿƤ UserGenome
    my($db) = MBGD::DB->new($dbname);
    my($sql) = "select sp from genome_user";
    my($sth) = $db->execute($sql);
    my(@sp_list);
    while(my$ref = $sth->fetchrow_hashref()) {
        my($ug) = $ref->{'sp'};
        push(@sp_list, $ug);
    }
    my($splist) = join(',', @sp_list);

    my($opt) = "-usergenome=$uid -dbname=$dbname -splist=$splist";

    #
#print STDERR "sss>>>>$splist\n";
writeLogMsg($uid, "LOG :: Make taball($fileTaball) for $uid :: $splist");
    system("$cmd $opt >> $fileTaball");
}

###############################################################################
# for BLAST-DB
sub createBldbSymLnk {
    my($uid) = shift;
    my($dh);
    my($file);
    my($n) = 0;

    #
    my($dirMbgd) = "$ENV{'MBGD_HOME'}/database/bldb";
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUser) = "$dirUgDb/bldb";

    #
    rmtree($dirUser, 0) if (-e $dirUser);
    mkpath($dirUser, 0, 0750);

    #
    $dh = new DirHandle($dirMbgd) || die("Can not open $dirMbgd($!)");
    while($file = $dh->read()) {
        next if ($file =~ /^\./);

        my($fileOld) = sprintf("%s/%s", $dirMbgd, $file);
        my($fileNew) = sprintf("%s/%s", $dirUser, $file);
        symlink($fileOld, $fileNew) if (! -e "$fileNew");
    }
}

###############################################################################
#
sub updateBlastDb {
    my($uid) = shift;
    my($db) = shift;
    my($sp) = shift;
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUser) = "$dirUgDb/bldb";

    my($tab) = "gene g, proteinseq ps";
    my($opt) = {};
    $opt->{'columns'} = "g.name name, ps.seq seq";
    $opt->{'where'}   = "g.aaseq=ps.id and g.sp='$sp'";

    #
    my($res) = $db->select_fetch($tab, $opt);

    my($filename) = "$dirUser/$sp";
    my($fh) = new FileHandle(">$filename") || die("Can not open $filename($!)");

    foreach my$ent (@{$res->{'INFO'}}) {
        my($name) = $ent->{'name'};
        my($seq)  = $ent->{'seq'};

        $seq =~ s#(.{1,60})#$1\n#g;

        $fh->print(">$sp:$name\n");
        $fh->print($seq, "\n");
    }
    $fh->close();

    # BLAST  DB κ
    execFormatdb($filename, "-l /dev/null");
}

###############################################################################
#
sub updateBlastDbAll {
    my($uid) = shift;
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUser) = "$dirUgDb/bldb";

    #
    my(@spList) = getUserGenomeIdListStatusOk($uid);

    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new($dbname);
    foreach my$sp (@spList) {
        updateBlastDb($uid, $db, $sp);
    }
}

###############################################################################
# for BLAST+dp results
sub createBldpSymLnk {
    my($uid) = shift;
    my($dh);
    my($file);
    my($n) = 0;

    #
    my($dirMbgd) = "$ENV{'MBGD_HOME'}/database/bldp";
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUser) = "$dirUgDb/bldp";

    #
    rmtree($dirUser, 0) if (-e $dirUser);
    mkpath($dirUser, 0, 0750);

    #
    $dh = new DirHandle($dirMbgd) || die("Can not open $dirMbgd($!)");
    while($file = $dh->read()) {
        next if ($file !~ /^blastdpres\./);

        my($fileOld) = sprintf("%s/%s", $dirMbgd, $file);
        my($fileNew) = sprintf("%s/%s", $dirUser, $file);
        symlink($fileOld, $fileNew) if (! -e "$fileNew");
    }
}

###############################################################################
#
sub updateBlastdpRes {
    my($uid) = shift;
    my($dbname) = getDbname($uid);
    my($fileOld);
    my($fileNew);

    #
    my($ext_sp) = 'user';
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($dirUgDbDp) = "$dirUgDb/bldp";
    my($file_bldp) = "$dirUgDbDp/blastdpres.$ext_sp";
    mkpath("$dirUgDbDp", 0, 0750);

writeLogMsg($uid, "LOG :: Update bldp for $uid");

    #
    my($cols) = "spname1, spname2, from1, to1, from2, to2, ident, eval, score, pam";
    my($tab) = "homology";
    my($order) = "sp1, sp2, name1, name2";
    my($sql) = "select $cols from $tab order by $order";
    my($cmd) = "$main::CMD_mysql -q -D $dbname -e '$sql' --skip-column-names";
    my($fhr) = IO::File->new("$cmd |") || die("Can not open $cmd($!)");
    my($fhw) = IO::File->new(">$file_bldp") || die("Can not open $file_bldp($!)");
    while (my$line=$fhr->getline()) {
        $line =~ s#[\r\n]*$##;

        my(@dat) = split(/\s+/, $line);
        my($pd) = pack($main::PACK_TEMPL38, @dat);
        $fhw->print($pd);
    }
    $fhw->close();
    $fhr->close();

    #
    $ENV{'DIR_SPINDEX'} = "$dirUgDbDp";
    my($cmd_spindex) = "$ENV{'MBGD_HOME'}/binaries/spindex";
    my($cmd) = "$cmd_spindex $ext_sp";
    system("$cmd");
    my($cmd) = "$cmd_spindex -m $ext_sp";
    system("$cmd");

    return;
}

###############################################################################
#
sub miscUpdate {
    my($uid) = shift;

    #
    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                      '1175MiscUpdate.pl');

    #
    my($dbname) = getDbname($uid);
    my($opt) = "-usergenome=$uid -dbname_accum=$dbname";

    #
    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($fileUdate) = "$dirUgDb/udate";

    #
writeLogMsg($uid, "LOG :: misc update for $uid");
    system("$cmd $opt > $fileUdate");
}

###############################################################################
# $main::DBNAME_TMP ˺ߤΡ֥饹ơ֥פ򥳥ԡ
#     cluster_tables ơ֥ tabid='default' 쥳
#
# $main::DBNAME_TMP ˺ߤΡall/default ơ֥פ򻲾ȲǽȤ
#     cluster_default ơ֥
#     clusttab_default ơ֥
#     clusttree_default ơ֥
#
sub copyDefaultCluster {
    my($uid) = shift;
    my($sql);
    my(@tabid_list) = ('default', 'all', 'addspec');

    #
    my($dbname) = getDbname($uid);
    my($db) = new MBGD::DB("dbi:mysql:$dbname");

    my($sql_where_tabs) = '';
    foreach my$tabID (@tabid_list) {
        if ($sql_where_tabs ne '') {
            $sql_where_tabs .= ',';
        }
        $sql_where_tabs .= "'$tabID'";
    }

    ############################################################################
    # cluster_tablesʣ
    $sql = "create table if not exists cluster_tables "
         . "(spec text, cmd text,"
         . "tabid char(12) not null,"
         . "cdate timestamp,"
         . "primary key(tabid))";
    $db->do($sql);

    #
    $sql = "delete from cluster_tables where tabid in ($sql_where_tabs)";
    $db->do($sql);
    $sql = "insert cluster_tables "
         . "select * from $main::DBNAME_TMP.cluster_tables "
         . "where tabid in ($sql_where_tabs)";
    $db->do($sql);

    #
    foreach my$tabID (@tabid_list) {
        # cluster_$tabID
        $sql = "drop table if exists cluster_$tabID";
        $db->do($sql);
        $sql = "create table cluster_$tabID ("
             . "clustid int,"
             . "sp char(5) not null,"
             . "name char($main::SIZE_NAME) not null,"
             . "spname char(30) not null,"
             . "dom smallint not null,"
             . "from1 int,"
             . "to1 int,"
             . "primary key (spname, dom),"
             . "unique key (sp, name, dom)"
             . ")"
             . " ENGINE=MRG_MyISAM UNION=($main::DBNAME_TMP.cluster_$tabID)";
        $db->do($sql);

        # clusttree_$tabID
        $sql = "drop table if exists clusttree_$tabID";
        $db->do($sql);
        $sql = "create table clusttree_$tabID ("
             . "clustid int not null,"
             . "tree mediumtext,"
             . "primary key (clustid)"
             . ")"
             . " ENGINE=MRG_MyISAM UNION=($main::DBNAME_TMP.clusttree_$tabID)";
        $db->do($sql);

        # clusttab_$tabID
        $sql = "drop table if exists clusttab_$tabID";
        $db->do($sql);
        $sql = "create table clusttab_$tabID ("
             . "clustid int not null,"
             . "homclustid int default NULL,"
             . "phylopat text not null,"
             . "spnum int,"
             . "orfnum int,"
             . "gene varchar(24),"
             . "descr text,"
             . "mbgd varchar(24),"
             . "cog  varchar(24),"
             . "kegg varchar(24),"
             . "tigr varchar(24),"
             . "data mediumtext,"
             . "primary key (clustid)"
             . ")"
             . " ENGINE=MRG_MyISAM UNION=($main::DBNAME_TMP.clusttab_$tabID)";
        $db->do($sql);

        next if ($tabID =~ /^all$/i);

        # clustxref_$tabID
        $sql = "drop table if exists clustxref_$tabID";
        $db->do($sql);
        $sql = "create table clustxref_$tabID ("
             . "clustid    int(11) NOT NULL,"
             . "xref_db    varchar(24) NOT NULL,"
             . "xref_id    varchar(24) NOT NULL,"
             . "n_xref     int(11) default '0',"
             . "total_xref int(11) default '0',"
             . "primary key (clustid, xref_db, xref_id)"
             . ")"
             . " ENGINE=MRG_MyISAM UNION=($main::DBNAME_TMP.clustxref_$tabID)";
        $db->do($sql);

    }

    return;
}

###############################################################################
#
sub makeDefaultCluster {
    my($uid) = shift;
    my($sql);
    my(@tabList) = ('cluster', 'clusttab', 'clusttree');
    my($tab);

    #
    my($dbname) = getDbname($uid);
    my($db) = new MBGD::DB("dbi:mysql:$dbname");

    # Ťǥեȥ饹ơ֥κ
    foreach $tab ('cluster_tables', @tabList) {
        $sql = "drop table if exists ${tab}_all";
        $db->do($sql);
    }

    #
    my($cmd) = sprintf("%s/build/%s", $ENV{'MBGD_HOME'},
                                      '1180MakeDefaultCluster.pl');

    #
    my($www) = new MBGD::WWW();
    my($uInfo) = $www->uInfo();
    $uInfo->setUID($uid);
    my($hom) = $uInfo->loadHomolParams();

    # 򤵤Ƥʪоݤ˥ǥեȥ饹
    my($type) = 'user_genome';
    my($splist) = $hom->getParams()->{'species'};
    $splist =~ s#\|#,#g;

###    $type = 'all';      # ʪȤ˥ǥեȥ饹

    # ǥեȥ饹̾ user_genome Ȥƺ
    my($opt) = "-usergenome=$uid -dbname=$dbname -mbgd_home=$ENV{'MBGD_HOME'} -type=$type -splist='$splist'";
    system("$cmd $opt");
#print STDERR "CMD>>> $cmd $opt\n";

    if ($type =~ /^user_genome/i) {
        # ǥեȥ饹ơ֥̾ѹ
        foreach $tab (@tabList) {
            $sql = "rename table ${tab}_user_genome to ${tab}_all;";
            $db->do($sql);
        }

        # cluster_tables ˤϡtabid='user_genome' ϿƤ
        # ϿƤ 'user_genome'  'all' ˹
        $sql = "update cluster_tables set tabid='all';";
        $db->do($sql);
    }

    $db->disconnect();
}

###############################################################################
#
sub updateGeneUserFunccat {
    my($uid) = shift;
    my($tabid) = shift;
    my($sql);

    if (($tabid !~ /^all$/i) &&
        ($tabid !~ /^default$/i) && 
        ($tabid !~ /^\d+$/)) {
        return;
    }

    #
    my($dbname) = getDbname($uid);
    my $db = MBGD::DB->new();
    $db->connect("dbi:mysql:$dbname");

    my($tmptab) = "tmp_clust";

#    $sql = "create temporary table $tmptab "
#         . "select c2.sp,c2.name,c1.class "
#         . "from cluster_$tabid c2 "
#         . "left join clusttab_$tabid c1 on c1.clustid=c2.clustid and c2.dom=1";
#    $db->execute($sql);

    # gene_user ơ֥ funccat 򹹿
#    $sql = "update gene_user g, $tmptab t set g.funccat=t.class "
#         . "where g.sp=t.sp and g.name=t.name";
#    $db->execute("$sql");

}

###############################################################################
#
sub updateGeneUserAalen {
    my($uid) = shift;
    my($sql);

    #
    my($dbname) = getDbname($uid);
    my $db = MBGD::DB->new();
    $db->connect("dbi:mysql:$dbname");

    # gene_user ơ֥ aalen 򹹿
    $sql = "update gene_user g, proteinseq " .
        " set g.aalen=proteinseq.length " .
        " where g.aaseq=proteinseq.id";
    $db->execute($sql);
}

###############################################################################
#
sub updateChromosomeUserLen {
    my($uid) = shift;
    my($sql);

    #
    my($dbname) = getDbname($uid);
    my $db = MBGD::DB->new();
    $db->connect("dbi:mysql:$dbname");

    # chromosome_user ơ֥ seq_length 򹹿
    $sql = "update chromosome_user c, dnaseq d " .
        " set c.seq_length=d.length " .
        " where c.seq=d.id";
    $db->execute($sql);
}

###############################################################################
#
sub clearWorkFiles {
    my($uid) = shift;

    #
    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($dh) = new DirHandle("$dirBaseUg") || return;
    my($file);
    while($file = $dh->read()) {
        if (($file =~ /^domclust\./) ||
            ($file =~ /^homSearch\./) ||
            ($file =~ /^rpsBlastRes/)) {
            unlink("$dirBaseUg/$file");
        }
    }

    # tmp
    rmtree("$dirBaseUg/tmp", 0);
    mkpath("$dirBaseUg/tmp", 0, 0750);
}

###############################################################################
#
sub setUserGenomeDbStatus {
    my($uid) = shift;
    my($sta) = shift;
    my($mod) = shift || 'Ok';

    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($fileUgDbReady) = "$dirUgDb/status$mod";

    if ($sta) {
        if (! -e $fileUgDbReady) {
            my($fh) = new FileHandle(">$fileUgDbReady");
            $fh->print($$);
            $fh->close();
        }
        else {
            # ɵ open & closeʥॹ׹
            my($fh) = new FileHandle(">>$fileUgDbReady");
            $fh->close();
        }
    }
    else {
        unlink($fileUgDbReady) if (-e $fileUgDbReady);
    }
}

###############################################################################
# ǽηв
sub getLastUpdate {
    my($uid) = shift;
    my($lastUpdate) = 365 * 100;             # ǯΥե
    my($gid);

    for($gid = 1; $gid <= $MbgdUserGenomeCommon::MAX_USER_GENOME; $gid++) {
        my($refUserGenome) = loadUserGenomeInfo($uid, $gid);
        if (! $refUserGenome) {
            next;
        }
        if ($refUserGenome->{'last_update'} < $lastUpdate) {
            $lastUpdate = $refUserGenome->{'last_update'};
        }
    }

    return $lastUpdate;
}

###############################################################################
#
sub setBuildingStatus {
    my($uid) = shift;
    my($sta) = shift;

    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($filename) = "$dirUgDb/nowBuilding";
    my($fh) = new FileHandle(">$filename");
die "cannot open $filename\n" if (! $fh);
    $fh->print($sta, "\n");
    $fh->close();
}

###############################################################################
#
sub getBuildingStatus {
    my($uid) = shift;
    my($fh);

    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($filename) = "$dirUgDb/nowBuilding";

    my($try);
    for($try = 0;; $try++) {
        $fh = new FileHandle("$filename");
        if ($fh) {
            last;
        }

        if (2 < $try) {
            return 'unknown';
        }
        sleep(1);
    }
    my($sta) = $fh->getline();
    $sta =~ s#[\r\n]*$##;
    $fh->close();

    if ($sta eq $MbgdUserGenomeCommon::STA_blastSearch) {
        # BLAST  DP ϡĤΥޥɤǼ¹Ԥ뤿
        # DP μ¹Ԥ֥ե¸ߤ̵ͭȽǤ
        my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
        my($fileMdpin) = "$dirBaseUg/tmp/list." . $main::pid;
        if (-e $fileMdpin) {
            $sta = $MbgdUserGenomeCommon::STA_dpAlignment;
        }
    }

    return $sta;
}

###############################################################################
#
sub setupUserGenomeEnv {
    my($uid) = shift;

    my($modeMymbgd) = isMymbgdMode($uid);
    if (! $modeMymbgd) {
        return;
    }

    my($isReadyUserGenome) = isReadyUserGenome($uid);
    if (! $isReadyUserGenome) {
        return;
    }

    my($dirUserGenome) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid); 
    my($dirUserDb)   = $main::DIR_userDb = "$dirUserGenome/database";
    my($fileTaball)  = "$dirUserDb/taball";
    my($fileTax)     = "$dirUserDb/tax";
    my($fileUdate)   = "$dirUserDb/udate";
    my($fileSpectab) = "$dirUserDb/spec.tab";

    # 桼ΥǡۺѤ
    $main::ENV{'MYSQL_DB'} = $main::ENV{'MYSQL_DBTMP'} = getDbname($uid);
    $main::ENV{'MYSQL_USER_DATABASE'} = "$dirUserGenome/database";
    $main::ENV{'MBGD_TMP'} = "$dirUserGenome/tmp";
    mkpath($main::ENV{'MBGD_TMP'}, 0, 0750) if (! -e $main::ENV{'MBGD_TMP'});
    $main::FILE_taball = $fileTaball if (-e $fileTaball);
    $main::FILE_tax = $fileTax if (-e $fileTax);
    $main::FILE_udate = $fileUdate if (-e $fileUdate);
    $main::FILE_spectab = $fileSpectab if (-e $fileSpectab);

    # uid  schema ϢŤ
    my($fileOld) = "$ENV{'MBGD_HOME'}/schema/mbgd";
    my($fileNew) = "$ENV{'MBGD_HOME'}/schema/$main::ENV{'MYSQL_DB'}";
#    symlink($fileOld, $fileNew) if (! -e $fileNew);

    return;
}

###############################################################################
#
sub isReadyUserGenome {
    my($uid) = shift;
    my($mod) = shift || 'Ok';

    my($dirUgDb) = getDirUserGenomeDatabase($uid);
    my($fileUgDbReady) = "$dirUgDb/status$mod";
    if (-e $fileUgDbReady) {
#        if ($mod =~ /^ok$/i) {
#            my(@spList) = getBlastTargetSpec($uid);
#            if (scalar(@spList) != 0) {
#                # BLAST+dp ̤¹ԤǤʪ濫
#                return $main::FALSE;
#            }
#        }
        return $main::TRUE;
    }

    return $main::FALSE;
}

###############################################################################
#
sub _buildUserGenomeMysql {
    my($uid) = shift;

print STDERR "==>buileUserGenomeMysql\n";
    createUserGenomeDatabase($uid);
    createUserGenomeTable($uid);

    # BLAST+dp Ѥʪ򥯥ꥢ
    clearBlastFinishedSpecAll($uid);
print STDERR "==>buileUserGenomeMysql:OK\n";
}

###############################################################################
#  UserGenome ñ̤ BLAST+dp ¹Ԥ
sub execBlastDpByUg {
    my($uid) = shift;
    my($sp) = shift;
    my($fileQry) = shift;

    # BLAST+dp оݤȤʤʪ
    my(@spList) = getBlastTargetSpec($uid, $sp);
    if (scalar(@spList) == 0) {
        return;
    }
    my($fileSps) = getFileBlastDb($uid, @spList);

    # BLAST+mdp
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_blastSearch);
print STDERR "execBlastMdp: $sp: @spList\n";
    my($fileBlastDpRes) = execBlastMdp($uid, $sp, $fileSps, $fileQry);

    # BLAST Ѥʪ¸
    addBlastFinishedSpec($uid, $sp, @spList);

    return $fileBlastDpRes;
}

###############################################################################
#  UserGenome ñ̤ RPS-BLAST ¹Ԥ
sub execRpsBlastByUg {
    my($uid) = shift;
    my($sp) = shift;
    my($fileQry) = shift;

    if (0) {
        # RPS-BLAST ѤǤ
        return;
    }

    # RPS-BLAST
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_motifSearch);
    my($fileRpsBlastRes) = execRpsBlast($uid, $fileQry);
print STDERR "execRpsBlast\n";

    # motif Ѥʪ¸
    addMotifFinishedSpec($uid, $sp);

    return $fileRpsBlastRes;
}

###############################################################################
#
sub _buildUserGenomeData {
    my($uid) = shift;
    my($sta);

    #
    my($dirUgDb) = getDirBaseUserGenome($uid);
    my($dirUgDbSp) = "$dirUgDb/species";

    ###########################################################################
    # ǡ۳
print STDERR "==>buileUserGenomeData\n";
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_buildingGenomeDb);

    my($doHomologySearch) = 0;
    my($doMotifSearch) = 0;
    my(@spUgList) = convUserGenome($uid);
    foreach my$sp (@spUgList) {
        # UserGenome  ProteinSeq
        my($fileQry) = "$dirUgDb/species/$sp/gm/$sp.protseq";

        # BLAST+mdp
        my(@spList) = getBlastTargetSpec($uid, $sp);
        if (scalar(@spList) == 0) {
            # оʪ̵
            next;
        }

print STDERR ">>execBlastDpByUg: $sp\n";
        setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_blastSearch);
        my($fileBlastDpRes) = execBlastDpByUg($uid, $sp, $fileQry);
        if (-e "$fileBlastDpRes" && -s "$fileBlastDpRes") {
            # BLAST+dp ̤줿
            clearHomologyRes($uid, $sp, @spList);
            loadHomSearchRes($uid, $fileBlastDpRes);
            $doHomologySearch = 1;
        }

        # RPS-BLAST
        setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_motifSearch);
        my($fileRpsBlastRes) = execRpsBlastByUg($uid, $sp, $fileQry);
        if (-e "$fileRpsBlastRes" && -s "$fileRpsBlastRes") {
            # RPS-BLAST ̤줿
            loadRpsBlastRes($uid, $fileRpsBlastRes);
            $doMotifSearch = 1;
        }
    }
    if (! $doHomologySearch && ! $doMotifSearch) {
        #  BLAST+dp/RPS-BLAST ̤ʤ ===> ǡ۽λ
        setUserGenomeDbStatus($uid, $main::TRUE);
        return;
    }

    #
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_buildingHomDb);
    loadUserGenome($uid);
    updateGeneUserAalen($uid);
    updateChromosomeUserLen($uid);

print STDERR "makeUserGenomeTaball\n";
    makeUserGenomeTaball($uid);
#    createSptitSymlink($uid);
#    createSptitForUserGenome($uid);
#    createBldpSymLnk($uid);
    updateBlastdpRes($uid);
    createBldbSymLnk($uid);
    updateBlastDbAll($uid);
#    miscUpdate($uid);

    #
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_postProcessing);
	if (0) {
		makeDefaultCluster($uid);
	}
	else {
		copyDefaultCluster($uid);
	}

    #
    my($mymbgdId) = $uid;
    if (MbgdUserGenomeCommon::isMymbgdMode($uid)) {
        $mymbgdId = MbgdUserGenomeCommon::uid2mymbgdid($uid);
    }
    my($refUserInfo) = MbgdUserGenomeCommon::loadUserInfo($mymbgdId);
    $refUserInfo->{'update_gene_funccat'} = 'yes';
    MbgdUserGenomeCommon::saveUserInfo($mymbgdId, $refUserInfo);

    #
    ###########################################################################
    # ǡ۽λ
    setUserGenomeDbStatus($uid, $main::TRUE);
print STDERR "Done\n";
}

###############################################################################
#
sub buildUserGenomeData {
    my($uid) = shift;
    my(%args) = @_;
    my($pid) = -1;

    my($modBuild) = 'Build';
    if (isReadyUserGenome($uid, $modBuild)) {
        my($dirUgDb) = getDirUserGenomeDatabase($uid);
        my($fileUgDbReady) = "$dirUgDb/status$modBuild";
        my($fh) = new FileHandle("$fileUgDbReady");
        if ($fh) {
            $pid = $fh->getline();
        }

        my($sta) = kill(0 => $pid);
        if ($sta) {
            # ¹ Build ץ ID ֤
            return $pid;
        }

        # ơBuild פʤΤ˳ץ¸ߤʤ
        unlink($fileUgDbReady);
    }

    #
    setBuildingStatus($uid, $MbgdUserGenomeCommon::STA_buildingGenomeDb);
    $pid = fork();
    if ($pid) {
        # ƥץǤϡǡ۽λޤ now building ɽ

    }
    elsif (defined($pid)) {
        # ҥץǥǡۤԤ
        close(STDOUT);
        close(STDERR);

	my $dir = getDirBaseUserGenome($uid);
	open(STDOUT, ">>$dir/mbgd_hom.o$$");
	open(STDERR, ">>$dir/mbgd_hom.e$$");
	print STDERR "==================\n";

         #
         clearLogMessage($uid);

         # Ķѿ
         MbgdUserGenomeCommon::setupUserGenomeEnv($uid);

        # ơ֥ǡ
        setUserGenomeDbStatus($uid, $main::TRUE, $modBuild);

        # ե󹹿λĤˤ
        clearWorkFiles($uid);

        if (! isReadyUserGenome($uid)) {
            # MySQL database/tables ι
print STDERR "===>112\n";
            _buildUserGenomeMysql($uid);
        }
print STDERR "===>113\n";

        # ǡۼ¹
        _buildUserGenomeData($uid, $pid);

        # ǡ۽λ
        setUserGenomeDbStatus($uid, $main::FALSE, $modBuild);
        setUserGenomeDbStatus($uid, $main::TRUE, 'Fin');#۽λĤ
        print STDERR "DBG :: [[FINISHED]] build UserGenome\n";
        exit(0);
    }
    else {
        # fork 
    }

    return $pid;
}

###############################################################################
#
sub checkMyMbgdId {
    my($uid) = shift;

    my($dirUid) = getDirBaseUserGenome($uid, {'no_convert' => 1});
    if (-e $dirUid) {
        return $main::FALSE;
    }

    return $main::TRUE;
}

###############################################################################
#
sub saveUserInfo {
    my($mymbgdId) = shift;
    my($formOpt) = shift;
    my($key);

    my($dirUid) = getDirBaseUserGenome($mymbgdId, {'no_convert' => 1});
    mkpath($dirUid, 0, 0750) if (! -e $dirUid);
    my($fh) = new FileHandle(">$dirUid/contact") || return;
    foreach $key (@MbgdUserGenomeCommon::LST_KEY_USERINFO) {
        next if ($key =~ /^pass/i);

        $fh->print($key, "\t", $formOpt->{"$key"}, "\n");
    }
    $fh->close();
}

###############################################################################
#
sub loadUserInfo {
    my($mymbgdId) = shift;
    my($formOpt) = {};
    my($line);

    #
    my($dirUid) = getDirBaseUserGenome($mymbgdId, {'no_convert' => 1});
    my($fh) = new FileHandle("$dirUid/contact") || return $formOpt;
    while($line = $fh->getline()) {
        $line =~ s#[\r\n]*$##;
        my($key, $val) = split("\t", $line);

        next if ($key =~ /^\s*$/);

        $formOpt->{"$key"} = $val;
    }
    $fh->close();

    $formOpt->{'email'} = $mymbgdId;

    return $formOpt;
}

###############################################################################
#
sub setupUserDir {
    my($uid) = shift;                 # Cookie  ID
    my($formOpt) = shift;
    my($email) = $formOpt->{'email'};

    if ($uid =~ /^\s*$/) {
        print STDERR "WARNING :: UID is blank!!\n";
        return;
    }
    if ($email =~ /^\s*$/) {
        print STDERR "WARNING :: ID(email) is blank!!\n";
        return;
    }

    #
    my($dirUid)   = "$ENV{'MBGD_HOME'}/MBGDUserData/$uid";
    my($dirEmail) = "$ENV{'MBGD_HOME'}/MBGDUserData/$email";

    if (! -e $dirEmail) {            # email Ȥˤǥ쥯ȥʤ
        rename($dirUid, $dirEmail);# Cookie ǥ쥯ȥѤ
        mkpath($dirEmail, 0, 0750) if (! -e $dirEmail);

        #  uid Ȥ MySQL ³
#        MbgdUserGenomeCommon::saveDbname($uid, {'force'=>1, 'new_uid'=>1});
    }
}

###############################################################################
#
sub saveDbname {
    my($uid) = shift;
    my($opt) = shift;

return;

    #
    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($filename) = "$dirBaseUg/dbname";
    if (-e $filename) {
        return if (! $opt->{'force'});       # 񤭤ʤ
    }
    if ($opt->{'new_uid'}) {
        $uid = time() . '_' . $$;
    }

    #
    mkpath($dirBaseUg, 0, 0750) if (! -e $dirBaseUg);
    my($fh) = new FileHandle(">$filename");
    $fh->print('uid', "\t", $uid, "\n");
    $fh->close();

    return;
}

###############################################################################
#
sub getDbname {
    my($uid) = shift;
    my($dbname) = 'mbgd_';
    my($line);

    #
    my($dirBaseUg) = MbgdUserGenomeCommon::getDirBaseUserGenome($uid);
    my($filename) = "$dirBaseUg/dbname";
    my($fh) = new FileHandle("$filename");
    if (! $fh) {
print STDERR "User>$uid\nfilename>$filename\n";
#    confess "Can't open dbname file: $filename\n" if (! $fh);
        # ե뤬¸ߤʤä
        setUserGenomeDbStatus($uid, $main::FALSE);
		return $dbname;
    }

    while($line = $fh->getline()) {
        if ($line =~ /^uid\t(\S+)$/) {
            touchLastAccess($uid);
            my($dbnameSaved) = $dbname . $1;
#            my($db) = MBGD::DB->new($dbnameSaved);
#            if (!$db->sta_connect()) {
#                $fh->close();
#                unlink("$filename");
#                return $dbname;
#            }
            $dbname = $dbnameSaved;

            last;
        }
    }
    $fh->close();

    return $dbname;
}

###############################################################################
#
sub escapeHtmlChar {
    my($str) = shift;

    $str =~ s#\<#&lt;#g;
    $str =~ s#\>#&gt;#g;
    $str =~ s#\&#&amp;#g;
    $str =~ s#\"#&quot;#g;
    $str =~ s#\'#&rsquo;#g;

    return $str;
}

###############################################################################
#
sub printHtmlUserInfo {
    my($formOpt) = shift;
    my($mod) = shift || 'edit';

    #
    my($title, $btnName);
    my($cgi);
    if ($mod =~ /^edit/i) {
        $title = 'Edit';
        $btnName = "Update";
        $cgi = "/htbin/MyMBGD/editUserInfo.cgi";
    }
    else {
        $title = 'Create';
        $btnName = "Create";
        $cgi = "/htbin/MyMBGD/createId.cgi";
    }

    print<<EOB;
<script language="JavaScript" src="/js/mbgd.js"></script>
<script language="JavaScript" src="/MyMBGD/mymbgd.js"></script>
<script>
function doSubmit() {
    var doc = this.document;
    var elm;
    var val;

    elm = doc.fui.email;
    trim(elm);
    val = elm.value;
    if (val == '') {
        alert('Please Input Your E-mail address.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
    if (val.length < 4) {
        alert('Please Input Longer E-mail address.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
    if ($MbgdUserGenomeCommon::MAX_email < val.length) {
        alert('Please Input Shorter E-mail address.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
    if (! validateMailAddress(val, true)) {
        alert('Please input your valid E-mail address.');
        elm.focus();
        return;
    }

    elm = doc.fui.name;
    val = elm.value;
    if (val.match(/^\\s*\$/i)) {
        alert('Please Input Your name.');
        elm.focus();
        return;
    }

    elm = doc.fui.pass1;
    val = elm.value;
EOB

if ($mod =~ /^edit/i) {
    print<<EOB;
    if (val != '') {
EOB
}
else {
    print<<EOB;
    if (val == '') {
        alert('Please Input Your Password.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
EOB
}

    print<<EOB;
    if (val.length < 4) {
        alert('Please Input Longer Password.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
    if ($MbgdUserGenomeCommon::MAX_email < val.length) {
        alert('Please Input Shorter Password.(4-$MbgdUserGenomeCommon::MAX_email characters)');
        elm.focus();
        return;
    }
EOB
if ($mod =~ /^edit/i) {
    print<<EOB;
    }
EOB
}

    print<<EOB;
    if (val.match(/[^a-z0-9]/i)) {
        alert('Password Contains Wrong Character.');
        elm.focus();
        return;
    }

    if (val != doc.fui.pass2.value) {
        alert('Password is not same.');
        doc.fui.pass2.value = '';
        doc.fui.pass2.focus();
        return;
    }

    elm = doc.fui.organization;
    val = elm.value;
    if (val.match(/^\\s*\$/i)) {
        alert('Please Input Your organization.');
        elm.focus();
        return;
    }

    doc.fui.submit();
}
</script>
EOB

    if ($formOpt->{'msg'} !~ /^\s*$/) {
        print "<h3>", $formOpt->{'msg'}, "</h3>\n";
    }

    print<<EOB;
<form name="fui" method="post" action="$cgi">
<table bgcolor="#007000" width="100%">
<tr><td><font color="#e8f8e8">$title MyMBGD Account</font></td>
</tr>
<tr><td>
<table bgcolor="#e8f8e8" width="100%" border>
<tr><th>E-mail(Account)</th>
EOB

    if ($mod =~ /^edit/i) {
    print<<EOB;
    <td>$formOpt->{'email'}
        <input type="hidden" name="email" value="$formOpt->{'email'}"></td>
EOB
    }
    else {
    print<<EOB;
    <td><input type="text" name="email" value="$formOpt->{'email'}" size="40">
        Your E-mail address(yourname\@your.domain)</td>
EOB
    }

    my($name) = escapeHtmlChar($formOpt->{'name'});
    my($org)  = escapeHtmlChar($formOpt->{'organization'});
    print<<EOB;
</tr>

<tr><th>Name</th>
    <td><input type="text" name="name" value="$name" size="40"></td>
</tr>

<tr><th>Password</th>
    <td><input type="password" name="pass1" value="" size="20">
        Password may consist of A-Z, a-z and 0-9.</td>
</tr>

<tr><th>Password</th>
    <td><input type="password" name="pass2" value="" size="20">
        [Re-type]</td>
</tr>

<tr><th>Organization</th>
    <td><input type="text" name="organization" value="$org" size="60"></td>
</tr>

</table>
</td></tr>
</table>
<input type="button" name="" value="$btnName" onclick="doSubmit();">
</form>
EOB
}

###############################################################################
#
sub printHtmlError {
    my($opt) = shift;

    print<<EOB;
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Frameset//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html>
<head>
<title></title>
<link rel="stylesheet" href="/MyMBGD/mymbgd.css" type="text/css" />
<script language="JavaScript" src="/MyMBGD/mymbgd.js"></script>
</head>

<body>
<h3>$opt->{'msg'}</h3>

</body>
</html>

EOB

}

###############################################################################
#
sub printHtmlBuildStatus {
    my($uid) = shift;
    my($sta);

    my($staBuild) = getBuildingStatus($uid);

    print "<table border>\n";
    print "<tr><th align=center>\n";
    print "Building Status";
    print "</th></tr>\n";
    print "<tr><td>\n";
    print "<table>\n";
    foreach $sta (@MbgdUserGenomeCommon::STAlist) {
        print "<tr><td align=center>\n";
        print "<b>\n" if ($staBuild eq $sta);
        print $sta, "\n";
        print "</b>\n" if ($staBuild eq $sta);
        print "</td></tr>\n";
    }
    print "</table>\n";
    print "</td></tr>\n";
    print "</table>\n";
}

###############################################################################
#
sub getFileBlastDb {
    my($uid) = shift;
    my(@spList) = @_;

    #
#    my($dbname) = $ENV{'MYSQL_DB'};
    my($dbname) = getDbname($uid);
    my($db) = new MBGD::DB($dbname);

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($filename) = "$dirUg/tmp/protseqSps.$$";
    my($dirUgDb) = getDirBaseUserGenome($uid);

    my($fh) = new FileHandle(">$filename");
    foreach my $sp (@spList) {
        if ($sp =~ /^ug\d+$/i) {
            my($fileQry) = "$dirUgDb/species/$sp/gm/$sp.protseq";
            my($fhr) = FileHandle->new("$fileQry");
            if ($fhr) {
                while (my$line=$fhr->getline()) {
                    $fh->print($line);
                }
                $fhr->close();
            }
            next;
        }

        my(@listGeneInfo) = MBGD::Gene->get($db, [$sp]);
        foreach my $geneInfo (@listGeneInfo) {
            my($aaseq) = $geneInfo->{'aaseq'};
            next if (! $aaseq);

            my($spec) = $geneInfo->{'sp'};
            my($name) = $geneInfo->{'name'};
            my($refSeq) = MBGD::ProteinSeq->fetch($geneInfo->{'aaseq'});
            my($seq) = $refSeq->{'seq'};
            $seq =~ s#(.{1,60})#$1\n#g;

            $fh->print(">$spec:$name\n");
            $fh->print($seq);
        }
    }
    $fh->close();

    return $filename;
}

###############################################################################
#
sub execBlastMdp {
    my($uid) = shift;
    my($sp) = shift;
    my($fileSps) = shift;
    my($fileQry) = shift;
    my($cmd);

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($dirTmp) = "$dirUg/blast.$sp.$$";
    File::Path::rmtree($dirTmp) if (-e $dirTmp);
    File::Path::mkpath($dirTmp, 1, 0750);
    my($fileDb)       = "$dirTmp/blastdb";
    my($fileBlastOut) = "$dirTmp/blastout";
    my($fileList)     = "$dirTmp/list";
    my($fileMdpOut)   = "$dirTmp/mdpout";
    my($fileResult)   = "$dirTmp/blastdp.result";
    my($fileRest)     = "$dirTmp/rest";

    # fileQry ޤե
    $cmd = "$main::CMD_cat $fileSps $fileQry > $fileDb";
    system("$cmd");

    #
    $cmd = "$main::CMD_mqblast -TMPDIR=$dirTmp -retainTmp -OUTFILE='$fileBlastOut' -mode=qsub -CHECK -CHECK_COUNT $fileDb $fileQry";
writeLogMsg($uid, "LOG :: Execute mqblast for $uid");
    system("$cmd");

    sleep(1);

    #
    $cmd = "$main::CMD_createList $fileBlastOut > $fileList";
    system("$cmd");

    #
    unlink("$fileResult") if (-e "$fileResult");

    my($fileDbIn) = $fileDb;
    for(my $retry = 0;; $retry++) {
        if (10 < $retry) {
            # ȥ饤򷫤֤Ƥη̤뤳ȤǤʤä
            print <<EOH;
<html>
<head>
<title>Sorry, Internal Error.</title>
<link rel="stylesheet" type="text/css" href="/css/mbgd_top.css" />
</head>
<body>
Sorry, System got internal error.<br>
Please contact server-admin.<br>
<A HREF="mailto:$main::adminMailAddress">$main::adminMailAddress</a>
</body>
</html>
EOH

            die("ERROR :: 'dp2' retry over error.");
        }

        my($fileListFlat) = $fileList . "_flat";
        my($fhList) = new FileHandle("$fileList");
        my($fhFlat) = new FileHandle(">$fileListFlat");
        while(my $line = $fhList->getline()) {
            my($orf1, $orf2) = ($line =~ /^(\S+)\s+(\S+)/);
            $fhFlat->print("$orf1\n");
            $fhFlat->print("$orf2\n");
        }
        $fhList->close();
        $fhFlat->close();

        my($fileDb4mdp) = sprintf("$fileDb.%d", $retry);
        main::compactionFasta($fileDbIn, $fileDb4mdp, $fileListFlat);

        #
        my($dirTmp) = "$dirUg/dp2.$sp.$$";
        $cmd = "$main::CMD_execdp2 -EXECDIR=$dirTmp -retainTmp $fileList $fileDb4mdp $fileMdpOut";
writeLogMsg($uid, "LOG :: Execute dp2 for $uid");
        system("$cmd");
        if (! -e "$fileMdpOut" || -z "$fileMdpOut") {
            writeLogMsg($uid, "LOG :: WARNING :: execdp2 got no results for $uid. [CMD:$cmd]");
        }

        #
        unlink("$fileResult.new") if (-e "$fileResult.new");
        $cmd = "$main::CMD_updateList -listfile=$fileList -outfile=$fileMdpOut -resultout=$fileResult.new > $fileRest";
        system("$cmd");
        if (! -e "$fileResult.new" || -z "$fileResult.new") {
            writeLogMsg($uid, "LOG :: WARNING :: updateList got no results for $uid. [CMD:$cmd]");
        }

        # ¹Է̤ΤޤȤ
        $cmd = "$main::CMD_cat $fileResult.new >> $fileResult";
        system("$cmd");

        if (-z "$fileRest") {
writeLogMsg($uid, "LOG :: Finished execmdp for $uid");
            last;
        }

        # Ƽ¹Խ
writeLogMsg($uid, "LOG :: Retry execmdp for $uid");
        unlink("$fileList");
        rename("$fileRest", "$fileList");
        $fileDbIn = $fileDb4mdp;
    }

    return $fileResult;
}

###############################################################################
#
sub execRpsBlast {
    my($uid) = shift;
    my($fileQry) = shift;
    my($cmd);

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($dirTmp) = "$dirUg/rpsblast.$$";
    my($fileBlastOut) = "$dirTmp/rpsblastout";
    my($fileResult)   = "$dirTmp/rpsblast.result";

    #
    $cmd = "$main::CMD_mqblast -TMPDIR=$dirTmp -retainTmp -OUTFILE=$fileBlastOut -mode=qsub -CopyAll -CopyDB=0 -BLKSIZ=200 -BLASTOPT='-e 1' -CHECK -PROGRAM='rpsblast' $main::FILE_cdd $fileQry";
writeLogMsg($uid, "LOG :: Execute rpsblast for $uid");
writeLogMsg($uid, "LOG :: CMD :: $cmd");
    system("$cmd");

    #
    my($file_cddid) = "$ENV{'MBGD_HOME'}/package/cdd/cddid.tbl.gz";
    $cmd = "$main::CMD_convrps -FILE_CDDID='$file_cddid' -output=work $fileBlastOut > $fileResult";
    system("$cmd");

    return $fileResult;
}

###############################################################################
#
sub clueanupWorkFiles {
    my($uid) = shift;
    my($dir, $file);
    my($dh);

    #
    $dir = $ENV{'HOME'};
    $dh = new DirHandle("$dir");
    if ($dh) {
        foreach $file ($dh->read()) {
            if (($file =~ /^STDIN\.[oe]\d+$/) ||
                ($file =~ /^jobid\.\d+$/)) {
writeLogMsg($uid, "LOG :: unlink $dir/$file for $uid");
                unlink("$dir/$file");
            }
        }
    }

    #
    $dir = getDirBaseUserGenome($uid);
    $dh = new DirHandle("$dir");
    if ($dh) {
        foreach $file ($dh->read()) {
            if (($file =~ /^domclust\.[oe]\d+$/) ||
                ($file =~ /^mbgd_hom\.[oe]\d+$/)) {
writeLogMsg($uid, "LOG :: unlink $dir/$file for $uid");
                unlink("$dir/$file");
            }
        }
    }

}

###############################################################################
#
sub initializeUserGenomeDatabase {
    my($uid) = shift;
    my($dir, $file);
    my($dh);

    #
    $dir = getDirBaseUserGenome($uid);
    $dh = new DirHandle("$dir");
    if ($dh) {
        foreach $file ($dh->read()) {
            if (($file =~ /\.clstinf$/) ||
                ($file =~ /\.param$/) ||
                ($file =~ /\.pref$/) ||
                ($file =~ /\.tabid$/) ||
                ($file =~ /\.param$/) ||
                ($file =~ /\.param$/) ||
                ($file =~ /^dbname$/) ||
                ($file =~ /^contact$/) ||
                (isUserGenome($file))) {
                next;
            }
            if ($main::DEBUG) {
                print STDERR "DBG :: rmtree($dir/$file)";
            }
            else {
writeLogMsg($uid, "LOG :: rmtree $dir/$file for $uid");
                rmtree("$dir/$file");
            }
        }
    }

    #
    foreach $file ('database', 'species', 'tmp') {
        mkpath("$dir/$file", 0, 0750);
    }
}

###############################################################################
#
sub convertHomology {
    my($fileIn) = shift;
    my($fileOut) = shift;
    my($line);

    #
    my($fhr) = new FileHandle("$fileIn");
    my($fhw) = new FileHandle(">$fileOut");
    while($line = $fhr->getline()) {
        $line =~ s#[\r\n]*$##;

        my($sporf1, $from1, $to1, $sporf2, $from2, $to2, @dat) = split(/\t/, $line);
        my($sp1, $name1) = split(/:/, $sporf1);
        my($sp2, $name2) = split(/:/, $sporf2);

        $fhw->print(join("\t", $sp1, $name1, $sporf1, $from1, $to1,
                               $sp2, $name2, $sporf2, $from2, $to2,
                               @dat), "\n");
        $fhw->print(join("\t", $sp2, $name2, $sporf2, $from2, $to2,
                               $sp1, $name1, $sporf1, $from1, $to1,
                               @dat), "\n") if ($sporf1 ne $sporf2);
    }
    $fhw->close();
    $fhr->close();

    return;
}

###############################################################################
#
sub convertProtmotif {
    my($fileIn) = shift;
    my($fileOut) = shift;
    my($line);

    #
    my($fhr) = new FileHandle("$fileIn");
    my($fhw) = new FileHandle(">$fileOut");
    while($line = $fhr->getline()) {
        $line =~ s#[\r\n]*$##;

        my($sp1, $name1, $from1, $to1, @dat) = split(/\t/, $line);

        $fhw->print(join("\t", $sp1, $name1, $from1, $to1, @dat), "\n");
    }
    $fhw->close();
    $fhr->close();

    return;
}

###############################################################################
#
sub clearHomologySearchRes {
    my($db) = shift;
    my($sp1) = shift;
    my($sp2) = shift;

    if ((! MbgdUserGenomeCommon::isUserGenome($sp1)) &&
        (! MbgdUserGenomeCommon::isUserGenome($sp2))) {
        # UserGenome ꤵƤʤ
        return;
    }

    my($tab) = 'homology_user';
    my($sql) = "delete from $tab where sp1='$sp1'";
    if (! $sp2) {
        $sql .= " or sp2='$sp1'";
    }
    else {
        $sql .= " or sp2='$sp1'";
    }
print STDERR "SQL :: $sql\n" if ($main::DEBUG);
    $db->execute("$sql");
}

###############################################################################
#
sub clearMotifSearchRes {
    my($db) = shift;
    my($sp) = shift;

    if (! MbgdUserGenomeCommon::isUserGenome($sp)) {
        return;
    }

    my($tab) = 'protmotif_user';
    my($sql) = "delete from $tab where sp='$sp'";
print STDERR "SQL :: $sql\n" if ($main::DEBUG);
    $db->execute("$sql");
}

###############################################################################
#
sub clearClusteringRes {
    my($db) = shift;
    my($tabId) = shift;
    my($tab);
    my($sql);

    #
    foreach my$t ('cluster_', 'clusttab_', 'clusttree_', 'clustcat_', 'cluster_score_', 'clustxref_', ) {
        $tab = "$t$tabId";
        $sql = "drop table if exists $tab";
print STDERR "SQL :: $sql\n" if ($main::DEBUG);
        $db->execute("$sql");
    }

    $tab = 'cluster_tables';
    $sql = "delete from $tab where tabid='$tabId'";
print STDERR "SQL :: $sql\n" if ($main::DEBUG);
    $db->execute("$sql");
}

###############################################################################
#
sub clearAllClusteringRes {
    my($db) = shift;
    my($sp) = shift;
    my($tab);
    my($sql);

    if (! MbgdUserGenomeCommon::isUserGenome($sp)) {
        return;
    }

    # tabid θ
    my(@tabIdList);
    $tab = 'cluster_tables';
    my$opt = {};
    $opt->{'field'} = "*";
    $opt->{'where'} = "spec like '%$sp%'";
    my($res) = $db->select_fetch($tab, $opt);
    foreach my$r (@{$res->{INFO}}) {
        push(@tabIdList, $r->{'tabid'});
    }

    #
    foreach my$tabId (@tabIdList) {
        clearAllClusteringRes($db, $tabId);
    }
}

###############################################################################
#
sub clearUserGenome {
    my($uid) = shift;
    my($sp) = shift;

    if (! MbgdUserGenomeCommon::isUserGenome($sp)) {
        return;
    }

    my($dbname) = MbgdUserGenomeCommon::getDbname($uid);
    my($db) = new MBGD::DB($dbname);

    # ۥ̤κ
    clearHomologySearchRes($db, $sp);

    # ո̤κ
    clearMotifSearchRes($db, $sp);

    # 饹󥰷̤κ
    clearClusteringRes($db, $sp);

    return;
}

###############################################################################
#
sub isUserGenome {
    my($sp) = shift;

    if ($sp =~ /^ug[0-9]+$/i) {
        return $main::TRUE;
    }
    return $main::FALSE;
}

###############################################################################
#
sub getGenomeSpid {
    my($id) = shift;

    return sprintf("gm%05d", $id);
}

###############################################################################
#
sub getUserGenomeSpid {
    my($id) = shift;

    return sprintf("gu%05d", $id);
}

###############################################################################
#
sub getUserGenomeSpec {
    my($id) = shift;

#    return sprintf("ug%d", $id);
#    my(%ginfo) = getGenomeInfo($id);
    my($ginfo) = loadUserGenomeInfo(0, $id);
    return $ginfo->{'sp'};
}

###############################################################################
#
sub getUserGenomeId {
    my($spec) = shift;

    my($id) = ($spec =~ /ug(\d+)/);

    return $id;
}

###############################################################################
#
sub touchLastAccess {
    my($uid) = shift;

    my($dirUg) = getDirBaseUserGenome($uid);
    my($fileLastAccess) = "$dirUg/lastAccess";
    new FileHandle(">$fileLastAccess");
}

###############################################################################
#
sub printLoginStatusCookie {
    my($uid) = shift;
    my($sta) = shift;
    my($vali) = shift;
    my($vald) = shift;
    my($vale) = shift;
    my($date19700101) = 'Thu, 01-Jan-1970 00:00:00 GMT';
    my(@cookies);

    if ($sta) {
        my($t) = time() + 3600 * 24 * 7;
        my($wday, $mon, $mday, $hms, $year) = split(/\s+/, scalar(gmtime($t)));
        my($expires) = "$wday, $mday-$mon-$year $hms GMT";
        my($path) = "/";

        if (! $vali) {
            $vali = $uid;
        }
        if ($vali eq $uid) {
            $vald = '';
        }
        push(@cookies, {'name'=>'mymbgd_i', 'value'=>"$vali", 'expires'=>"$expires", 'path'=>"$path"});
        push(@cookies, {'name'=>'mymbgd_d', 'value'=>"$vald", 'expires'=>"$expires", 'path'=>"$path"});
    }
    else {
        push(@cookies, {'name'=>'mymbgd_i', 'value'=>'', 'expires'=>"$date19700101", 'path'=>'/'});
        push(@cookies, {'name'=>'mymbgd_d', 'value'=>'', 'expires'=>"$date19700101", 'path'=>'/'});
    }

    if ($vale) {
        push(@cookies, {'name'=>'mymbgd_e', 'value'=>"$vale", 'path'=>'/'});
    }
    else {
        push(@cookies, {'name'=>'mymbgd_e', 'value'=>'', 'expires'=>"$date19700101", 'path'=>'/'});
    }

    foreach my$c (@cookies) {
        my($name)    = $c->{'name'};
        my($value)   = $c->{'value'};
        $value =~ s/(\W)/sprintf("%%%02X", ord($1))/ego;
        my($path)    = $c->{'path'};
        my($domain)  = $c->{'domain'};
        my($expires) = $c->{'expires'};
        my($secure)  = $c->{'secure'};
        my($port)    = $ENV{'SERVER_PORT'};
        my($cookie) = '';
#        $cookie .= "version=1; ";
        $cookie .= "$name=$value; ";
        $cookie .= "path=$path; " if ($path);
        $cookie .= "domain=$domain; " if ($domain);
        $cookie .= "secure=$secure; " if ($secure);
        $cookie .= "port=$port; " if ($port);
        $cookie .= "expires=$expires; " if ($expires);

        print "Set-Cookie: $cookie\n";
#        print STDERR "Set-Cookie: $cookie\n";
    }
}

###############################################################################
#
sub isMymbgdMode {
    my($uid) = shift;
    my($cgi) = shift;
    my($mymbgdId);

    #
    if (! $cgi) {
        $cgi = new CGI();
    }
    $mymbgdId = $cgi->cookie('mymbgd_i');
    if (! $mymbgdId) {
        return $main::FALSE;
    }

    return $main::TRUE;
}

###############################################################################
#
sub pass2hash {
    my($wwwUser) = shift;
    my($wwwPass) = shift;

    my($cmd) = "$main::CMD_htpasswd -nbs $wwwUser $wwwPass";
    my($name, $hashpass) = split(':', `$cmd`);
    $hashpass =~ s#[\r\n]*$##;

    return $hashpass;
}

###############################################################################
#
sub loadUserHashpass {
    my($uid) = shift;
    my($wwwUser) = shift;

    #
    my($dirUg) = getDirBaseUserGenome($wwwUser, {'no_convert' => 1});
    my($filePass) = "$dirUg/pass2hash";
    my($fh) = new FileHandle("$filePass");
    if (! $fh) {
        print STDERR "ERROR :: Can not open $filePass($!)\n";
        return '';
    }
    my($hashpass) = $fh->getline();
    $fh->close();
    $hashpass =~ s#[\r\n]*$##;

    return $hashpass;
}

###############################################################################
#
sub saveUserPass {
    my($uid) = shift;
    my($wwwUser) = shift;
    my($wwwPass) = shift;

    #
    my($hashpass) = pass2hash($wwwUser, $wwwPass);

    my($dirUg) = getDirBaseUserGenome($wwwUser, {'no_convert' => 1});
    mkpath($dirUg, 0, 0750) if (! -e "$dirUg");
    my($filePass) = "$dirUg/pass2hash";
    my($fh) = new FileHandle(">$filePass") || return;
    $fh->print($hashpass, "\n");
    $fh->close();

    return $hashpass;
}

###############################################################################
# Ͽ桼ǧ
sub isValidUserPass {
    my($uid) = shift;
    my($wwwUser) = shift;
    my($wwwPass) = shift;

    # ¸Ƥ PASS(hash)
    my($hashpass0) = loadUserHashpass($uid, $wwwUser);

    #
    my($hashpass1) = pass2hash($wwwUser, $wwwPass);
    if ($hashpass0 ne $hashpass1) {
        # ֥饦Ϥ줿 password hashͤȥե¸Ƥͤפʤ
        return $main::FALSE;
    }

    return $main::TRUE;
}

###############################################################################
# Ͽ桼ǧ
sub isValidUserHash {
    my($uid) = shift;
    my($wwwUser) = shift;
    my($hashpass2) = shift;

    # ¸Ƥ hashpass
    my($hashpass0) = loadUserHashpass($uid, $wwwUser);
    if ($hashpass0 ne $hashpass2) {
        # ֥饦Ϥ줿 hashpass2 ȥե¸Ƥ(hashpass0)פʤ
        return $main::FALSE;
    }

    return $main::TRUE;
}

###############################################################################
#
sub uid2mymbgdid {
    my($uid) = shift;

    my($cgi) = new CGI();
    my($mymbgd_i) = $cgi->cookie('mymbgd_i');
    if (! $mymbgd_i) {
        return $uid;
    }
    return $mymbgd_i;
}

###############################################################################
#
sub validateMailAddressAtom {
    my($adrs) = shift;
    my($msg) = '';

    if ($adrs =~ /^\s*$/) {
        $msg = "Found blank part";
        return $msg;
    }

    # invalid characters
    #     0x00-0x1f
    #     space
    #     ()<>@,;:\".[]
    if ($adrs =~ /[\x00-\x1f \(\)\<\>\@\,\;\:\\\"\.\[\]]/) {
        $msg = "Found invalid character(s)";
        return $msg;
    }

    # о e-mail Ʊ̾Τǥǥ쥯ȥ뤳Ȥ⤢
    # ɲäԲĤȤ롣
    if ($adrs =~ /[\'\$\&\!\`\*\?\~\/\|]/) {
        $msg = "Found invalid character(s)";
        return $msg;
    }

    return '';
}

###############################################################################
#
sub validateMailAddress {
    my($adrs) = shift;
    my($msg) = '';

    my(@parts) = split('@', $adrs);
    if (scalar(@parts) != 2) {
        $msg = "Please input a '@' character";
        return $msg;
    }

    for(my$idx = 0; $idx < scalar(@parts); $idx++) {
        my($part) = $parts[$idx];
        foreach my$atom (split(/\./, $part)) {
            $msg = validateMailAddressAtom($atom);
            if ($msg) {
                if ($idx == 0) {
#                    $msg = "Invalid name(E-mail)";
                }
                else {
#                    $msg = "Invalid domain(E-mail)";
                }
                return $msg;
            }
        }
    }

    if ($adrs =~ /^\-/) {
        $msg = "can not start with '-'";
        return $msg;
    }

    return '';
}

###############################################################################
#
sub makeExportUgDataFile {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($file_export) = shift;

    #
    my($dbname) = getDbname($uid);
    my($db) = MBGD::DB->new("dbi:mysql:$dbname");
    if (!$db->sta_connect()) {
        return;
    }

    #
    my($dirUg) = getDirBaseUserGenome($uid);
    my($dir_work) = "$dirUg/work_export_ug_$$";
    mkpath("$dir_work/gm/data", 0, 0750);

    #
    my($sql);
    my($sth);

    # genome
    my($sp) = getUserGenomeSpec($idUserGenome);
    $sql = "select * from genome_user where sp='$sp'";
    $sth = $db->execute($sql);
    if ($sth->rows() == 0) {
        rmtree("$dir_work");
        return;
    }
    my$ref = $sth->fetchrow_hashref();
    my($file_genome) = "$dir_work/gm/genome.txt";
    my($fh_genome) = FileHandle->new(">$file_genome");
    foreach my$k ('sp',
                  'orgname', 'abbrev', 'strain',
                  'taxid',
                  'date_release',
                  'specweight') {
        my($v) = $ref->{"$k"};
        $fh_genome->print($k, "\t", $v, "\n");
    }
    $fh_genome->close();

    #
    $sql = "select * from chromosome_user where sp='$sp' order by seqno";
    $sth = $db->execute($sql);
    while (my$ref = $sth->fetchrow_hashref()) {
        my($chrid) = $ref->{'id'};
        my($seqno) = $ref->{'seqno'};
        my($seqid) = $ref->{'seq'};

        # chromosome
        my($file_chromosome) = "$dir_work/gm/data/$seqno.chromosome.txt";
        my($fh_chromosome) = FileHandle->new(">$file_chromosome");
        foreach my$k ('name',
                      'seqno',
                      'shape',
                      'type',
                      'accession',
                      'length') {
            my($v) = $ref->{"$k"};
            $fh_chromosome->print($k, "\t", $v, "\n");
        }
        $fh_chromosome->close();

        # gene
        $sql = "select * from gene g "
             . "where g.sp='$sp' and g.chrid=$chrid";
        $sth = $db->execute($sql);
        my($file_gene) = "$dir_work/gm/data/$seqno.gene";
        my($fh_gene) = FileHandle->new(">$file_gene");
        while (my$g_ref = $sth->fetchrow_hashref()) {
            my(@g) = ();
            foreach my$k ('name', 'gene',
                          'from1', 'to1', 'dir',
                          'type',
                          'descr') {
                push(@g, $g_ref->{"$k"});
            }
            $fh_gene->print(join("\t", @g), "\n");
        }
        $fh_gene->close();

        # dnaseq
        $sql = "select * from dnaseq where id=$seqid";
        $sth = $db->execute($sql);
        my($file_chrseq) = "$dir_work/gm/data/$seqno.chrseq";
        my($fh_chrseq) = FileHandle->new(">$file_chrseq");
        my($cs_ref) = $sth->fetchrow_hashref();
        $fh_chrseq->print($cs_ref->{'seq'});
        $fh_chrseq->close();

        # geneseq
        $sql = "select g.name, gs.seq from gene g, geneseq gs "
             . "where g.sp='$sp' and g.chrid=$chrid and g.ntseq=gs.id";
        $sth = $db->execute($sql);
        my($file_geneseq) = "$dir_work/gm/data/$seqno.geneseq";
        my($fh_geneseq) = FileHandle->new(">$file_geneseq");
        while (my$gs_ref = $sth->fetchrow_hashref()) {
            $fh_geneseq->print(">", $gs_ref->{'name'}, "\n");
            $fh_geneseq->print($gs_ref->{'seq'}, "\n");
        }
        $fh_geneseq->close();

        # protseq
        $sql = "select * from proteinseq ps, gene g "
             . "where g.sp='$sp' and g.chrid=$chrid and g.aaseq=ps.id";
        $sth = $db->execute($sql);
        my($file_protseq) = "$dir_work/gm/data/$seqno.protseq";
        my($fh_protseq) = FileHandle->new(">$file_protseq");
        while (my$ps_ref = $sth->fetchrow_hashref()) {
            $fh_protseq->print(">", $ps_ref->{'name'}, "\n");
            $fh_protseq->print($ps_ref->{'seq'}, "\n");
        }
        $fh_protseq->close();

    }

    #
    my($cmd) = "$main::CMD_tar cfz $file_export -C $dir_work ./gm";
    system("$cmd");

    rmtree("$dir_work");

    return;
}

###############################################################################
#
sub createGeneTabForProtein {
    my($fileProtein) = shift;
    my($fileGeneTab) = shift;

    #
    my(%name_hash);
    my($fhp) = FileHandle->new("$fileProtein");
    while (my$line=$fhp->getline()) {
        if ($line =~ /^>\s*(\S+)/) {
            $name_hash{"$1"} = $1;
        }
    }
    $fhp->close();

    #
    my(@name_list) = keys(%name_hash);
    my($fhg) = FileHandle->new(">$fileGeneTab");
    foreach my$name (sort(@name_list)) {
        $fhg->print(join("\t", $name, '', -1, -1, 1, ''), "\n");
    }
    $fhg->close();

    return;
}

###############################################################################
#
sub save_mode_chromosome {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;
    my($modUserChr) = shift;

    my($dir) = MbgdUserGenomeCommon::getDirUserChromosome($uid, $idUserGenome, $idUserChromosome);
    my($filename) = "$dir/mode";
    my($fh) = FileHandle->new(">$filename") || return;
    $fh->print($modUserChr, "\n");
    $fh->close();

    return;
}

###############################################################################
#
sub load_mode_chromosome {
    my($uid) = shift;
    my($idUserGenome) = shift;
    my($idUserChromosome) = shift;

    my($dir) = MbgdUserGenomeCommon::getDirUserChromosome($uid, $idUserGenome, $idUserChromosome);
    my($filename) = "$dir/mode";
    my($fh) = FileHandle->new("$filename") || return;
    my($modUserChr) = $fh->getline();
    $fh->close();

    return $modUserChr;
}

###############################################################################
#
sub is_metagenome {
    my($uid) = shift;
    my($sp) = shift;

    #
    my($idUserGenome) = ($sp =~ /^ug(\d+)$/i);
    if (!$idUserGenome) {
        return;
    }

    for(my$idUserChromosome = 1; $idUserChromosome <= $MbgdUserGenomeCommon::MAX_USER_CHROMOSOME; $idUserChromosome++) {
        my($sta) = load_mode_chromosome($uid, $idUserGenome, $idUserChromosome);
        if ($sta =~ /$MbgdUserGenomeCommon::MODE_USERCHR_metagenome\:/) {
            return 1;
        }
    }

    return;
}

###############################################################################
#
sub conv_ug2gm {
    my($uid) = shift;
    my($gid) = shift;

    my($dirUgSpid) = MbgdUserGenomeCommon::getDirUserGenomeSpid($uid, $gid);

    my($file_ug) = "$dirUgSpid/gu/last_update";
    my($tm_ug) = -M "$file_ug";

    my($file_gm) = "$dirUgSpid/gm/genome.txt";
    my($tm_gm) = -M "$file_gm";

    if ((-e "$file_ug") && (-e "$file_gm") && (-M "$file_gm" < -M "$file_ug")) {
        # Found newer 'genome.txt'
        return;
    }


    my($cmd) = "$ENV{'RECOG_HOME'}/WWW/bin/conv_ug2gm.pl -UID='$uid' $gid";
    system("$cmd");

    return;
}

###############################################################################
1;#
###############################################################################
