#!/usr/bin/perl
use strict;
use GD;

package PhylopatImage;
$PhylopatImage::MIN_WIDTH = 2;
$PhylopatImage::BAR_WIDTH = 600;
$PhylopatImage::BAR_HEIGHT = 15;

###############################################################################
#
sub new {
    my($class) = shift;
    my($self) = {};

    bless($self, $class);

    $self->_init();

    return $self;
}

###############################################################################
#
sub _init {
    my($self) = shift;

    return;
}

###############################################################################
#
sub set_size {
    my($self) = shift;
    my($xlen) = shift;
    my($ylen) = shift;

    $xlen = $PhylopatImage::BAR_WIDTH if (! $xlen);
    $ylen = $PhylopatImage::BAR_HEIGHT if (! $ylen);

    $self->set_xlen($xlen);
    $self->set_ylen($ylen);

    return;
}

###############################################################################
#
sub set_xlen {
    my($self) = shift;
    my($xlen) = shift;

    $self->{'XLEN'} = $xlen;

    return;
}

###############################################################################
#
sub get_xlen {
    my($self) = shift;

    return $self->{'XLEN'};
}

###############################################################################
#
sub update_xlen {
    my($self) = shift;

    my($cell_width) = $self->get_cell();
    my($border_width) = $self->get_border();
    my($patlen) = $self->get_pattern_length();
    my($xlen) = ($cell_width + $border_width) * $patlen;

    $self->set_xlen($xlen);

    return;
}

###############################################################################
#
sub set_ylen {
    my($self) = shift;
    my($ylen) = shift;

    $self->{'YLEN'} = $ylen;

    return;
}

###############################################################################
#
sub get_ylen {
    my($self) = shift;

    return $self->{'YLEN'};
}

###############################################################################
#
sub set_scale {
    my($self) = shift;
    my($scale) = shift;

    $scale = 1 if (! $scale);
    $self->{'SCALE'} = $scale;

    return;
}

###############################################################################
#
sub get_scale {
    my($self) = shift;

    return $self->{'SCALE'};
}

###############################################################################
#
sub set_spcolor {
    my($self) = shift;
    my($spcolor) = shift;

    if ($spcolor =~ /^off$/i) {
        $spcolor = '';
    }
    $self->{'SPCOLOR'} = $spcolor;

    return;
}

###############################################################################
#
sub get_spcolor {
    my($self) = shift;

    return $self->{'SPCOLOR'};
}

###############################################################################
#
sub set_border {
    my($self) = shift;
    my($border_width) = shift;

    $border_width = 1 if (! $border_width);
    $self->{'BORDER_WIDTH'} = $border_width;

    $self->update_xlen();

    return;
}

###############################################################################
#
sub get_border {
    my($self) = shift;

    return $self->{'BORDER_WIDTH'};
}

###############################################################################
#
sub set_cell {
    my($self) = shift;
    my($cell_width) = shift;

    $cell_width = $PhylopatImage::MIN_WIDTH if ($cell_width < $PhylopatImage::MIN_WIDTH);
    $self->{'CELL_WIDTH'} = $cell_width;

    $self->update_xlen();

    return;
}

###############################################################################
#
sub get_cell {
    my($self) = shift;

    return $self->{'CELL_WIDTH'};
}

###############################################################################
#
sub set_patterm {
    my($self) = shift;
    my($pattern) = shift;

    $self->{'PATTERN'} = $pattern;

    $self->update_xlen();

    return;
}

###############################################################################
#
sub get_pattern {
    my($self) = shift;
    my($ofs) = shift;

    my($pat) = $self->{'PATTERN'};
    if ($ofs) {
        $pat = substr($pat, $ofs, 1);
    }

    return $pat;
}

###############################################################################
#
sub get_pattern_length {
    my($self) = shift;

    return length($self->{'PATTERN'});
}

###############################################################################
###############################################################################
package PhylopatImageGif;
use PhylopatImage;
our(@ISA) = qw( PhylopatImage );

###############################################################################
#
sub _init {
    my($self) = shift;
    my(@args) = @_;

    $self->SUPER::_init(@args);

    $self->{'PAT2OFS'} = {};
    my($ofs) = 0;
    foreach my$pat (0 .. 9, 'A' .. 'Z') {
        $self->{'PAT2OFS'}->{"$pat"} = $ofs;
        $ofs++;
    }

    return;
}

###############################################################################
#
sub pat2ofs {
    my($self) = shift;
    my($pat) = shift;

    #
    $pat = uc($pat);
    my($ofs) = $self->{'PAT2OFS'}->{"$pat"};

    return $ofs;
}

###############################################################################
#
sub set_color {
    my($self) = shift;
    my($color) = shift;

    $color = "#e8f8e8,#009900" if (! $color);

    $self->{'COLOR'} = [];
    foreach my$c (split(/,/, $color)) {
        if ($c !~ /^#/) {
            $c = "#$c";
        }
        push(@{$self->{'COLOR'}}, $c);
    }

    return;
}

###############################################################################
#
sub get_color_length {
    my($self) = shift;

    return scalar(@{$self->{'COLOR'}});
}

###############################################################################
#
sub get_color {
    my($self) = shift;
    my($ofs) = shift;

    #
    my($ncolor) = $self->get_color_length();
    $ofs %= $ncolor;

    my($col) = $self->{'COLOR'}->[$ofs];

    return $col;
}

###############################################################################
#
sub get_color_by_pattern {
    my($self) = shift;
    my($pat) = shift;

    #
    my($ofs)= $self->pat2ofs($pat);
    my($col) = $self->get_color($ofs);

    return $col;
}

###############################################################################
#
sub conv_color {
    my($self) = shift;
    my($colorstr) = @_;
    my($R,$G,$B);

    $colorstr =~ s/^#//;
    $R = hex(substr($colorstr,0,2));
    $G = hex(substr($colorstr,2,2));
    $B = hex(substr($colorstr,4,2));
    return ($R,$G,$B);
}

###############################################################################
#
sub init_gd {
    my($self) = shift;

    my($xlen) = $self->get_xlen();
    my($ylen) = $self->get_ylen();
    $self->{'GD'} = GD::Image->new($xlen, $ylen);
    $self->{'GD_COLOR'} = {};

    return;
}

###############################################################################
#
sub get_gd_color {
    my($self) = shift;
    my($col) = shift;

    my($color);
    if (!exists($self->{'GD_COLOR'}->{"$col"})) {
        my(@rgb) = $self->conv_color($col);
        $color = $self->{'GD'}->colorAllocate(@rgb);
        $self->{'GD_COLOR'}->{"$col"} = $color;
    }
    $color = $self->{'GD_COLOR'}->{"$col"};

    return $color;
}

###############################################################################
#
sub filledRectangle {
    my($self) = shift;
    my(@args) = @_;

    $self->{'GD'}->filledRectangle(@args);

    return;
}

###############################################################################
#
sub gif {
    my($self) = shift;
    my(@args) = @_;

    return $self->{'GD'}->gif();
}

###############################################################################
#
sub print_phylopat {
    my($self) = shift;
    my(@alt_list) = @_;

    my($www) = MBGD::WWW->new();
    my($uInfo) = $www->uInfo;
    my($uinfo_param_ref) = $uInfo->loadParams();

    #
    $self->init_gd();

    #
    my($xlen)       = $self->get_xlen();
    my($ylen)       = $self->get_ylen();
    my($scale)      = $self->get_scale();
    my($spcolor)    = $self->get_spcolor();
    my($border)     = $self->get_border();
    my($phylopat)   = $self->get_pattern();
    my($patlen)     = $self->get_pattern_length();
    my($cell_width) = $self->get_cell();
##print STDERR ">>>$patlen, $cell_width, $xlen\n";

    my($color_delim) = "#888888";

    #
    my($idx) = 0;
    my($x0) = 0;
    foreach my$pat (split(//, uc($phylopat))) {
        #
        my($col) = $self->get_color_by_pattern($pat);
        my($color) = $self->get_gd_color($col);

        my($sp) = $alt_list[$idx];
        my($key) = "spec_color_$sp";
        if ($pat && $spcolor && ($uinfo_param_ref->{"$key"} !~ /^\s*$/)) {
            $col = $uinfo_param_ref->{"$key"};
            $color = $self->get_gd_color($col);
        }

        my($x1) = $x0 + $cell_width * $scale;
        my(@pos) = (int($x0), 0, int($x1), $ylen);
        $self->filledRectangle(@pos, $color);
        $x0 = $x1;

        #
        $col = $color_delim;
        $color = $self->get_gd_color($col);
        $x1 = $x0 + $border * $scale;
        @pos = (int($x0), 0, int($x1), $ylen);
        $self->filledRectangle(@pos, $color);
        $x0 = $x1;

        $idx++;
    }

    print $self->gif();

    return;
}

###############################################################################
###############################################################################
package PhylopatImageMap;
use PhylopatImage;
our(@ISA) = qw( PhylopatImage );

###############################################################################
#
sub print_phylopat {
    my($self) = shift;
    my($name) = shift;
    my($id) = shift;
    my(@alt_list) = @_;

    #
#    $self->init_gd();
    my($map) = '';

    #
    my($xlen)       = $self->get_xlen();
    my($ylen)       = $self->get_ylen();
    my($scale)      = $self->get_scale();
    my($border)     = $self->get_border();
    my($phylopat)   = $self->get_pattern();
    my($patlen)     = $self->get_pattern_length();
    my($cell_width) = $self->get_cell();
##print STDERR ">>>$patlen, $cell_width, $xlen\n";

#    my($color_delim) = "#888888";

    #
    $map .= "<map";
    $map .= " name=\"$name\"" if ($name);
    $map .= " id=\"$id\"" if ($id);
    $map .= ">\n";
    my($x0) = 0;
    foreach my$pat (split(//, uc($phylopat))) {
        #
#        my($col) = $self->get_color_by_pattern($pat);
#        my($color) = $self->get_gd_color($col);

        my($x1) = $x0 + $cell_width * $scale;
        my(@pos) = (int($x0), 0, int($x1), $ylen);
#        $self->filledRectangle(@pos, $color);
        my($alt) = shift(@alt_list);
        $map .= '<area';
        $map .= ' title="' . $alt . '"';
        $map .= ' alt="' . $alt . '"';
        $map .= ' shape="rect"';
        $map .= ' coords="' . join(',', @pos) . '"';
        $map .= ' />' . "\n";
        $x0 = $x1;

        #
#        $col = $color_delim;
#        $color = $self->get_gd_color($col);
        $x1 = $x0 + $border * $scale;
        @pos = (int($x0), 0, int($x1), $ylen);
#        $self->filledRectangle(@pos, $color);
        $x0 = $x1;
    }
    $map .= "</map>\n";

#    print $self->gif();

    return $map;
}

###############################################################################
1;
###############################################################################
