#!/usr/bin/perl -s
use FileHandle;
use IO::File;
use MBGD;
use MBGD::DB;
use RECOG::RecogCommon;

package CoreAlignerUtil;

###############################################################################
$main::CMD_corealign = "$ENV{'MBGD_HOME'}/WWW/bin/RECOG/corealign.pl";
if (! -e $main::CMD_corealign) {
    die("Can not found $main::CMD_corealign");
}
$main::CMD_updateCoreAlignerProgress = "$ENV{'MBGD_HOME'}/WWW/bin/RECOG/updateCoreAlignerProgress.pl";

###############################################################################
                                                # DB $B%"%/%;%9%(%i!<(B
$CoreAlignerUtil::ERRNO_DB_ACCESS                = "F0001";
                                                # $B%U%!%$%k%"%/%;%9%(%i!<(B
$CoreAlignerUtil::ERRNO_FILE_ACCESS              = "F0002";
                                                # fork() $B%(%i!<(B
$CoreAlignerUtil::ERRNO_FORK                     = "F0003";
                                                # $BBP1~$9$k%/%i%9%?!<$,L5$$(B
$CoreAlignerUtil::ERRNO_NO_CLUST_TAB             = "F0101";
                                                # $B2r@O%*%W%7%g%s0[>o(B
$CoreAlignerUtil::ERRNO_NG_OPTIONS               = "F0102";
                                                # $B2r@O7k2L0[>o(B
$CoreAlignerUtil::ERRNO_NG_ANALYZE               = "F0103";

%CoreAlignerUtil::ERRMSG = ( $CoreAlignerUtil::ERRNO_DB_ACCESS
                            => "DB access error.",
                          $CoreAlignerUtil::ERRNO_FILE_ACCESS,
                            => "File access error.",
                          $CoreAlignerUtil::ERRNO_FORK,
                            => "Fork error.",
                          $CoreAlignerUtil::ERRNO_NO_CLUST_TAB,
                            => "No cluster table.",
                          $CoreAlignerUtil::ERRNO_NG_OPTIONS,
                            => "Ng options.",
                          $CoreAlignerUtil::ERRNO_SERVER_INTERNAL,
                            => "System Internal Error.",
                        );

#
$CoreAlignerUtil::TYPE_KEY_STR = 1; # $BJ8;zNs%*%W%7%g%s(B
$CoreAlignerUtil::TYPE_KEY_NUM = 2; # $B?tCM%*%W%7%g%s(B

###############################################################################
#
$CoreAlignerUtil::STA_error    = -99;
$CoreAlignerUtil::STA_finished =  -1;
$CoreAlignerUtil::STA_waiting  =   0;

$CoreAlignerUtil::STA_STRerror    = 'error';
$CoreAlignerUtil::STA_STRfinished = 'finished';
$CoreAlignerUtil::STA_STRwaiting  = 'waiting';
$CoreAlignerUtil::STA_STRrunning  = 'running';

###############################################################################
my(%MSG_logHash) = ( 'Reading data files'                        =>  1,
                     'Collecting conserved neighbors'            =>  1,
                     'Creating a directed graph'                 =>  1,
                     'Converting the graph into a triplet graph' =>  1,
                     'Eliminating loops'                         =>  1,
                     'Finding longest paths'                     =>  1,
                     'Output the result'                         =>  1,
                     'Done'                                      => -1,
                     );
my(%MSG_progressHash) = 
                   ( 'Reading data files'                        =>   10,
                     'Collecting conserved neighbors'            =>   20,
                     'Creating a directed graph'                 =>   30,
                     'Converting the graph into a triplet graph' =>   45,
                     'Eliminating loops'                         =>   60,
                     'Finding longest paths'                     =>   75,
                     'Output the result'                         =>   90,
                     'Done'                                      =>  100,
                     );

###############################################################################
#
sub getCoreIndexTableName {
    my($tabname) = 'core_tables_idx';

    return $tabname;
}

###############################################################################
#
sub getCoreResultTableName {
    my($coreTabId) = shift;

    my($tabname) = sprintf("core_result_%s", $coreTabId);

    return $tabname;
}

###############################################################################
#
sub createCoreIndexTable {
    my($db) = shift;

    #
    my($tab) = getCoreIndexTableName();
    my($sql) = "create table if not exists $tab ("
             . "core_tab_id     varchar(50) NOT NULL default '',"
             . "status          int         NOT NULL default 0,"
             . "status2         varchar(50) NOT NULL default '',"
             . "progress        int         NOT NULL default 0,"
             . "clust_tab_id    varchar(50) NOT NULL default '',"
             . "cmd             mediumtext  NOT NULL,"
             . "cdate           timestamp   NOT NULL default CURRENT_TIMESTAMP,"
             . "primary key (core_tab_id)"
             . ")";
    $db->do($sql);

    return;
}

###############################################################################
#
sub assignCoreTabId {
    my($db) = shift;
    my($ref) = shift;

    #
    if ($ref->{'status2'} =~ /^\s*$/) {
        $ref->{'status2'} = 'Start';
    }
    if ($ref->{'progress'} =~ /^\s*$/) {
        $ref->{'progress'} = 0;
    }

    # MySQL $B$G$N%G!<%?9`L\<oJL!'?tCM(B(=0), $BJ8;zNs(B(=1)
    my(%keyHash) = ('core_tab_id'   => 1,
                    'status'        => 0,
                    'status2'       => 1,
                    'progress'      => 0,
                    'clust_tab_id'  => 1,
                    'cmd'           => 1,
                    );
    my($cols) = '';
    my($vals) = '';
    foreach my$key (keys(%keyHash)) {
        my($val) = $ref->{"$key"};

        $cols .= ',' if ($cols ne '');
        $cols .= $key;
        $vals .= ',' if ($vals ne '');
        if ($keyHash{"$key"} != 0) {
            $val = $db->{'conn'}->quote($val);
            $vals .= "$val";
        }
        else {
            $vals .= $val;
        }
    }
    

    #
    my($tab) = getCoreIndexTableName();
    my($sql) = "insert $tab ($cols) values ($vals)";
    print STDERR "SQL :: $sql\n" if ($main::DEBUG);
    $db->do($sql);

    return;
}

###############################################################################
#
sub updateCoreStatus {
    my($db) = shift;
    my($coreTabId) = shift;
    my($status) = shift;

    #
    my($tab) = getCoreIndexTableName();
    my($sql) = '';
    $sql .= "update $tab set ";
    $sql .= "status=$status ";
    $sql .= "where core_tab_id='$coreTabId'";
    $db->do($sql);

    return;
}

###############################################################################
#
sub updateCoreProgress {
    my($db) = shift;
    my($coreTabId) = shift;
    my($status2) = shift;
    my($progress) = shift || 0;

    #
    my($tab) = getCoreIndexTableName();
    my($sql) = '';
    $sql .= "update $tab set ";
    $sql .= "status2='$status2', progress=$progress ";
    $sql .= "where core_tab_id='$coreTabId'";
    $db->do($sql);

    return;
}

###############################################################################
#
sub getCoreStatus {
    my($db) = shift;
    my($coreTabId) = shift;
    my($ref) = undef();

    if (!$coreTabId) {
        return $ref;
    }

    #
    my($tab) = getCoreIndexTableName();
    my($opt) = {};
    $opt->{'columns'} = '*';
    $opt->{'where'} = "core_tab_id='$coreTabId'";
    my($refRes) = $db->select_fetch($tab, $opt);
    if ($refRes->{'ROWS'} != 0) {
        # Core$B9=B$2r@O7k2L$"$j(B
        $ref = $refRes->{'INFO'}->[0];

        #
        if ($ref->{'status'} == $CoreAlignerUtil::STA_finished) {
            $ref->{'STATUS'}  = $CoreAlignerUtil::STA_STRfinished;
        }
        elsif (0 < $ref->{'status'}) {
            $ref->{'STATUS'}  = $CoreAlignerUtil::STA_STRrunning;
        }
        else {
            $ref->{'STATUS'}  = $CoreAlignerUtil::STA_STRerro;
        }

        my($species) = ($ref->{'cmd'} =~ /-SPECIES=(\S+)/);
        if ($species =~ /^\'(.*)\'$/) {
            $species = $1;
        }
        $ref->{'SPECIES'} = $species;


        $ref->{'progress'} = getProgressPercent($ref->{'status2'});
    }

    return $ref;
}

###############################################################################
#
sub createCoreTabId {
    my($coreTabId) = sprintf("%d_%d", time(), $$);

    return $coreTabId
}

###############################################################################
#
sub getCoreAlignerTabId {
    my($db) = shift;
    my($refOpt0) = shift;

    #
    my($tab) = getCoreIndexTableName();
    my($opt) = {};
    $opt->{'columns'} = '*';
    my($refRes) = $db->select_fetch($tab, $opt);
    foreach my$ref (@{$refRes->{'INFO'}}) {
        my($refOpt) = parseCoreCmdOpts($ref->{'cmd'});
        my($sta) = isSameCoreCmdOpts($refOpt0, $refOpt);
        if ($sta) {
            # $BF1$82r@O>r7o(B
            return $ref->{'core_tab_id'};
        }
    }

    # $BF1$82r@O>r7o$,L5$$(B
    my($coreTabId) = undef();

    return $coreTabId;
}

###############################################################################
# Core$B9=B$2r@O7k2L$r6hJL$9$k$?$a$N%3%^%s%I%*%W%7%g%s(B
# $B!J0l;~%U%!%$%k$J$I$O!"2r@O7k2L$K1F6A$r5Z$\$5$J$$$N$G=|30!K(B
sub isValidOptkey {
    my($key) = shift;
    my(%validOptKeys) = ('-CLUST_TAB_ID'   => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-SPECIES'        => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-refsp'          => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-OTHER_OPTIONS'  => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-OutGroup'       => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-out_species'    => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-SpGrp'          => $CoreAlignerUtil::TYPE_KEY_STR,
                         '-NBR_WIN'        => $CoreAlignerUtil::TYPE_KEY_NUM,
                         '-CONS_RATIO'     => $CoreAlignerUtil::TYPE_KEY_NUM,
                         '-NBR_CONS_RATIO' => $CoreAlignerUtil::TYPE_KEY_NUM,
                         '-MIN_SPCOV'      => $CoreAlignerUtil::TYPE_KEY_NUM,
                         '-SPCOV_REG_CUT'  => $CoreAlignerUtil::TYPE_KEY_NUM,
                         '-SPCOV_SPRATIO'  => $CoreAlignerUtil::TYPE_KEY_NUM,
                        );

    return $validOptKeys{"$key"};
}

###############################################################################
#
sub parseCoreCmdOpts {
    my($cmd) = shift;

    #
    my($refOpt) = {};
    my(@optList) = ($cmd =~ /\s(\-\S+)/gi);
    foreach my$opt (@optList) {
        my($k, $v) = split('=', $opt, 2);
        my($type_key) = isValidOptkey($k);
        if (!$type_key) {
            # $B$3$N(B $key $B$O<h$j=P$5$J$$(B
            next;
        }

        if ($v =~ /^\'(.*)\'$/) {
            $v = $1; # ' $B$G0O$^$l$F$$$k>l9g!"A08e$N(B ' $B$r=|5n(B
        }
        if ($type_key == $CoreAlignerUtil::TYPE_KEY_NUM) {
            # $B?tCM%*%W%7%g%s(B $B"*(B $BM-8z7e$r9g$o$;$k(B
            $v = sprintf("%.3f", $v);
        }

        $refOpt->{"$k"} = $v;
    }

    return $refOpt;
}

###############################################################################
#
sub isSameCoreCmdOpts {
    my($refOpt1) = shift;
    my($refOpt2) = shift;

    my(@keys1) = keys(%{$refOpt1});
    my(@keys2) = keys(%{$refOpt2});
    if (scalar(@keys1) != scalar(@keys2)) {
        # $B0z?t$N8D?t$,0c$&(B
        return 0;
    }

    foreach my$key (@keys1) {
        if (!exists($refOpt2->{"$key"})) {
            # $key $B$,(B $refOpt2 $B$KB8:_$7$J$$(B
            return 0;
        }

        my($type) = isValidOptkey($key);
        if ($type == $CoreAlignerUtil::TYPE_KEY_STR) {
            if ($refOpt1->{"$key"} ne $refOpt2->{"$key"}) {
                # $B%*%W%7%g%s$NCM$,0c$&(B
                return 0;
            }
        }
        elsif ($type == $CoreAlignerUtil::TYPE_KEY_NUM) {
            if ($refOpt1->{"$key"} != $refOpt2->{"$key"}) {
                # $B%*%W%7%g%s$NCM$,0c$&(B
                return 0;
            }
        }
        else {
        }
    }

    return 1;
}

###############################################################################
#
sub convFileCore {
    my($fileSrc) = shift;
    my($fileDst) = shift;
    my($delim) = shift;

    my($fhSrc) = FileHandle->new("$fileSrc") || die("Can not open $fileSrc($!)");
    my($fhDst) = FileHandle->new(">$fileDst") || die("Can not open $fileDst($!)");

    #
    $fhSrc->getline(); # $B%X%C%@9T$rFI$_<N$F$k(B
    while(my$line = $fhSrc->getline()) {
        $line =~ s#[\r\n]*$##;
        my($id, $gene, $dir, $clustId, @dat) = split("\t", $line);
        my($name) = join("\t", @dat);
        $fhDst->print(join($delim, $id, $gene, $dir, $clustId, $name), "\n");
    }
    $fhDst->close();
    $fhSrc->close();

    return;
}

###############################################################################
#
sub insertCoreFile {
    my($db) = shift;
    my($coreTabId) = shift;
    my($fileCore) = shift;

    #
    my($tab) = getCoreResultTableName($coreTabId);
    my($sql) = "drop table if exists $tab";
    $db->do($sql);

    #
    my($sql) = "create table $tab ("
             . "tab_id      varchar(24) NOT NULL default '',"
             . "gene        varchar(24) NOT NULL default '',"
             . "dir         smallint    NOT NULL,"
             . "cluster_id  smallint    NOT NULL,"
             . "name        mediumtext,"
	     . "order_num   serial,"		## core genome order
             . "primary key (tab_id)"
             . ")";
    $db->do($sql);

    #
    my($fileTmpCore) = "$ENV{'MBGD_HOME'}/work/tmp_core.$$";
    my($delim) = '#DeLiM#';
    convFileCore($fileCore, $fileTmpCore, $delim);

    #
    my($cols) = join(',', 'tab_id', 'gene', 'dir', 'cluster_id', 'name');
    my($sql) = "load data local infile '$fileTmpCore' "
             . "replace into table $tab "
             . "fields terminated by '$delim' "
             . "($cols)";
    $db->do($sql);

    unlink("$fileTmpCore");

    return;
}

###############################################################################
#
sub getSpeciesByClusterTabId {
    my($db) = shift;
    my($clustTabId) = shift;

    my($tab) = 'cluster_tables_idx';
    my($opt) = {};
    $opt->{'columns'} = "*";
    $opt->{'where'} = sprintf("clusterID='%s'", $clustTabId);

    #
    my($refRes) = $db->select_fetch($tab, $opt);
    if ($refRes->{'ROWS'} == 0) {
        print STDERR "DBG :: ERROR :: Can not found Cluster-Table-Id [$clustTabId]\n" if ($main::DEBUG);
        return;
    }
    my($ref) = $refRes->{'INFO'}->[0];

    my($refTabSpec) = {};
    ($refTabSpec->{'SPECIES'})   = ($ref->{'cmd'} =~ /\-spec=(\S+)/i);
    ($refTabSpec->{'OUT_GROUP'}) = ($ref->{'cmd'} =~ /\-Ooutgroup=(\S+)/i);

    return $refTabSpec;
}

###############################################################################
#
sub writeGenomes {
    my($db) = shift;
    my($refSpOrder) = shift;
    my($fileGenomes) = shift;
    my($order) = 1;
    my(@spec_incomplete_list);

    #
    my($refGenomes) = {};
    my($tab) = 'genome g, chromosome c';
    my($opt) = {};
    $opt->{'columns'} = "g.sp as sp, g.abbrev as abbrev, g.strain as strain, sum(c.seq_length) as seq_length, c.status as status";
    $opt->{'where'} = "g.id=c.genome";
    $opt->{'group'} = "g.id";
    my($refRes) = $db->select_fetch($tab, $opt);
    foreach my$ref (@{$refRes->{'INFO'}}) {
        my($sp) = $ref->{'sp'};
        $refGenomes->{"$sp"} = $ref;
        if ($ref->{'status'}) {
            push(@spec_incomplete_list, $sp);
        }
    }

    #
    my($fh) = FileHandle->new(">$fileGenomes") || die("Can not open $fileGenomes($!)");
    foreach my$sp (@{$refSpOrder}) {
        my($ref) = $refGenomes->{"$sp"};
        my($name) = $ref->{'abbrev'};
        $name .= "";
        $fh->print(join("\t", $sp, $name, $ref->{'seq_length'}, $order), "\n");

        $order++;
    }
    $fh->close();

    return @spec_incomplete_list;
}

###############################################################################
#
sub mapview_file {
    my($file_mapview) = shift;
    my($mapview_ref) = {};

    my($fh) = IO::File->new("$file_mapview") || die("Can not open $file_mapview($!)");
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        $line =~ s#[\r\n]*$##;

        my($spid, $sp, $chrid, $contigid, $seqno, $ordno, $accession, $from1, $to1, $dir) = split(/\t/, $line);

        my($ent) = $mapview_ref->{"$sp"}->{"$seqno"}->{"$ordno"} = {};
        $ent->{'chrid'}     = $chrid;
        $ent->{'contigid'}  = $contigid;
        $ent->{'accession'} = $accession;
        $ent->{'from1'}     = $from1;
        $ent->{'to1'}       = $to1;
        $ent->{'dir'}       = $dir;
    }
    $fh->close();

    return $mapview_ref;
}

###############################################################################
#
sub mapview_mysql {
    my($db) = shift;
    my($spec) = shift;
    my($mapview_ref) = {};

    my($tab) = "mapview";
    my($where) = "sp=?";
    my($sql) = "select * from $tab where $where";

    my($sth) = $db->prepare($sql);
    $sth->execute($spec);
    while (my$ref=$sth->fetchrow_hashref()) {

        my($sp)    = $ref->{'sp'};
        my($seqno) = $ref->{'seqno'};
        my($ordno) = $ref->{'ordno'};
        my($ent) = $mapview_ref->{"$sp"}->{"$seqno"}->{"$ordno"} = {};
        $ent->{'chrid'}     = $ref->{'chrid'};
        $ent->{'contigid'}  = $ref->{'contigid'};
        $ent->{'accession'} = $ref->{'accession'};
        $ent->{'from1'}     = $ref->{'from1'};
        $ent->{'to1'}       = $ref->{'to1'};
        $ent->{'dir'}       = $ref->{'dir'};
    }

    return $mapview_ref;
}

###############################################################################
#
sub writeGenes_mapview {
    my($db) = shift;
    my($spec) = shift;
    my($dirOut) = shift;
    my($mapview_ref) = shift;

    #
    my($filename) = "$dirOut/$spec.txt";
    my($fh) = IO::File->new(">$filename") || die("Can not open $filename($!)");

    #
    my(@title_list) = ('sp', 'name', 'gene', 'from1', 'to1', 'dir', 'type', 'descr');
    $fh->print(join("\t", @title_list), "\n");

    #
    my($ofs) = 0;
    my(@seqno_list) = sort {$a <=> $b} keys(%{$mapview_ref->{"$spec"}});
    foreach my$seqno (@seqno_list) {
        my(@ordno_list) = sort {$a <=> $b} keys(%{$mapview_ref->{"$spec"}->{"$seqno"}});
        foreach my$ordno (@ordno_list) {
            my($ent) = $mapview_ref->{"$spec"}->{"$seqno"}->{"$ordno"};

            my($tab) = 'gene';
            my($where) = "sp=? and contigid=?";
            my($order) = "from1";
            my($sql) = "select * from $tab where $where order by $order";
            my($sth) = $db->prepare($sql);
            $sth->execute($spec, $ent->{'contigid'});
            while (my$ref=$sth->fetchrow_hashref()) {
                my(@dat) = ();
                push(@dat, $ref->{'sp'});
                push(@dat, $ref->{'name'});
                push(@dat, $ref->{'gene'});
                push(@dat, $ref->{'from1'} + $ofs);
                push(@dat, $ref->{'to1'}   + $ofs);
                push(@dat, $ref->{'dir'});
                push(@dat, $ref->{'type'});
                push(@dat, $ref->{'descr'});

                $fh->print(join("\t", @dat), "\n");
            }
            $ofs += $ent->{'to1'} - $ent->{'from1'} + 1;
        }
    }
    $fh->close();

    return;
}

###############################################################################
# $BJ#?t@w?'BN$O!"2>A[E*$K#1K\$N@w?'BN$H8+$J$9(B
sub writeGenes {
    my($db) = shift;
    my($refSpOrder) = shift;
    my($dirOut) = shift;
    my(@keyList) = ('g.sp', 'g.name', 'g.gene', 'g.from1', 'g.to1', 'g.dir', 'g.type', 'g.descr');
    my(@keyList) = ('sp', 'name', 'gene', 'from1', 'to1', 'dir', 'type', 'descr');

    # $B@w?'BND9$N<hF@(B
    my($spIn) = '';
    foreach my$sp (@{$refSpOrder}) {
        $spIn .= ',' if ($spIn ne '');
        $spIn .= "'$sp'";
    }
    my($seq_length_ref) = {};
    my($tab) = 'chromosome';
    my($opt) = {};
    $opt->{'columns'} = "*";
    $opt->{'where'} = "sp in ($spIn)";
    my($refRes) = $db->select_fetch($tab, $opt);
    foreach my$ref (@{$refRes->{'INFO'}}) {
        my($sp)    = $ref->{'sp'};
        my($seqno) = $ref->{'seqno'};
        my($len)   = $ref->{'seq_length'};

        $seq_length_ref->{"$sp"}->{"$seqno"} = $len;
    }

    #
    my($seq_ofs_ref) = {};
    foreach my$sp (keys(%{$seq_length_ref})) {
        my($seqno_ref) = $seq_length_ref->{"$sp"};
        foreach my$seqno (keys(%{$seqno_ref})) {
            $seq_ofs_ref->{"$sp"}->{"$seqno"} = 0;
            for(my$i = 1; $i < $seqno; $i++) {
                $seq_ofs_ref->{"$sp"}->{"$seqno"} += $seqno_ref->{"$i"};
            }
        }
    }

    #
    foreach my$sp (@{$refSpOrder}) {
        my($filename) = "$dirOut/$sp.txt";
        my($fh) = FileHandle->new(">$filename") || die("Can not open $filename($!)");
        $fh->print(join("\t", @keyList), "\n");

        my($tab) = 'gene g, chromosome c';
        my($opt) = {};
        $opt->{'columns'} = "g.*, c.seqno";
        $opt->{'where'} = "g.chrid=c.id and g.sp='$sp'";
        $opt->{'order'} = "c.seqno, g.from1";
        my($refRes) = $db->select_fetch($tab, $opt);
        foreach my$ref (@{$refRes->{'INFO'}}) {
            my($out) = '';
            my($seqno) = $ref->{'seqno'};
            $ref->{'from1'} += $seq_ofs_ref->{"$sp"}->{"$seqno"};
            $ref->{'to1'}   += $seq_ofs_ref->{"$sp"}->{"$seqno"};
            foreach my$key (@keyList) {
                $out .= "\t" if ($out ne '');
                $out .= $ref->{"$key"};
            }
            $fh->print($out, "\n");
        }
        $fh->close();
    }

    return;
}

###############################################################################
#
sub writeClusterHeader {
    my($refSpOrder) = shift;
    my($fh) = shift;

    #
    my(@header) = ('ClusterID', 'Size', @{$refSpOrder}, 'FuncCat', 'Gene');
    $fh->print(join("\t", @header), "\n");

    return;
}

###############################################################################
#
sub writeCluster {
    my($clustid) = shift;
    my($subclustid) = shift;
    my($refSpOrder) = shift;
    my($refCluster) = shift;
    my($fh) = shift;

    return if ($clustid <= 0);

    #
    $fh->print(join('_', $clustid, $subclustid), "\t");

    #
    my($size) = 0;
    my($refOrf) = {};
    foreach my$sp (@{$refSpOrder}) {
        if (exists($refCluster->{'IN'}->{"$sp"})) {
            $refOrf->{"$sp"} = $refCluster->{'IN'}->{"$sp"};
        }
        elsif (exists($refCluster->{'OUT'}->{"$sp"})) {
            $refOrf->{"$sp"} = $refCluster->{'OUT'}->{"$sp"};
        }
        else {
            $refOrf->{"$sp"} = [];
        }
        $size += scalar(@{$refOrf->{"$sp"}});
    }

    #
    $fh->print($size, "\t");
    foreach my$sp (@{$refSpOrder}) {
        $fh->print(join(' ', @{$refOrf->{"$sp"}}), "\t");
    }

    #
    $fh->print('dmy_func', "\t");
    $fh->print('dmy_gene', "\n");

    return;
}

###############################################################################
#
sub writeClustTab {
    my($db) = shift;
    my($tabId) = shift;
    my($refSpOrder) = shift;
    my($fileClustTab) = shift;

    my($fh) = FileHandle->new(">$fileClustTab") || die("Can not open $fileClustTab($!)");
    writeClusterHeader($refSpOrder, $fh);

    my($refCluster) = {};
    my($prevCluster)    = -1;
    my($prevSubCluster) = -1;
    my($clustid, $subclustid, $sp, $name, $spname, $dom);
    my($subclustid);

    my($tab) = sprintf("cluster_domclust_cache_%s", $tabId);
    my($opt) = {};
    $opt->{'columns'} = '*';
    $opt->{'order'} = 'clustid, subclustid';
    my($refRes) = $db->select_fetch($tab, $opt);
    foreach my$ref (@{$refRes->{'INFO'}}) {
        $clustid    = $ref->{'clustid'};
        $subclustid = $ref->{'subclustid'};
        $sp         = $ref->{'sp'};
        $name       = $ref->{'name'};
        $spname     = $ref->{'spname'};
        $dom        = $ref->{'dom'};

        my($spname_dom) = $spname;
        if ($dom) {
            $spname_dom .= "($dom)";
        }

        #
        my($ioGroup) = 'IN';
        if ($subclustid == 0) {
            $ioGroup = 'OUT';
        }

        #
        if ($main::MODE =~ /^cluster$/i) {
            # $B%/%i%9%?!<C10L$G=PNO(B
            if ($clustid != $prevCluster) {
                writeCluster($prevCluster, 1, $refSpOrder, $refCluster, $fh);
            }
        }
        else {
            if (($clustid != $prevCluster) || ($subclustid != $prevSubCluster)) {
                if ($prevSubCluster != 0) {
                    # $B%5%V%/%i%9%?!<!J(Bin-group$B!KC10L$G=PNO(B
                    writeCluster($prevCluster, $prevSubCluster, $refSpOrder, $refCluster, $fh);

                    # in-group $B$N%G!<%?$r%/%j%"(B
                    $refCluster->{"$ioGroup"} = {};
                }
            }
        }

        if ($clustid != $prevCluster) {
            # $B%/%i%9%?!<$N%G!<%?$r%/%j%"(B
            $refCluster = {};
        }

        #
        if (!exists($refCluster->{"$ioGroup"}->{"$sp"})) {
            $refCluster->{"$ioGroup"}->{"$sp"} = [];
        }
        push(@{$refCluster->{"$ioGroup"}->{"$sp"}}, $spname_dom);


        $prevCluster    = $clustid;
        $prevSubCluster = $subclustid;
    }

    if ($main::MODE =~ /^cluster$/i) {
        writeCluster($prevCluster, 1, $refSpOrder, $refCluster, $fh);
    }
    else {
        writeCluster($prevCluster, $prevSubCluster, $refSpOrder, $refCluster, $fh);
    }

    #
    $fh->close();

    return;
}

###############################################################################
#
sub buildCoreAlignerOptions {
    my($refOpt) = shift;
    my($forCheck) = shift;

    my($opt) = '';
    foreach my$k (sort(keys(%{$refOpt}))) {
        next if ($k =~ /^CMD_/i);

        my($v) = $refOpt->{"$k"};
#        next if ($v =~ /^\s*$/);
	if (! $forCheck && $k eq 'OTHER_OPTIONS') {
	     $opt .= " $v";
	} else {

             $opt .= " -$k='$v'";
	}
    }

    return $opt;
}

###############################################################################
#
sub setupCoreAligner {
    my($dirBase) = shift;
    my($refFormOpt) = shift;

    my($clustTabId)   = $refFormOpt->{'CLUST_TAB_ID'};
    my($refsp)        = $refFormOpt->{'REF_SPEC'};
    my($nbrWin)       = $refFormOpt->{'NBR_WIN'};
    my($consRatio)    = $refFormOpt->{'CONS_RATIO'};
    my($nbrConsRatio) = $refFormOpt->{'NBR_CONS_RATIO'};
    my($minSpcov)     = $refFormOpt->{'MIN_SPCOV'};
    my($spcovRegCut)  = $refFormOpt->{'SPCOV_REG_CUT'};
    my($spcovSpRatio)  = $refFormOpt->{'SPCOV_SPRATIO'};
    my($otherOptions) = $refFormOpt->{'OTHER_OPTIONS'};
    my($spGroup)      = $refFormOpt->{'SP_GROUP'};

    my($coreTabId) = createCoreTabId();
    my($dirWork) = sprintf("%s/tmp_corealigner_%s", $dirBase, $coreTabId);
    File::Path::mkpath("$dirWork", 0, 0750);

    my($fileGenomes) = "$dirWork/genomes";
    my($dirGene)     = "$dirWork/gene";
    File::Path::mkpath("$dirGene", 0, 0750);
    my($fileClustTab)      = "$dirWork/clusttab";
    my($fileClustOut)      = "$dirWork/clustout";
    my($fileCoreClustOut)  = "$dirWork/core_clustout";
    my($fileAlignOut)      = "$dirWork/alignout";
    my($filePairOut)       = "$dirWork/pairout";
    my($fileLinkOut)       = "$dirWork/linkout";
    my($fileCoreAlignerOut) = "$dirWork/corealigner.out";
    my($fileCoreAlignerErr) = "$dirWork/corealigner.err";

    my($dbRecog) = MBGD::DB->new($main::DBNAME_RECOG);
    my($refTabSpec) = getSpeciesByClusterTabId($dbRecog, $clustTabId);
    my($refSpOrder) = [ split(/,/, $refTabSpec->{'SPECIES'}) ];

    #
    my($dbMbgd) = MBGD::DB->new($main::DBNAME_MBGD);
    my(@spec_incomplete_list) = writeGenomes($dbMbgd, $refSpOrder, $fileGenomes);
    if (! -e "$dirGene") {
        File::Path::mkpath("$dirGene", 0, 0750);
    }
    writeGenes($dbMbgd, $refSpOrder, $dirGene);
    foreach my$spec (@spec_incomplete_list) {
        my($mapview_ref) = mapview_mysql($dbMbgd, $spec);
        writeGenes_mapview($dbMbgd, $spec, $dirGene, $mapview_ref);
    }

    #
    writeClustTab($dbRecog, $clustTabId, $refSpOrder, $fileClustTab);

    my($refCoreOpt) = {};
    $refCoreOpt->{'CORE_TAB_ID'}    = $coreTabId;
    $refCoreOpt->{'CLUST_TAB_ID'}   = $clustTabId;
    $refCoreOpt->{'SPECIES'}        = $refTabSpec->{'SPECIES'};
    $refCoreOpt->{'genomes'}        = $refTabSpec->{'SPECIES'};
    $refCoreOpt->{'out_species'}    = $refTabSpec->{'SPECIES'};
    $refCoreOpt->{'datapath'}       = $main::MYSQL_DB;
    $refCoreOpt->{'clusttab'}       = $main::MYSQL_DBRECOG;
    $refCoreOpt->{'clustout'}       = $fileClustOut;
    $refCoreOpt->{'core_clustout'}  = $fileCoreClustOut;
    $refCoreOpt->{'alignout'}       = $fileAlignOut;
    $refCoreOpt->{'pairout'}        = $filePairOut;
    $refCoreOpt->{'linkout'}        = $fileLinkOut;
    $refCoreOpt->{'refsp'}          = $refsp;
    $refCoreOpt->{'NBR_WIN'}        = $nbrWin;
    $refCoreOpt->{'CONS_RATIO'}     = $consRatio;
    $refCoreOpt->{'NBR_CONS_RATIO'} = $nbrConsRatio;
    $refCoreOpt->{'MIN_SPCOV'}      = $minSpcov;
    $refCoreOpt->{'SPCOV_REG_CUT'}  = $spcovRegCut;
    $refCoreOpt->{'SPCOV_SPRATIO'}  = $spcovSpRatio;
    $refCoreOpt->{'OTHER_OPTIONS'}  = $otherOptions;
    $refCoreOpt->{'OutGroup'}       = $refTabSpec->{'OUT_GROUP'};
    $refCoreOpt->{'SpGrp'}          = $spGroup;
    $refCoreOpt->{'DIR_WORK'}       = $dirWork;
    $refCoreOpt->{'corealigner_out'} = $fileCoreAlignerOut;
    $refCoreOpt->{'corealigner_err'} = $fileCoreAlignerErr;
    #
    $refCoreOpt->{'CMD_PATH'} = $main::CMD_corealign;
    $refCoreOpt->{'CMD_OPT'}  = CoreAlignerUtil::buildCoreAlignerOptions($refCoreOpt);
    my $CMD_OPT_REGISTER  = CoreAlignerUtil::buildCoreAlignerOptions($refCoreOpt, 1);

    #
    my($refCoreAlignerIdx) = {};
    $refCoreAlignerIdx->{'core_tab_id'}  = $refCoreOpt->{'CORE_TAB_ID'};
    $refCoreAlignerIdx->{'status'}       = $CoreAlignerUtil::STA_waiting;
    $refCoreAlignerIdx->{'clust_tab_id'} = $refCoreOpt->{'CLUST_TAB_ID'};
    $refCoreAlignerIdx->{'cmd'}          = join(' ', $refCoreOpt->{'CMD_PATH'},
                                                  $CMD_OPT_REGISTER);
    assignCoreTabId($dbRecog, $refCoreAlignerIdx);

    return $refCoreOpt;
}

###############################################################################
#
sub execCoreAligner {
    my($dbRecog) = shift;
    my($refCoreOpt) = shift;

    updateCoreStatus($dbRecog, $refCoreOpt->{'CORE_TAB_ID'}, $$);

    #
    my($cmd) = join(' ', $refCoreOpt->{'CMD_PATH'},
                         $refCoreOpt->{'CMD_OPT'});

    #
    $cmd .= " > $refCoreOpt->{'corealigner_out'}";

    # $BI8=`%(%i!<=PNO$K=P$F$$$k(B CoreAligner $B$N%W%m%0%l%9>pJs$N<h$j9~$_(B
    $cmd = "($cmd) 2>&1 | $main::CMD_tee $refCoreOpt->{'corealigner_err'}";
    $cmd .= "| $main::CMD_updateCoreAlignerProgress -CORE_TAB_ID=$refCoreOpt->{'CORE_TAB_ID'}";

    warn("CMD :: $cmd\n") if ($main::DEBUG);
    my($fileLogCmd) = $refCoreOpt->{'corealigner_out'} . '.cmd';
    my($fh) = FileHandle->new(">$fileLogCmd");
    if ($fh) {
        $fh->print("$cmd");
        $fh->close();
    }
open(O,">/tmp/ooo");
print O "command: $cmd\n";
close(O);

    system("$cmd");

    return;
}

###############################################################################
#
sub readAlignout {
    my($filename) = shift;

    #
    my($ref) = {};
    $ref->{'LIST_ID'} = [];
    $ref->{'DATA'}    = {};

    #
    my($fh) = FileHandle->new("$filename") || die("Can not open $filename($!)");
    while(my$line = $fh->getline()) {
        $line =~ s#[\r\n]*$##;

        if ($line =~ /^#/) {
            $line =~ s/^#\s*//; # $B9TF,$N(B '#' $B$r<h$j=|$/(B
            my(@specList) = split(/,/, $line);
            $ref->{'SPECIES'} = [ @specList ];
        }
        else {
            my($id, $name, $dir, @dat) = split(/\t/, $line);
            my($clustId) = pop(@dat);

            push(@{$ref->{'LIST_ID'}}, $id); # ID $B$N=P8==g$rJ]B8(B
            $ref->{'DATA'}->{"$id"} = {};
            $ref->{'DATA'}->{"$id"}->{'NAME'}     = $name;
            $ref->{'DATA'}->{"$id"}->{'DIR'}      = $dir;
            $ref->{'DATA'}->{"$id"}->{'CLUST_ID'} = $clustId;
        }
    }
    $fh->close();

    return $ref;
}

###############################################################################
#
sub readCoreClustout {
    my($filename) = shift;

    #
    my($ref) = {};
    $ref->{'LIST_ID'} = [];
    $ref->{'DATA'}    = {};

    #
    my($fh) = FileHandle->new("$filename") || die("Can not open $filename($!)");
    while(my$line = $fh->getline()) {
        $line =~ s#[\r\n]*$##;

        if ($line =~ /^#/) {
            $line =~ s/^#\s*//; # $B9TF,$N(B '#' $B$r<h$j=|$/(B
            my(@specList) = split(/\t/, $line);
            $ref->{'SPECIES'} = [ @specList ];
        }
        else {
            my($id, @dat) = split(/\t/, $line);
            push(@{$ref->{'LIST_ID'}}, $id); # ID $B$N=P8==g$rJ]B8(B
            $ref->{'DATA'}->{"$id"} = {};
            foreach my$sp (@{$ref->{'SPECIES'}}) {
                $ref->{'DATA'}->{"$id"}->{"$sp"} = shift(@dat);
            }
        }
    }
    $fh->close();

    return $ref;
}

###############################################################################
#
sub writeCoreAlignerResult {
    my($fileAlignout) = shift;
    my($fileCoreClustOut) = shift;
    my($fileCoreResult) = shift;

    my($fh) = FileHandle->new(">$fileCoreResult") || die("Can not open $fileCoreResult($!)");

    #
    my($refAlign) = readAlignout($fileAlignout);

    #
    my($refCoreClust) = readCoreClustout($fileCoreClustOut);

    #
    my(@splist) = @{$refAlign->{'SPECIES'}};
    $fh->print('#SPECIES=', join(',', @splist), "\n");

    #
    foreach my$id (@{$refAlign->{'LIST_ID'}}) {
        $fh->print(join("\t", $id,
                        $refAlign->{'DATA'}->{"$id"}->{'NAME'},
                        $refAlign->{'DATA'}->{"$id"}->{'DIR'},
                        $refAlign->{'DATA'}->{"$id"}->{'CLUST_ID'},
                    ));
        foreach my$sp (@splist) {
            $fh->print("\t", $refCoreClust->{'DATA'}->{"$id"}->{"$sp"});
        }
        $fh->print("\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub printCoreAlignerResultHeader {
    my($refCoreSta) = shift;

    my(@list);

    push(@list, sprintf("#CORE_TAB_ID=%s", $refCoreSta->{'core_tab_id'}));
    push(@list, sprintf("#STATUS=%s",      $refCoreSta->{'STATUS'}));
    push(@list, sprintf("#STATUS2=%s",     $refCoreSta->{'status2'}));

    my($per) = $refCoreSta->{'progress'};
    $per = 99 if (100 <= $per);
    push(@list, sprintf("#PROGRESS=%s",    $per));
    push(@list, sprintf("#SPECIES=%s",     $refCoreSta->{'SPECIES'}));
    push(@list, sprintf("#QUERY=%s",      createDisplayCommandString($refCoreSta->{'cmd'})));

    my($http_host_ref) = RECOG::RecogCommon::get_http_host();
    push(@list, sprintf("#EXEC_SERVER=%s", $http_host_ref->{'NAME'}));
    push(@list, sprintf("#EXEC_PORT=%s",   $http_host_ref->{'PORT'}));
    push(@list, "#START_DATA");

    foreach my $l (@list) {
        print $l, "\n";
    }

    return;
}

sub createDisplayCommandString {
	my($cmd) = @_;
	my($refOpt) = parseCoreCmdOpts($cmd);
	my($cmdStr);
	my($separator) = "&";
	foreach $key (keys %{$refOpt}) {
		$keystr = $key;
		$keystr =~ s/^\-//;
		$cmdStr .= $separator if ($cmdStr);
		$cmdStr .= "$keystr=$refOpt->{$key}";
	}
	$cmdStr;
}

###############################################################################
#
sub printCoreAlignerResult {
    my($db) = shift;
    my($coreTabId) = shift;

    my($tab) = getCoreResultTableName($coreTabId);
    my($opt) = {};
    $opt->{'columns'} = '*';
    $opt->{'order'} = 'order_num';
    my($refRes) = $db->select_fetch($tab, $opt);
    foreach my$ref (@{$refRes->{'INFO'}}) {
        print join("\t", $ref->{'tab_id'},
                         $ref->{'gene'},
                         $ref->{'dir'},
                         $ref->{'cluster_id'},
                         $ref->{'name'});
        print "\n";
    }

    return;
}

###############################################################################
#
sub parseProgress {
    my($db) = shift;
    my($coreTabId) = shift;
    my($filename) = shift;

    #
    my($fh) = FileHandle->new("$filename") || die("Can not oprn $filename($!)");
    while(my$line = $fh->getline()) {
        my($status2) = $line;
        my($progress) = 0;

        $status2 =~ s#[\s\r\n]*$##; # $B9TKv$N!"6uGrJ8;z$d2~9TJ8;z$r=|5n(B

        if (exists($MSG_logHash{"$status2"})) {
            updateCoreProgress($db, $coreTabId, $status2, $progress);
            if ($MSG_logHash{"$status2"} < 0) {
                last;
            }
        }
    }
    $fh->close();

    return;
}

###############################################################################
#
sub getProgressPercent {
    my($progress) = shift;
    my($percent) = 0;

    if (exists($MSG_progressHash{"$progress"})) {
        $percent = $MSG_progressHash{"$progress"};
    }

    return $percent;
}

###############################################################################
1;#
###############################################################################
