#!/usr/bin/perl -w
use strict;
use File::Basename;
use Getopt::Std;
my $PROGRAM = basename $0;
my $USAGE=
"Usage: $PROGRAM -r REF_CLUSTER [-i INITIAL | -l FILE_LIST | FILE...]
-1: only 1 to 1 relationship
";

use DomRefine::General;

my %OPT;
getopts('r:i:l:1', \%OPT);

my $MODE = "cmpr";
my $OPT_1 = "";
if ($OPT{1}) {
    $MODE = "1to1";
    $OPT_1 = "-1";
}

my ($REF_CLUSTER, $SUFFIX);
if ($OPT{r}) {
    $REF_CLUSTER = $OPT{r};
    $SUFFIX = $REF_CLUSTER;
    $SUFFIX =~ s/^.*\///;
    $SUFFIX =~ s/\.cluster$//; # obsolete
    $SUFFIX =~ s/\.dom$//;
} else {
    die $USAGE;
}

my @FILE;
if ($OPT{i}) {
    my $initial = $OPT{i};
    @FILE = ("$initial",
	     "$initial.merged",
	     "$initial.merged.divide_cluster.out",
	     "$initial.merged.boundary_move.out",
	     "$initial.merged.bounded",
	     "$initial.merged.bounded.divided"
	);
} elsif ($OPT{l}) {
    @FILE = `cat $OPT{l}`;
    chomp(@FILE);
} else {
    @FILE = @ARGV;
}

my @EVALUATED_FILE = ();
for my $file (@FILE) {
    if ($file =~ /^\s*#/) {
	next;
    }
    if ($file =~ /\s/) {
	next;
    }
    if (-f $file) {
	my $evaluated_file = "$file.$MODE.$SUFFIX";
	push @EVALUATED_FILE, $evaluated_file;
	if (-f $evaluated_file) {
	    print STDERR "$evaluated_file exists.\n";
	    next;
	}
	system "cat $file | dom_renumber -d | dom_overlap $OPT_1 $REF_CLUSTER > $evaluated_file";
    }
}

if (@EVALUATED_FILE and $OPT{1}) {
    print_evaluation_summary(\@EVALUATED_FILE, "evaluate.$MODE.$SUFFIX");
}

################################################################################
### Function ###################################################################
################################################################################

sub print_evaluation_summary {
    my ($r_evaluated_file, $summary_file) = @_;

    my @num = ();
    for my $evaluated_file (@EVALUATED_FILE) {
	my $result = `wc -l ${evaluated_file}`;
	chomp($result);
	if ($result =~ /^(\d+) (\S+)$/) {
	    my ($num, $file) = ($1, $2);
	    if ($file ne $evaluated_file) {
		die;
	    }
	    push @num, $num;
	} else {
	    die;
	}
    }

    open(RESULT, ">$summary_file") || die;
    print RESULT join("\t", "Num", "Change", "Gain", "Loss", "Full List", "Gain List", "Loss List"), "\n";
    print RESULT join("\t", $num[0], "NA", "NA", "NA", ${$r_evaluated_file}[0], "NA", "NA"), "\n";
    for (my $i=1; $i<@num; $i++) {
	my $diff = $num[$i] - $num[$i-1];
	my @gain = ();
	my @loss = ();
	get_gain_loss(${$r_evaluated_file}[$i-1], ${$r_evaluated_file}[$i], \@gain, \@loss);
	my $gain = @gain;
	my $loss = @loss;
	print RESULT join("\t", $num[$i], $diff, $gain, -$loss, ${$r_evaluated_file}[$i], "'@gain'", "'@loss'"), "\n";
	check_diff_gain_loss($diff, $gain, $loss);
    }
    my $diff_total = $num[-1] - $num[0];
    my @gain_total = ();
    my @loss_total = ();
    get_gain_loss(${$r_evaluated_file}[0], ${$r_evaluated_file}[-1], \@gain_total, \@loss_total);
    my $gain_total = @gain_total;
    my $loss_total = @loss_total;
    print RESULT join("\t", $num[-1], $diff_total, $gain_total, -$loss_total, "(Total Change)"), "\n";
    check_diff_gain_loss($diff_total, $gain_total, $loss_total);
    close(RESULT);
}

sub check_diff_gain_loss {
    my ($diff, $gain, $loss) = @_;
    
    if ($diff == $gain - $loss) {
    } else {
	print STDERR "ERROR: $diff != $gain - $loss\n";
    }
}

sub get_gain_loss {
    my ($file1, $file2, $r_gain, $r_loss) = @_;

    my @motif1 = `cat $file1 | cut -f4`;
    chomp(@motif1);
    my %motif1;
    for my $motif (@motif1) {
	$motif1{$motif}++;
    }

    my @motif2 = `cat $file2 | cut -f4`;
    chomp(@motif2);
    my %motif2;
    for my $motif (@motif2) {
	$motif2{$motif}++;
    }
    
    my @motif = uniq(@motif1, @motif2);
    for my $motif (@motif) {
	if (! $motif1{$motif} and $motif2{$motif}) {
	    push @{$r_gain}, $motif;
	}
	if ($motif1{$motif} and ! $motif2{$motif}) {
	    push @{$r_loss}, $motif;
	}
    }
}
