#!/usr/bin/perl -s
use strict;
use DirHandle;
use FileHandle;
use File::Basename;
use File::Path;
use GD;

package GdGraphBar;

$GdGraphBar::DEFAULT_options_ref = {
    'image_w'       => 100,
    'image_h'       => 100,
    'colors'        => join(',', '0000FF', '00FF00', '00FFFF', 'FF0000', 'FF00FF', 'FFFF00'),
    'pad_t'         => 20,
    'pad_b'         => 50,
    'pad_l'         => 50,
    'pad_r'         => 20,
    'y_axis'        => '',
    'x_axis'        => '',
    'y_tick_size'   =>  5,
};

###############################################################################
#
sub new {
    my($class) = shift;
    my($w) = shift;
    my($h) = shift;
    my($self) = {};

    bless($self, $class);
    $self->_init($w, $h);

    return $self;
}

###############################################################################
#
sub _init {
    my($self) = shift;
    my($w) = shift;
    my($h) = shift;

    $self->init_options();
    if ($w <= 0) {
        $w = $self->get_option('image_w');
    }
    if ($h <= 0) {
        $h = $self->get_option('image_h');
    }
    $self->{'GD_IMAGE'} = GD::Image->new($w, $h);
    $self->{'GD_IMAGE'}->colorAllocate(0xff, 0xff, 0xff);

    #
    my($opt) = {
        'image_w' => $w,
        'image_h' => $h,
    };
    $self->set_options($opt);

    return;
}

###############################################################################
#
sub gd_image {
    my($self) = shift;

    my($gd_image) = $self->{'GD_IMAGE'};

    return $gd_image;
}

###############################################################################
#
sub init_options {
    my($self) = shift;

    $self->{'GD_OPTIONS'} = {};
    $self->set_options($GdGraphBar::DEFAULT_options_ref);

    return;
}

###############################################################################
#
sub set_options {
    my($self) = shift;
    my($opt_ref) = shift;

    foreach my$k (keys(%{$opt_ref})) {
        if (!exists($GdGraphBar::DEFAULT_options_ref->{"$k"})) {
            next;
        }

        $self->{'GD_OPTIONS'}->{"$k"} = $opt_ref->{"$k"};
    }

    return;
}

###############################################################################
#
sub unset_options {
    my($self) = shift;
    my($opt_ref) = shift;

    foreach my$k (keys(%{$opt_ref})) {
        delete($self->{'GD_OPTIONS'}->{"$k"});
    }

    return;
}

###############################################################################
#
sub get_option {
    my($self) = shift;
    my($k) = shift;

    my($val) = $self->{'GD_OPTIONS'}->{"$k"};

    return $val;
}

###############################################################################
#
sub color_allocate {
    my($self) = shift;

    #
    my($colors) = $self->get_option('colors');

    #
    my($gd_image) = $self->gd_image();
    $self->{'COLORS'} = [];
    foreach my$c (split(/,/, $colors)) {
        my($rgb) = hex($c);
        my($cr) = ($rgb >> 16) & 0xff;
        my($cg) = ($rgb >>  8) & 0xff;
        my($cb) = ($rgb >>  0) & 0xff;

#        print STDERR "DBG :: colorAllocate($cr, $cg, $cb) :: $c\n" if ($main::DEBUG);
        my($color) = $gd_image->colorAllocate($cr, $cg, $cb);
        push(@{$self->{'COLORS'}}, $color);
    }

    return;
}

###############################################################################
#
sub color {
    my($self) = shift;
    my($idx) = shift;

    $idx %= scalar(@{$self->{'COLORS'}});
#    print STDERR "DBG :: COLORS[$idx]\n" if ($main::DEBUG);
    my($color) = $self->{'COLORS'}->[$idx];

    return $color;
}

###############################################################################
#
sub plot {
    my($self) = shift;
    my($dat_ref) = shift;
    my($lab_ref) = shift;
    my($x1, $y1, $x2, $y2);
    my($log10) = log(10);

    #
    my($n_dat) = scalar(@{$dat_ref});
    my($image_w) = $self->get_option('image_w');
    my($image_h) = $self->get_option('image_h');
    my($pad_t) = $self->get_option('pad_t');
    my($pad_b) = $self->get_option('pad_b');
    my($pad_l) = $self->get_option('pad_l');
    my($pad_r) = $self->get_option('pad_r');
    my($bar_w) = ($image_w - $pad_l - $pad_r) / $n_dat;
    my($bar_h) = ($image_h - $pad_t - $pad_b);
    my($font_small) = GD::Font->Small;
    my($font_small_w) = $font_small->width;
    my($font_small_h) = $font_small->height;
    my($font_large) = GD::Font->Large;
    my($font_large_w) = $font_large->width;
    my($font_large_h) = $font_large->height;
    my($y_axis) = $self->get_option('y_axis');
    my($y_tick_size) = $self->get_option('y_tick_size');

    #
    $self->color_allocate();

    #
    my($min_val) = $dat_ref->[0];
    my($max_val) = $dat_ref->[0];
    foreach my$d (@{$dat_ref}) {
        if ($min_val > $d) {
            $min_val = $d;
        }
        elsif ($max_val < $d) {
            $max_val = $d;
        }
    }

    #
    my($min_plot_val);
    my($max_plot_val);
    if ($y_axis =~ /^log$/i) {
        $min_plot_val = log($min_val) / $log10;
        $max_plot_val = log($max_val) / $log10;
    }
    else {
        $min_plot_val = $min_val;
        $max_plot_val = $max_val;
    }
    print STDERR "DBG :: max_plot_val ($max_val) :: $max_plot_val\n" if ($main::DEBUG);

    #
    my($gd_image) = $self->gd_image();
    my($color_black) = $gd_image->colorAllocate(0, 0, 0);

    # draw axis
    $x1 = $pad_l;
    $y1 = $image_h - $pad_b;
    $x2 = $image_w - $pad_r;
    $y2 = $image_h - $pad_b;
    print STDERR "DBG :: line($x1, $y1, $x2, $y2)\n" if ($main::DEBUG);
    $gd_image->line($x1, $y1, $x2, $y2, $color_black);

    $x2 = $pad_l;
    $y2 = $pad_t;
    print STDERR "DBG :: line($x1, $y1, $x2, $y2)\n" if ($main::DEBUG);
    $gd_image->line($x1, $y1, $x2, $y2, $color_black);

    # draw scale
    my($scale_hash_ref) = {};
    my($max_axis_val);
    my($digits) = log($max_val) / $log10;
    my($digits2) = 10 ** int($digits);
    $max_axis_val = $max_val / $digits2;
    my(@scale_ticks);
    if ($max_axis_val <= 5) {
        $max_axis_val = 5 * $digits2;
        if ($y_axis =~ /^log$/i) {
            @scale_ticks = (5 * $digits2, $digits2, 0.5 * $digits2, 0.1 * $digits2);
        }
        else {
            @scale_ticks = (5 * $digits2, 3 * $digits2, $digits2);
        }
    }
    else {
        $max_axis_val = 10 * $digits2;
        if ($y_axis =~ /^log$/i) {
            @scale_ticks = (10 * $digits2, 5 * $digits2, $digits2);
        }
        else {
            @scale_ticks = (10 * $digits2, 6 * $digits2, 2 * $digits2);
        }
    }
    foreach my$k (@scale_ticks) {
        $scale_hash_ref->{"$k"} = 0;
    }
    if ($y_axis =~ /^log$/i) {
        $max_axis_val = log($max_axis_val) / $log10;
    }
    print STDERR "DBG :: max_axis_val ($max_plot_val) :: $max_axis_val\n" if ($main::DEBUG);
    foreach my$s (keys(%{$scale_hash_ref})) {
        my($s_val) = $s;
        if ($y_axis =~ /^log$/i) {
            $s_val = log($s_val) / $log10;
        }
        my($x1) = $pad_l;
        my($y1) = $image_h - $pad_b - ($s_val / $max_axis_val) * $bar_h;
        my($x1) = $x1 - $y_tick_size;
        my($y2) = $y1;
        $gd_image->line($x1, $y1, $x2, $y2, $color_black);
        my($x) = $pad_l - length($s) * $font_small_w - $y_tick_size - 1;
        my($y) = $image_h - $pad_b - ($s_val / $max_axis_val) * $bar_h - $font_small_h / 2;
        $gd_image->string($font_small, $x, $y, $s, $color_black);
    }

    # plot data
    for(my$i = 0; $i < $n_dat; $i++) {
        my($val) = $dat_ref->[$i];
        if ($y_axis =~ /^log$/i) {
            $val = log($val) / $log10;
        }

        #
        my($x1) = $pad_l + $bar_w * $i;
        my($x2) = $x1 + $bar_w;
        my($y2) = $image_h - $pad_b;
        my($y1) = $y2 - ($val / $max_axis_val) * $bar_h;
        my($color) = $self->color($i);
        print STDERR "DBG :: filledRectangle($x1, $y1, $x2, $y2, $color)\n" if ($main::DEBUG);
        $gd_image->filledRectangle($x1, $y1, $x2, $y2, $color);

        #
        if (defined($lab_ref) && ($lab_ref->[$i] ne '')) {
            my($x) = ($x1 + $x2) / 2 - ($font_small_w * length($lab_ref->[$i])) / 2;
            my($y) = $image_h - $pad_l + 1;
            print STDERR "DBG :: string($x, $y, $lab_ref->[$i])\n" if ($main::DEBUG);
            $gd_image->string($font_small, $x, $y, $lab_ref->[$i], $color_black);
        }
    }

    return;
}

###############################################################################
#
sub png {
    my($self) = shift;

    my($gd_image) = $self->gd_image();
    my($img) = $gd_image->png();

    return $img;
}

###############################################################################
#
sub jpeg {
    my($self) = shift;

    my($gd_image) = $self->gd_image();
    my($img) = $gd_image->jpeg();

    return $img;
}

###############################################################################
package main;
if ($0 eq __FILE__) {
    my($obj) = GdGraphBar->new(600, 400);

    my($dat_ref) = [10, 20, 30, 40, 30000];

    my($opt) = {
        'y_axis' => 'log',
    };
#    $obj->set_options($opt);
    $obj->plot($dat_ref, $dat_ref);

    print $obj->png();
}

###############################################################################
1;#
###############################################################################
