#!/usr/bin/perl

###############################################################################
# package MBGD::DB;
#   new()
#   get_name()
#   sta_connect()
#   connect()
#   set_mode()
#   disconnect()
#   err()
#   errstr()
#   prepare()
#   execute()
#   select_fetch()
#   access_error()
#   do()
#   list_tables()
#   exist_table()
#   drop_tables()
#   drop_work_tables()
#   check_updated()
#   store_object()
#   fetch_objects()
#   get_objects()
#   find_objects()
#   fetch_object()
#   get_table()
#   create_table()
#   flush_table()
#   define_cache()
#   destroy_cache()
#   assign_cache()
#   convKeywordForQuickSearch()
#
###############################################################################
# package MBGD::ObjTab;
#   store()
#
###############################################################################

package MBGD::DB;
use DBI;
use MBGD;
use MBGD::DBTable;
use MBGD::Schema;
use MBGD::SQL;
use Carp;

%DBobj;

###############################################################################
sub new {
	my($class, $DBS, $option) = @_;
	my($this) = {};
	bless $this, $class;
	
	if (! $DBS) {
		if ($ENV{MYSQL_DB}) {
			if ($ENV{MYSQL_DB} !~ /^dbi:/) {
				$DBS = "dbi:mysql:$ENV{MYSQL_DB}";
			} else {
				$DBS = "$ENV{MYSQL_DB}";
			}
		} else {
			$DBS = $MBGD::Config{default_dbs}
		}
	} elsif ($DBS !~ /^dbi:mysql/i) {
		$DBS = "dbi:mysql:$DBS";
	}
	return $DBobj{$DBS} if ($DBobj{$DBS}->{conn} and ! ($option->{new}));

	($dmy, $dbdrv, $dbname) = split(/:/, $DBS);
	if ($dbname =~ /database=([^;]+)/) {
		$dbname = $1;
	}
	$this->{dbdrv} = $dbdrv;
	$this->{dbname} = $dbname;
	$this->{sql} = "MBGD::SQL_${dbdrv}"->new;
	$this->{conn} = $this->connect($DBS,
			$option->{username}, $option->{PW});
	if ($option->{mode}) {
		$this->{mode} = $option->{mode};
	}

	$DBobj{$DBS} = $this;

	return $this;
}

###############################################################################
sub get_name {
    my($this) = shift;

    return $this->{dbname};
}

###############################################################################
sub sta_connect {
    my($this) = shift;

    if (defined($this->{'conn'})) {
        return 1;
    }

    return;
}

###############################################################################
sub connect {
	my($this, $DBS, $username, $PW, $opt) = @_;

	if ($DBS !~ /^dbi:mysql/i) {
		$DBS = "dbi:mysql:$DBS";
	}
    if (defined($ENV{'MYSQL_MYCNF'}) && -e "$ENV{'MYSQL_MYCNF'}") {
        # $B4D6-JQ?t(B $MYSQL_MYCNF $B$G(B MySQL $B$N(B config $B%U%!%$%k;XDj$5$l$F$$$k(B
		$DBS .= ";mysql_read_default_file=$ENV{'MYSQL_MYCNF'}";
    }
	elsif (!defined($username) || !defined($PW)) {
        if (-e "$ENV{'RECOG_HOME'}/etc/my.cnf") {
            $DBS .= ";mysql_read_default_file=$ENV{'RECOG_HOME'}/etc/my.cnf";
        }
        elsif (-e "$ENV{'MBGD_HOME'}/etc/my.cnf") {
            $DBS .= ";mysql_read_default_file=$ENV{'MBGD_HOME'}/etc/my.cnf";
        }
        else {
            $DBS .= ";mysql_read_default_file=$ENV{'HOME'}/.my.cnf";
        }
	}
	$this->{conn} = DBI->connect($DBS, $username, $PW, $opt);
#	$this->{conn}->{LongReadLen} = 10000;
	$this->{conn};
}

###############################################################################
sub set_mode {
	my($this, $mode, $opt) = @_;
	$this->{mode} = $mode;
	foreach $k (keys %{$opt}) {
		$this->{$k} = $opt->{$k};
	}
}

###############################################################################
sub disconnect {
	my($this) = @_;
	$this->{conn}->disconnect;
	undef $this->{conn};
}

###############################################################################
sub err {
	my($this) = @_;
	return $this->{conn}->err();
}

###############################################################################
sub errstr {
	my($this) = @_;
	return $this->{conn}->errstr();
}

###############################################################################
sub prepare {
	my($this, @args) = @_;
print STDERR "SQL :: @args\n" if ($main::DEBUG);
	$this->{conn}->prepare(@args);
}

###############################################################################
sub execute {
	my($this, @args) = @_;
	my $sth = $this->{conn}->prepare(@args);
	if (! $sth->execute()) {
confess ">>Error SQL: @args\n";
		print STDERR join(' ', @args),"   $this->{conn}, $sth\n";
		$this->{sth} = $sth;
		return 0;
	}
	return $sth;
}

###############################################################################
sub select_fetch {
	my($this, $tables, $opt) = @_;
	my($sql);
	my($refRes);
	my($sth);
	if ($opt->{limit}) {
		$sql = "select count(*) from $tables";
		$sql .= " where $opt->{where}" if ($opt->{where});
		$sth = $this->execute($sql);
		if (! $sth) {
			$this->access_error($sth, $sql);
			return undef;
		}
		($refRes->{MAXROWS}) = $sth->fetchrow_array;
	}
	$opt->{columns} = '*' if (! $opt->{columns});
	$sql = "select $opt->{columns} from $tables";
	$sql .= " where $opt->{where}" if ($opt->{where});
	$sql .= " group by $opt->{group}" if ($opt->{group});
	$sql .= " having $opt->{having}" if ($opt->{having});
	$sql .= " order by $opt->{order}" if ($opt->{order});
	$sql .= " limit $opt->{limit}" if ($opt->{limit});

	print STDERR "$sql\n" if ($main::DEBUG);
	$sth = $this->execute($sql);
	if (! $sth) {
		$this->access_error($sth, $sql);
		return undef;
	}
	$refRes->{ROWS} = $sth->rows;
	if (! $opt->{limit}) {
		$refRes->{MAXROWS} = $refRes->{ROWS};
	}
	$refRes->{NAME} = $sth->{NAME};
	$refRes->{INFO} = [];
	while (my $ent = $sth->fetchrow_hashref) {
		push(@{$refRes->{INFO}}, $ent);
	}
	$sth->finish;
	return $refRes;
}

###############################################################################
sub access_error {
	my($sth, $sql) = @_;
	print STDERR "DB access error\n";
	if ($sth) {
		print STDERR "( ", $sth->err, " )\n";
		print STDERR "Msg::", $sth->errstr, "\n";
	}
	print STDERR "SQL::", $sql, "\n";
}

###############################################################################
sub do {
	my($this, @args) = @_;
	foreach my $stmt (@args) {
		$this->{conn}->do($stmt);
	}
}

###############################################################################
sub list_tables {
	my($this) = @_;
	return $this->{conn}->tables;
}

###############################################################################
sub exist_table {
	my($this, $table) = @_;
	foreach my $t ($this->list_tables) {
        my($db);
        my($tab) = ($t =~ /\`{0,1}([^\`]+)\`{0,1}/);
        if ($t =~ /\./) {
            $t =~ s#\`##g;
            ($db, $tab) = split(/\./, $t);
        }
        if ($tab eq $table) {
            return 1;
        }
	}
	return 0;
}

###############################################################################
sub drop_tables {
	my($this, @tables) = @_;
	foreach my $t ($this->list_tables) {
		if (! @tables || grep($t eq $_, @tables)) {
#			foreach $sql ( $this->{sql}->drop_table($this, $t) ) {  # for Pg/Sybase
			foreach $sql ( $this->{sql}->drop_table($t) ) {         # for MySQL
print STDERR "SQL :: $sql\n" if ($main::DEBUG);
				$this->do($sql);
			}
			$this->{tables}->{$t} = {};
		}
	}
}

###############################################################################
sub drop_work_tables {
	my($this) = @_;
	foreach my $t ($this->list_tables) {
		if ($t =~ /^(.*)_work$/) {
			$this->do(MBGD::SQL::drop_table($this, $t));
print STDERR ">$1<<\n" if ($main::DEBUG);
			$this->{tables}->{$1} = {};
		}
	}
}

###############################################################################
sub check_updated {
	my($this, $class, $condition, $opt) = @_;
	my $tab = $this->get_table($class, {mode=>'update'});
	return -1 if (! $table);
	my $sth = $tab->difference($condition);
	while (my @a = $sth->fetchrow_array) {
		print join(' ', @a),"\n";
	} 
}

###############################################################################
sub store_object {
	my($this, $obj, $opt) = @_; 
	my $class = $obj->get_schema_classname;
	my $tabopt = {mode=>'update', create=>1};
	$tabopt->{tablename} = $opt->{tablename} if ($opt->{tablename});

	my $table = $this->get_table($class, $tabopt);
	return -1 if (! $table);

	my $id = $table->store_object($obj, $opt);
	return $id;
}

###############################################################################
sub fetch_objects {
	my($this, $class, @idlist) = @_;
	my $table = $this->get_table($class);
	return () if (! $table);
	$table->fetch_object(@idlist);
}

###############################################################################
sub get_objects {
	my($this, $class, $keyvalues, $fetch_opt) = @_;
	my $table = $this->get_table($class, $fetch_opt);
	return () if (! $table);
	$table->get_object($keyvalues, $fetch_opt);
}

###############################################################################
sub find_objects {
	my($this, $class, $condition, $opt) = @_;
	my $table = $this->get_table($class, $opt);
	return () if (! $table);
	$table->find_object($condition, $opt);
}

###############################################################################
sub fetch_object {
	my($this, $obj, $fetch_opt) = @_;
	my $table = $this->get_table($obj->get_schema_classname);
	return -1 if (! $table);
	if (my $id = $obj->get_identifier) {
		return $table->fetch_object($id, $fetch_opt);
	} elsif ($keyvalue = $obj->primary_key_value) {
		return $table->get_object([$keyvalue], $fetch_opt);
	} else {
	}
}

###############################################################################
sub get_table {
	my($this, $class, $opt) = @_;
	my $mode = 'read';
    my($tabopt) = {};
	$mode = $opt->{mode} if ($opt->{mode});
	my $tablename;
	if ($opt->{tablename}) {
		$tablename = $opt->{tablename};
		$tabopt->{tablename} = $tablename;
	} else {
		$tablename = MBGD::Schema::get_schema_classname($class);
	}

#	if ($this->{tables}->{$class}->{$mode}) {
	if ($this->{tables}->{$tablename}->{$mode}) {
#		return $this->{tables}->{$class}->{$mode};
		return $this->{tables}->{$tablename}->{$mode};
	}

	if ($this->exist_table($tablename) == 0) {
		if ($opt->{create}) {
			$tabopt->{create} = 1;
		} else {
			return 0;
		}
	}
#	return $this->{tables}->{$class}->{$mode}
#		= MBGD::DBTable->new($this, $class, $tabopt);
	return $this->{tables}->{$tablename}->{$mode}
		= MBGD::DBTable->new($this, $class, $tabopt);
}

###############################################################################
sub create_table {
	my($this, $class) = @_;
	MBGD::DBTable->new($this, $class, {create => 1});
}

###############################################################################
sub flush_table {
	my($this, $table) = @_;
	$this->{conn}->do("flush table $table");
}

###############################################################################
# mysql key cache (key buffer)
sub define_cache {
	my($this, $name, $size) = @_;
	$this->{conn}->do("set global $name.key_buffer_size=$size");
}

###############################################################################
sub destroy_cache {
	my($this, $name) = @_;
	$this->{conn}->do("set global $name.key_buffer_size=0");
}

###############################################################################
sub assign_cache {
	my($this, $tbllist, $keycache) = @_;
	if (ref($tbllist) eq 'ARRAY') {
		$tbllist = join(' ', @{$tbllist});
	}
	$this->{conn}->do("cache index $tbllist in $keycache");
}

###############################################################################
sub convKeywordForQuickSearch {
	my($keyword) = @_;
	my($tmpkwd, $kw, $newkw);
	if ($keyword =~ /"([^"]+)"/) {
		$newkw = $keyword;
		while ($keyword =~ /"([^"]+)"/g) {
			my $origphrase = $phrase = $1;
			$phrase =~ s/ /#_#/g;
			$newkw =~ s/$origphrase/$phrase/;
		}
		$keyword = $newkw;
	}
	@kwlist = split(/ /, $keyword);
	if (@kwlist > 1) {
		foreach $kw (@kwlist) {
			$kw = "+$kw" if ($kw !~ /^[+-<>()~]/);
			$tmpkwd .= ' ' if ($tmpkwd);
			$tmpkwd .= $kw;
		}
	} else {
		$tmpkwd = $keyword;
	}
	$tmpkwd =~ s/#_#/ /g;
	$tmpkwd;
}

###############################################################################
package MBGD::ObjTab;

$objtab = 'otab';

###############################################################################
sub store {
	my($db, $tabname) = @_;
	my $sth = $db->{conn}->prepare(
		"insert into $objtab (tabname) values ($tabname)");
	$sth->execute;
	$sth = $db->{conn}->prepare($this->get_idval);
	$sth->execute;
	my @ret = $sth->fetchrow_array;
	return $ret[0];
}

###############################################################################
1;
###############################################################################
