#!/usr/local/bin/perl

package MBGD::DBTable;

use DBI;
use MBGD;
use MBGD::Schema;
use MBGD::SQL;
use Carp;

sub new {
	my($class, $db, $classref, $option) = @_;
	## $classref is a specification of the class of table, which can
	## be either of Schema object reference, MBGD object reference,
	## or string of MBGD class name.

	my($this) = {};
	bless $this, $class;

	if (ref($classref) eq 'MBGD::Schema') {
		$this->{schema} = $classref;
		$classname = $this->{schema}->{classname};
	} elsif (ref($classref) =~ /MBGD::/) {
		$classname = ref($classref);
	} elsif ($classref =~ /MBGD::/) {
		$classname = $classref;
	} else {
		$classname = "MBGD::$classref";
	}

	if (! $this->{schema}) {
		$this->{schema} = MBGD::Schema->new("$classname", $db->{dbname});
	}
	if ($option->{tablename}) {
		$this->{tablename} = $option->{tablename};
	} else {
		$this->{tablename} = $this->{schema}->{classname};
	}

	$this->{db} = $db;

	$this->{upd_tablename} = $this->{tablename};
	if ($option->{recreate} == 1) {
		$this->recreate_table;
	} elsif ($option->{create} == 1) {
		$this->create_table;
	}
##	if ($option->{prepare_insert} == 1) {
##		$this->prepare_for_insert;
##	}
	return $this;
}
sub recreate_table {
	my($this, $opt) = @_;
	$this->drop_table($this->{tablename});
	$this->create_table($this->{tablename}, $opt);
}
sub drop_table {
	my($this) = @_;
	$this->{db}->do( $this->{db}->{sql}->drop_table($this->{tablename}) );
}
sub create_table {
	my($this, $opt) = @_;
	my $sql;
	if ($opt->{script}) {
		$this->{db}->do($opt->{script});
	} else {
		my $sqlopt = { tablename => $this->{tablename} };
		if (exists($opt->{'merge_table'})) {
			$sqlopt->{'merge_table'} = $opt->{'merge_table'};
		}
		if ($this->{temporal}) {
			$sqlopt->{temporal} = 1;
			$sqlopt->{tablename} = $this->{work_tablename};
		}
		@sql = $this->{schema}->convSQL($this->{db}->{sql}, $sqlopt);
	}
	foreach $sql (@sql) {
		print STDERR $sql, "\n" if ($main::DEBUG);
		$this->{db}->do($sql);
	}
}
sub insert {
	my($this, $obj, @attr) = @_;
	if (! @attr) {
		@attr = $this->{schema}->get_attributes({for_insert=>1});
	}
	my $sth = $this->prepare_for_insert(@attr);
	my @values = $obj->getValues(@attr);
	$sth->execute(@values);
print STDERR "#### inserted: ", $obj->primary_key_value, "\n" if ($main::DEBUG);
	return $sth;
}
sub update {
	my($this, $obj, @attr) = @_;
	if (! @attr) {
		@attr = $this->{schema}->get_attributes({for_update=>1});
	}
	my $sth = $this->prepare_for_update($obj, \@attr);
	my @values = $obj->getValues(@attr);
	$sth->execute(@values);
print STDERR "#### updated: ", $obj->primary_key_value, "\n" if ($main::DEBUG);
	return $sth;
}
sub prepare_for_insert {
	my($this, @attr) = @_;
	my $sql;
	if (! @attr) {
		@attr = $this->{schema}->get_attributes({for_insert=>1});
	}
	my $fieldnum = 0 + @attr;
	$sql = "insert into $this->{upd_tablename} (" .
		join(',', @attr) .
		") values(" .  ("?, " x ($fieldnum - 1)) . "?)";
	$sth = $this->{db}->prepare($sql);
	return $sth;
}
sub prepare_for_update {
	my($this, $obj, $attr) = @_;
	my $sql;
	my(@primkeys) = $obj->primary_key_value;
	my $query = $this->create_query_for_keysearch(\@primkeys);
	if(! @$attr) {
		@$attr = $this->{schema}->get_attributes({for_update=>1});
	}
	$sql = "update $this->{upd_tablename} set " . join('=?,', @$attr) ."=?";
        $sql .= " where $query";
	$sth = $this->{db}->prepare($sql);
	return $sth;
}
#sub check_obj_equivalency {
#	my($this, $object, $retrieved, $opt) = @_;
#	foreach my $r (@{$retrieved}) {
#		if ($object->equivalent($r, $opt)) {
#			return $r->get_identifier;
#		}
#	}
#	return 0;
#}

sub store_object {
	my($this, $object, $opt) = @_;
	my($retid);
	my(@values);
	if ($this->{schema} != $object->{schema}) {
		warn "Schema mismatch!!: >$this->{schema}->{classname}<>$object->{schema}->{classname}<\n";
	}
	my @primkeys = $object->primary_key_value;

	if ($this->{temporal}) {
		if (ref($opt->{attr}) eq 'ARRAY') {
			@attr = @{$opt->{attr}};
##		} else {
##			## use default for @attr
##			@attr = $this->{schema}->get_attributes({for_insert=>1});
		}
		$sth = $this->insert($object, @attr);
		return $retid = $this->{db}->{conn}->selectrow_array(
			$this->{db}->{sql}->get_idval($this->{tablename}) );
#	} elsif (($ret) = $this->get_object(\@primkeys, {depth => 1} )) {
	} elsif (($ret) = $this->get_object(\@primkeys)) {
		## find duplicated entry
		$retid = $ret->get_identifier;
		if ( $object->equivalent($ret, { for_insert=>1,
#						ignore_ref=>1,
				}) > 0 ) {
			## same entry
			return $retid;
		} elsif ($opt->{ignore}) {
			## abort
			return -1;
		} elsif ($opt->{check}) {
print STDERR "#### (check) updated: ", $object->primary_key_value, "\n" if ($main::DEBUG);
			return 'UPD';
		} else {
			## update it
			my @attr;
			if (ref($opt->{attr}) eq 'ARRAY') {
				@attr = @{$opt->{attr}};
##			} else {
##				## use default for @attr
##				@attr = $this->{schema}->
##					get_attributes({for_update=>1});
			}
			$sth = $this->update($object, @attr);
		}
	} elsif ($opt->{check}) {
print STDERR "#### (check) insert: ", $object->primary_key_value, "\n" if ($main::DEBUG);
		return 'INS';
	} elsif ($opt->{ignore}) {
		## ignore
	} else {
		if (ref($opt->{attr}) eq 'ARRAY') {
			@attr = @{$opt->{attr}};
##		} else {
##			## use default for @attr
##			@attr = $this->{schema}->get_attributes({for_insert=>1});
		}
		$sth = $this->insert($object, @attr);
		$retid = $this->{db}->{conn}->selectrow_array(
			$this->{db}->{sql}->get_idval($this->{tablename}) );
	}

	return $retid;
}
sub query {
	my($this, $restriction, $opt) = @_;
	my $projection = $opt->{fields};
	$projection = '*' if (! $projection);
	if ($opt->{count}) {
		$projection = "count($projection)";
	}
	my $sql = "select $projection from $this->{tablename}";
	$sql .= " where $restriction" if ($restriction);
	$sql .= " order by $opt->{order}" if ($opt->{order});
	$sql .= " limit $opt->{limit}" if ($opt->{limit});
	my $sth = $this->{db}->prepare($sql);

	my$sta = $sth->execute();
	if (! $sta) {
		Carp::confess "SQL: $sql\n";
	}
	return $sth;
}
sub find_object {
	my($this, $condition, $opt) = @_;
	my $sth = $this->query($condition, $opt);
	if ($opt->{count}) {
		my($count) = $sth->fetchrow_array;
		return $count;
	}
	my $occ_check = 0;
	if ($opt->{depth} eq 'recursive') {
		$occ_check = {};
	} elsif ($opt->{depth}) {
		$occ_check = $opt->{depth};
	}
	return $this->construct_object_from_searchres($sth, $occ_check);
}
sub create_query_for_keysearch {
	my($this, $keylist, $opt) = @_;
	my $query;
	my $flag1 = 0;
	my @keynames;
	if ($opt->{keys}) {
		@keynames = split(/:/, $opt->{keys});
	} else {
		@keynames = $this->{schema}->primary_key();
	}
	foreach my $key (@{$keylist}) {
		next if (! $key);
		$query .= " or" if ($flag1);
		my $flag2 = 0;
		$query .= " (";
		@keys = split(/:/, $key);

		for (my $i = 0; $i < @keynames; $i++) {
			next if (! $keys[$i]);
			$query .= " and" if ($flag2);
			$query .= ( " $keynames[$i] = " .
				 $this->{db}->{conn}->quote( $keys[$i],
				 $this->{schema}->get_type($keynames[$i]) )
				);
			$flag2++;
		}
		$query .= " )";
		$flag1++;
	}
	if ($opt->{add_cond}) {
		$query = "($query) and ($opt->{add_cond})";
	}
	$query;
}
sub construct_object_from_searchres {
	my($this, $sth, $occ_check) = @_;
	my @RetVal;
	my $classname = $this->{schema}->{perl_classname};
	while ( my $rowdata = $sth->fetchrow_hashref ) {
		my $retobj = "$classname"->new($this->{db}, $Values);
#		my $retobj = "$classname"->new($Values);
		my $Values = {};
		foreach my $attr (keys %{$rowdata}) {
			if ( (ref($occ_check) || $occ_check > 0) &&
				(my $newclass = $this->{schema}->{foreign}->
							{$attr})
				    && $rowdata->{$attr} =~ /^\d+$/) {
				## recursive mode

				$dbt = MBGD::DBTable->new($this->{db},
					"$newclass");
				if (my @obj = $dbt->__fetch_object(
					$occ_check, $rowdata->{$attr})) {
					$Values->{$attr} = $obj[0];
				} else {
				}
			} else {
				$Values->{$attr} = $rowdata->{$attr};
			}
		}
		$retobj->setValues($Values);
		push(@RetVal,$retobj);
	}
	@RetVal;
}
sub fetch_object {
	my($this, @idlist) = @_;
	if (ref($idlist[0]) eq 'ARRAY') {
		@idlist = @{$idlist};
	} elsif (ref($idlist[$#idlist]) == 'HASH') {
		$opt = pop(@idlist);
	}
##print STDERR join(' ', @idlist),"\n";
	return () if (! @idlist);

	my($occ_check) = 0;
	if ($opt->{depth} eq 'recursive') {
		$occ_check = {};
	} elsif ($opt->{depth}) {
		$occ_check = $opt->{depth};
	}
	return $this->__fetch_object($occ_check, @idlist);
}
sub __fetch_object {
	my($this, $occ_check, @idlist) = @_;
	if (ref($occ_check)) {
		## recursive mode
		foreach my $id (@idlist) {
			return () if ($occ_check->{$this->{tablename},$id});
			$occ_check->{$this->{tablename}, $id} = 1;
		}
	} elsif ($occ_check) {
		return () if ($occ_check-- <= 0);
	}
	$query = "id in (" . join(',', @idlist) . ")";
	my $sth = $this->query($query);
	return $this->construct_object_from_searchres($sth, $occ_check);
}
sub get_object {
	my($this, $keylist, $opt) = @_;
	return () if (! (0+@{$keylist}) );

	$query = $this->create_query_for_keysearch($keylist, $opt);
	my $sth = $this->query($query, $opt);
	if ($opt->{count}) {
		my($count) = $sth->fetchrow_array;
		return $count;
	}

	my $occ_check = 0;
	if ($opt->{depth} eq 'recursive') {
		$occ_check = {};
	} elsif ($opt->{depth}) {
		$occ_check = $opt->{depth};
	}
	return $this->construct_object_from_searchres($sth, $occ_check);
}
sub get_object_by_query {
	my($this, $sql) = @_;
	my $sth = $this->{db}->prepare($sql);
	if (! $sth->execute) {
		Carp::confess "SQL: $sql\n";
	}
#	$sth = $sql;
	if ($opt->{count}) {
		my($count) = $sth->fetchrow_array;
		return $count;
	}
	my $occ_check = 0;
	if ($opt->{depth} eq 'recursive') {
		$occ_check = {};
	} elsif ($opt->{depth}) {
		$occ_check = $opt->{depth};
	}
	return $this->construct_object_from_searchres($sth, $occ_check);
}
sub get_object_id {
	my($this, $key) = @_;
	my @obj = $this->get_object($key);
	return -1 if (! @obj);
	return $obj[0]->get_identifier;
}
sub difference {
	## for update procedure
	my($this, $condition) = @_;
	return if (! $this->{work_tablename});

	my(@primkeys) = $this->{schema}->primary_key;
	my $query = "select * from $this->{tablename} t1, " .
		"$this->{work_tablename} t2 where ";
	my $flag = 0;
	foreach $key (@primkeys) {
		$query .= " and" if ($flag);
		$query .= " t1.$key = t2.$key";
		$flag++;
	}
	$flag = 0;
	$query .= " and not (";
	foreach $attr ($this->{schema}->get_attributes(
			{for_update=>1,  ignore_ref=>1})) {
		$query .= " and" if ($flag);
		$query .= " t1.$attr = t2.$attr";
		$flag++;
	}
	$query .= ")";
	$condition =~ s/(\w+)\s*=/t1.$1=/g;
	$query .= " and ( $condition )" if ($condition);
	my $sth = $this->{db}->prepare($query);
	$sth->execute;
	return $sth;
}
sub get_composite_object {
	my($this, $table, $restriction) = @_;
	my @path = $this->{schema}->
			find_composite_path($table);
	my(@tables);
	foreach my $cl (@path) {
		push(@tables, MBGD::DBTable->new($this->{db}, $cl));
	}
	my $projection = "$table.*";
	return $this->create_join_query(\@tables, $restriction, $projection);
}
sub create_join_query {
	my($this, $tables, $restriction, $projection) = @_;

	my @tables;
	if (ref($tables) eq 'ARRAY') {
		@tables = @{$tables};
	} else {
		@tables = ($tables);
	}

	my @schemas = ($this->{schema});
	foreach $t (@tables) {
		 push(@schemas, $t->{schema});
	}

	my @joinfields = $this->{schema}->get_join_field(@schemas);
	return if (! @joinfields);

	my @tablenames = ($this->{tablename});
	foreach $t (@tables) {
		 push(@tablenames, $t->{tablename});
	}
	$projection = '*' if (! $projection);

	my $query = "select $projection from " . join(",", @tablenames) .
		" where " . join(' and ', @joinfields);
	$query .= " and $restriction" if ($restriction);
	return $query;
}
sub join {
	my($this, $tables, $restriction, $projection) = @_;
	my $query = $this->create_join_query($tables, $restriction, $projection);
	my $sth = $this->{db}->prepare($query);
	$sth->execute;
	while (@a = $sth->fetchrow_array) {
		print join(' ', @a),"\n";
	}
}
sub disable_keys {
	if ($this->{db}->{dbdrv} eq 'mysql') {
		$this->alter_options("disable keys");
	}
}
sub enable_keys {
	if ($this->{db}->{dbdrv} eq 'mysql') {
		$this->alter_options("enable keys");
	}
}
sub alter_options {
	my($this, $option) = @_;
	$sql = "alter table $this->{upd_tablename} $option";
}
sub load_from_file {
	my($this, $filename, $opt) = @_;
	my $sep = $opt->{sep};
	$sep = "\t" if (! $sep);
	if ($this->{db}->{dbdrv} eq 'mysql') {
		my $replace, $options, $prio;
		if ($opt->{ignore}) {
			$replace = 'ignore';
		} else {
			$replace = 'replace';
		}
		if ($opt->{prio} =~ /^(low priority|concurrent)/) {
			$prio = $opt->{prio};
		}
		if ($opt->{ignore_lines}) {
			$options .= "ignore $opt->{ignore_lines} lines ";
		}
		my $attrs = join(',',
			$this->{schema}->get_attributes({for_insert=>1}));

		my $sql = "load data local $prio infile '$filename' $replace " .
					"into table $this->{upd_tablename} " .
					$options .
					"($attrs) ";

if ($main::DEBUG) {
		print STDERR "SQL: $sql\n";
} else {
		$this->{db}->do($sql);
}
	} else {
		$this->prepare_for_insert;
		open(F, $filename) || die;
		while (<F>) {
			@F = split(/$sep/);
			$sth->execute(@F);
		}
		close(F);
	}
}
sub flush {
	my($this) = @_;
	$this->{db}->flush_table($this->{tablename});
}
sub assign_cache {
	my($this, $cachename) = @_;
	$this->{db}->assign_cache($this->{tablename}, $cachename);
}

if (__FILE__ eq $0) {
print __FILE__, "\n";

use MBGD::DB;
use MBGD::DBTable;
#$db = MBGD::DB->new("dbi:Pg:dbname=mbgd");
$db = MBGD::DB->new;

$a = MBGD::DBTable->new($db, 'Genome');
$b = MBGD::DBTable->new($db, 'Chromosome');
$c = MBGD::DBTable->new($db, 'DNASeq');
$d = MBGD::DBTable->new($db, 'Gene');
$e = MBGD::DBTable->new($db, 'ProteinSeq');
$f = MBGD::DBTable->new($db, 'GeneSeq');

#$s = $a->get_composite_object('gene', "genome.sp='mge'");
#print "$s\n";
print $a->join([$b,$c,$d,$e], "genome.sp in ('hin')",
	"gene.sp, gene.name, proteinseq.seq");
}


1;
