#include <algorithm>
#include <iostream>
#include <string>
#include <cstring>
#include <vector>
#include <iterator>
#include <ext/hash_set>
#include <set>
#include "Tree.h"
#include "HomData.h"
#include "DistMat.h"
#include "Hcluster.h"
#include "ClusterTree.h"
#include "ClusterInfo.h"
#include "ClusterAssignment.h"
#include "Options.h"
#include "SpecSet.h"
#include "SpecTree.h"
#include "TaxMap.h"
//#include "gc_detect_leaks.h"
#include "DEBUG.h"

const string MERGETREE_VERSION = "0.9.10b";

/*
#include "DetectMemoryLeaks.h"
#define new new(__FILE__, __LINE__)
*/


using namespace std;
using namespace __gnu_cxx;

char *treefile, *homfile, *taxfile, *assignfile;
vector<string> homfiles;
vector<string> homdir;
char printmat_mode;

int DEBUG::verbose_flag = 0;
int DEBUG::debug_flag = 0;

/*
double Hcluster::missDist = 260;
double Hcluster::missScore = 55;
*/

static double cutoffRatio = 0.95;

void usage() {
	cerr << "MergeTree ver. " << MERGETREE_VERSION << endl;
	cerr << "Usage: mergetree treefile [homfiles ...]" << endl;
}
void help() {
	usage();
	cerr << "\
    -S     use similarity as a measure of relatedness [on]\n\
    -d     use distance (or disimilarity) as a measure of relatedness\n\
    -c#    cutoff score/distance (can also be spcified as -S# or -d#) [60]\n\
    -m#    score/distance for missing relationships (m<c)\n\
    -p#    ratio of phylogenetic pattern overlap for tree cutting [0.5]\n\
    -o#    output format (default: 1:Tree)\n\
           1:Tree 11:TreeDetail\n\
    -OoutputScore=#        output score/distance at the root of each cluster\n\
    -Ometa=sp1,..          the specified genomes are treated as metagenomes\n\
    -OtaxMapOut=#          output taxonomy mapping of metagenomic data\n\
    -OtaxMapSpec=sp1,..    target species for taxonomy mapping [=meta]\n\
";
}
void getargs(int argc, char ** argv, Options *opt) {
	 int fn = 0;
	for (int i = 1; i < argc; i++) {
		char *p = argv[i];
		if (*argv[i] == '-') {
			switch (*++p) {
			case 'S':
				opt->simtype = 1;
				if (*++p) {
					opt->cutoff = atof(p);
				}
				break;
			case 'd':
				opt->simtype = 0;
				if (*++p) {
					opt->cutoff = atof(p);
				}
				break;
			case 'c':
				opt->cutoff = atof(++p);
				break;
			case 'm':
				opt->missdist = atof(++p);
				break;
			case 'p':
				opt->phylocutratio = atof(++p);
				break;
			case 't':
				taxfile = ++p;
				break;
			case 'h':
				help();
				exit(0);
			case 'o':
				opt->outputStyle = atoi(++p);
				break;
			case 'O':
			case '-':
				++p;
				if (strncmp(p, "taxMapOut", 9) == 0) {
					opt->taxMapOut = 1;
					if (p[9] == '=') {
						opt->taxMapOut = 2;
						opt->taxinfo_filename = &p[10];
					}
				} else if (strncmp(p, "taxMapSpec=", 11) == 0) {
					if (! opt->taxMapOut) {
						opt->taxMapOut = 1;
					}
					opt->taxMapSpecList = p + 11;
				} else if (strncmp(p, "meta=", 5) == 0) {
					opt->metaSpecList = p + 5;
				} else if (strncmp(p, "metaPref=", 9) == 0) {
					opt->metaSpecPref = p + 9;
				} else if (strncmp(p, "partial=", 8) == 0) {
					opt->partialSpecList = p + 8;
				} else if (strncmp(p, "noReplaceSpTreeLeafName", 23) == 0) {
					opt->noReplaceSpTreeLeafName = true;
				} else if (strncmp(p, "outputScore", 11) == 0) {
					opt->outputScore = 1;
					if (p[11] == '=') {
						opt->scoreout_filename = &p[12];
					}
				} else if (strncmp(p, "entireGroupAverage", 18) == 0) {
					opt->entireGroupAverage = 1;
				} else if (strncmp(p, "bestRatio=", 10) == 0) {
					opt->bestRatio4Assign = atof(&p[10]);
				} else if (strncmp(p, "checkHit=", 9) == 0) {
					opt->checkHit4Assign = atoi(&p[9]);
				} else if (strncmp(p, "sequential", 10) == 0) {
					opt->sequential = 1;
					opt->printAssign = 1;
					opt->assignOnly = 1;
				} else if (strncmp(p, "readEachCluster", 15) == 0) {
					opt->readEachCluster = 1;
				} else if (strncmp(p, "printAssign", 11) == 0) {
					opt->printAssign = 1;
				} else if (strncmp(p, "assignOnly", 10) == 0) {
					opt->printAssign = 1;
					opt->assignOnly = 1;
				} else if (strncmp(p, "assignfile=", 11) == 0) {
					assignfile = &p[11];
				} else if (strncmp(p, "corrDist", 8) == 0) {
					if (p[8] == '=') {
						opt->corrDist = atoi(&p[9]);
					} else {
						opt->corrDist = 3;
					}
				} else if (strncmp(p, "UM_tolerance=", 13) == 0) {
					opt->um_tolerance = atof(&p[13]);
				} else if (strncmp(argv[i], "--printmat", 10) == 0) {
					printmat_mode = 1;
					if (argv[i][10] == '=' &&
							isdigit(argv[i][11])) {
						printmat_mode = (argv[i][11] - '0');
					}
				} else if (strncmp(argv[i], "--verbose", 9) == 0) {
					DEBUG::verbose_flag = 1;
				} else if (strncmp(argv[i], "--debug", 7) == 0) {
					if (argv[i][7] == '=' && isdigit(argv[i][8])) {
						DEBUG::debug_flag = atoi(&argv[i][8]);;
					} else {
						DEBUG::debug_flag = 1;
					}
				}
				break;
			}
		} else if (fn == 0) {
			treefile = argv[i];
			fn++;
		} else if (fn == 1) {
			homfile = argv[i];
			homfiles.push_back(homfile);
		}
	}
}
void setOptValues(Options *opt) {
	if (opt->simtype == 0) {
		SimValue::setReprType("dist");
	} else {
		SimValue::setReprType("score");
	}
	if (opt->missdist) {
		if (SimValue::getReprType() == 's') {
			Hcluster::missScore = opt->missdist;
		} else {
			Hcluster::missDist = opt->missdist;
		}
	} else {
		if (SimValue::getReprType() == 's') {
			opt->missdist = Hcluster::missScore = opt->cutoff * cutoffRatio;
		} else {
			opt->missdist = Hcluster::missDist = opt->cutoff / cutoffRatio;
		}
	}
	if (! opt->min_alilen) {
		opt->min_alilen = 250;
	}
}
Domain *getThisDom(HomData *hdata, Domain *dom) {
	if (hdata->dom1 != NULL && hdata->dom1->getName() == dom->getName()) {
		return(hdata->dom1);
	} else if (hdata->dom2 != NULL && hdata->dom2->getName() == dom->getName()) {
		return(hdata->dom2);
	} else {
		return(NULL);
	}
}
Domain *getOtherDom(HomData *hdata, Domain *dom) {
	if (hdata->dom1 != NULL && hdata->dom1->getName() == dom->getName()) {
		return(hdata->dom2);
	} else if (hdata->dom2 != NULL && hdata->dom2->getName() == dom->getName()) {
		return(hdata->dom1);
	} else {
		return(NULL);
	}
}
void resetAll() {
	PairDist::reset();
	ClusterNode::reset();
	DomainPool::resetToMarkedPos();
}
void outputHeader(Options *opt) {
	if (opt->outputStyle == ClusterTree::TREEDETAIL) {
		cout << "# simtype=" << opt->simtype << endl;
		cout << "# cutoff=" << opt->cutoff << endl;
		/* similarity */
/*
		if (opt->simtype == 1) {
			cout << "# missdist=" << opt->missscore << endl;
		} else {
			cout << "# missdist=" << opt->missdist << endl;
		}
*/
		cout << "# missdist=" << opt->missdist << endl;
		cout << "# phylocutratio=" << opt->phylocutratio << endl;
		cout << "# distscale=" << opt->distscale << endl;
	}
}
int main(int argc, char **argv) {
	ReadHomData rhom;
	ReadTree rtree;
	HomDataSet *homDataRead = NULL;
	ClusterInfo* clInfo;
	DistTreeList* treeList;
	Options opt;
	SpecTree *spTree = NULL;
	TaxonomyMapping *taxMap = NULL;
	SpecSetInstances* specSetInst = SpecSetInstances::getInstance();
	int spnum_in_sptree = 0;
	bool noUpdateMode = false;
	SpecSet addedSpSet;
	FILE *scoreFp = NULL;

	if (argc < 2) {
		usage();
		exit(1);
	}
	SimValue::setReprType("score");

	getargs(argc, argv, &opt);

	if (rtree.openFile(treefile) < 0) {
		cerr << "failed to open treefile" << endl;
		exit(1);
	}
	if (taxfile) {
		SpecTreeRead spRead;
		spTree = spRead.readSPfile(taxfile);
		if (opt.noReplaceSpTreeLeafName) {
			spTree->unset_replaceLeafName();
		}
//		spTree->print();
		spnum_in_sptree = SpecSet::getSPnum();
	}

	if (opt.metaSpecList || opt.metaSpecPref ||
			opt.taxMapSpecList || opt.partialSpecList) {
		specSetInst->setSpecialSpecies(opt.metaSpecPref, opt.metaSpecList, opt.taxMapSpecList, opt.partialSpecList, spnum_in_sptree);
	}
	if (opt.taxMapOut && ! specSetInst->taxQueryMode()) {
		cerr << "No species for taxonomy mapping: taxmap mode is canceled\n";
		opt.taxMapOut = 0;
	}
	if (opt.taxMapOut) {
		FILE *ofp = NULL;
		if (opt.taxinfo_filename) {
			ofp = fopen(opt.taxinfo_filename, "w");
			if (ofp == NULL) {
				cerr << "Can't open taxmap output file\n";
			}
		} else {
			ofp = stdout;
		}
		if (spTree == NULL) {
			cerr << "Fatal: taxonomy file is not specified\n";
			exit(1);
		}
		taxMap = new TaxonomyMapping(spTree, specSetInst, opt.bestRatio4Assign, ofp);
	}
	if (opt.scoreout_filename) {
		if (! (scoreFp = fopen(opt.scoreout_filename, "w")) ) {
			cerr << "Can't open scoreout file\n";
			exit(1);
		} 
	}

	/* read the original options recorded in the tree file */
	rtree.readOptions(&opt);

	setOptValues(&opt);

	/* read clustering trees from file */
	if (DEBUG::verbose_flag) {
		cerr << "reading tree data\n";
	}
	AllClustersInfo *clustInfo = AllClustersInfo::readClusters(&rtree);
	/* dominfo is extracted from clustInfo */
	DomInfo *domInfo = clustInfo->getDomInfo();
	ClusterAssignment *clustAssign = NULL;

	rhom.setInfoAll(domInfo, &opt, specSetInst);

	if (homfiles.size() == 0) {
		noUpdateMode = true;
	} else {
		if (assignfile) {
			if (DEBUG::verbose_flag) {
				cerr << "reading assign file\n";
			}
			clustAssign = ReadAssignment::readAssign(assignfile);
			if (! clustAssign) {
				cerr << "exit" << endl;
				exit(1);
			}
			if (! opt.readEachCluster) {
				if (DEBUG::verbose_flag) {
					cerr << "reading homology data\n";
				}
				if (rhom.openFiles(homfiles) < 0) {
					cerr << "failed to open homfile" << endl;
					exit(1);
				}
				if (homDataRead) {
					homDataRead->clearHomData();
				}
				homDataRead = rhom.readAllData( homDataRead );
				addedSpSet.setSpecSet( homDataRead->getAddedSpSet() );
				if (DEBUG::verbose_flag) {
					cerr << "Done.\n";
				}
			}
		} else {
			if (rhom.openFiles(homfiles) < 0) {
				cerr << "failed to open homfile" << endl;
				exit(1);
			}
//			rhom.setInfoAll(domInfo, &opt, specSetInst);

			if (DEBUG::verbose_flag) {
				cerr << "reading homology data\n";
			}
			if (opt.entireGroupAverage) {
				ClusterAssignment::entireGroupAverageMode(Hcluster::missScore, Hcluster::missDist);
			} else if (opt.bestRatio4Assign >= 0 || opt.checkHit4Assign >= 0 || opt.cutoff >= 0) {
				ClusterAssignment::setParam(opt.bestRatio4Assign, opt.checkHit4Assign, opt.cutoff);
			}

			if (opt.sequential) {
				HomDataSet homDataRead_each;
				ClustAssignList *clAssignList;
				clustAssign = new ClusterAssignment();
				while (rhom.readEachData(&homDataRead_each)) {
					if (homDataRead_each.getDataSize()) {
						GeneData *gdata = homDataRead_each.getGeneDataByName( rhom.getCurrGeneName() );
						clAssignList = ClusterAssignment::assignGeneToCluster(&homDataRead_each, gdata, NULL, clustInfo);
						clustAssign->addHitList(clAssignList);
					}
				}
			} else {
				homDataRead = rhom.readAllData();
				addedSpSet.setSpecSet( homDataRead->getAddedSpSet() );

				if (DEBUG::verbose_flag) {
					cerr << "assigninig clusters to each gene\n";
				}
				clustAssign = ClusterAssignment::assignGeneToClusterAll(homDataRead, clustInfo);
			}
			if (opt.printAssign) {
				clustAssign->printAssignment();
			}
			if (opt.assignOnly) {
				exit(0);
			}
		}
	}
	if (opt.taxMapOut) {
		/* add species not found in SpecTree into specSetInst->ignore_specSet */
		specSetInst->addUndefinedSpecies(spnum_in_sptree);
	}

	if (printmat_mode == 2) {
		domInfo->printDomains();
		homDataRead->printAllGenes();
		cout << "//" << endl;
		homDataRead->printAllHom();
	}
	if (DEBUG::verbose_flag) {
		cerr << "cluster_num " << clustInfo->getSize() << endl;
	}
	
//	DomainPool::markCurrent();

	outputHeader(&opt);

	string curr_homName;

	if (opt.readEachCluster) {
		/* reading gene data */
		if (DEBUG::verbose_flag) {
			cerr << "reading gene data" << endl;
		}
		if (rhom.openFile(homfiles[0]) < 0) {
			cerr << "failed to open homfile" << endl;
			exit(1);
		}
		// read gene data
		homDataRead = rhom.readAllData();
		
	}

	DomainPool::markCurrent();

	int clustnum = clustInfo->getSize();
	for (int i = 0; i < clustnum; i++) {

		int clustid = clustInfo->getClustID(i);
		if (DEBUG::verbose_flag) {
			cerr << "cluster " << clustid << endl;
		}
		clInfo = clustInfo->getClusterInfo(i);
		treeList = clInfo->getTreeList();

		if (DEBUG::verbose_flag) {
			list<TreeNode*> all_leaves = treeList->getAllLeaves();
			cerr << "size: " << all_leaves.size() << endl;
		}

		if (opt.readEachCluster) {
			vector<string> homdata_files;
			char int_buf[50];
			for (unsigned int j = 1; j < homfiles.size(); j++) {
				sprintf(int_buf,  "%s/%d", homfiles[j].c_str(), clustid);
				homdata_files.push_back(int_buf);
			}
			homDataRead->clearHomData();
			if (rhom.openFiles(homdata_files) < 0) {
/*
				cerr << "failed to open homfile" << endl;
				exit(1);
*/
			} else {
				if (DEBUG::verbose_flag) {
					cerr << "Reading homdata\n";
				}
				homDataRead = rhom.readAllData(homDataRead);

//cerr << "Done\n";
/*
sleep(15);
cerr << "clear\n";
homDataRead->clearHomData();
resetAll();
cerr << "Done\n";
sleep(60);
*/
			}
		}

//detectMemoryLeaksStart(std::cout);

		list<ClustAssignData*> *assList;
/*
		map<string, Domain*> hitGeneMap;
*/

		if (! noUpdateMode) {

			/* list of hits found in new genomes */

			assList = clustAssign->findAssignment(clustid);
			if (assList == NULL) continue;

/*
			for ( list<ClustAssignData*>::iterator p = assList->begin();
				p != assList->end(); p++) {
				Domain *dom = (*p)->getDomain();
				hitGeneMap.insert( map<string,Domain*>::value_type(dom->getName(),dom) );
				if (DEBUG::debug_flag) {
					cout << "Assigned: " << clustid << ": " << dom->getName() << endl;
				}
			}
			delete assList;
*/
		}

		if (printmat_mode == 2) {
			vector<PairDist *>* distSet = treeList->convDistMat();
			for (unsigned int j = 0; j < distSet->size(); j++) {
				(*distSet)[j]->print();
			}
			delete distSet;
			continue;
		}

		string homName = clInfo->getUpperName();
		if (! homName.empty() && homName != curr_homName) {

			cout << "HomCluster " << homName;
			curr_homName = homName;
			HomClusterData *homclData = clustInfo->getHomClusterInfo(homName);
			if (opt.outputStyle == ClusterTree::TREEDETAIL) {
				cout << " : " << homclData->score << " " << homclData->dist;
			}
			if (opt.outputScore) {
				ClusterNode::outputScoreData(
					scoreFp, ("HomCluster "+homName).c_str(), homclData->score, homclData->dist);
			}
			cout << endl;
		}

	    int treenum = 0;
		/* multiple merged clusters */
	    for (list<DistTree*>::iterator pt = treeList->begin();  pt != treeList->end(); pt++) {
		DistTree *tree = *pt;
		DistMat *distMat = NULL;
//		ClusterNode::markCurrent();

		map<string, Domain*> hitGeneMap;

		if (! noUpdateMode) {



			for ( list<ClustAssignData*>::iterator p = assList->begin(); p != assList->end(); p++ ) {
				if (! (*p)->check_subdata(treenum)) {
					/* the domain is not added to this sub tree because of a bad score */
					continue;
				}
				Domain *dom = (*p)->getDomain();
//cout << "CD:" << clustid << " " << *dom << endl;
				hitGeneMap.insert( map<string,Domain*>::value_type(dom->getName(),dom) );
				if (DEBUG::debug_flag) {
					cout << "Assigned: " << clustid << ": " << dom->getName() << endl;
				}
			}

			vector<PairDist *>* distSet = tree->convDistMat();


//			leaves = treeList->getAllLeaves();
			list<TreeNode*> &leaves = tree->getAllLeaves();
			HomSetMap homSetMap;
			map<string,Domain*> addedData;
			HomDataList homList;

			if (DEBUG::verbose_flag) {
				cerr << "Getting homdata\n";
			}
			for (list<TreeNode*>::iterator p = leaves.begin(); p != leaves.end(); p++) {
//				TreeNode *n = *p;
				Domain *origDom = (*p)->getDomain();
				Domain *dom = DomainPool::duplicate(origDom);
				dom->placeOn(origDom);

				homDataRead->getGeneHomologs(dom, &homList, true);
				if (! homList.empty()) {
				    for (HomDataList::iterator hp = homList.begin();
						hp != homList.end(); hp++) {

					Domain *otherDom = getOtherDom(*hp, dom);
//cout << "OtherDom:" << *otherDom << endl;

					if (otherDom->getClustID() &&
						otherDom->getClustID() != clustid) {
						continue;
					}
					if (otherDom == NULL) {
						cerr << "?????" << endl;
						break;
					}


					map<string,Domain*>::iterator p
					    = hitGeneMap.find(otherDom->getOrigName());
					if (p == hitGeneMap.end()) {
						continue;
					}
					Domain *hitDom = (*p).second;

					/* assign domnum */
					otherDom->placeOn(hitDom);

					homSetMap.add(*hp);
				
/*
					addedData[ otherDom->getName() ] = otherDom;
*/
					addedData[ otherDom->getName() ] = hitDom;
				    }
				}
			}

			if (DEBUG::verbose_flag) {
				cerr << "Getting homdata2\n";
			}
			/* check reverse direction and add homology within added orgs */
			for (map<string,Domain*>::iterator p = addedData.begin();
					p != addedData.end(); p++) {
				Domain *dom = p->second;
/*
Domain origdom;
dom->getOrigDomain(&origdom);
dom = &origdom;
*/
				homDataRead->getGeneHomologs(dom, &homList, true);
				for (HomDataList::iterator hp = homList.begin();
						hp != homList.end(); hp++) {
					Domain *thisDom = getThisDom(*hp, dom);
					Domain *otherDom = getOtherDom(*hp, dom);
					if (otherDom->getClustID() &&
						otherDom->getClustID() != clustid) {
						continue;
					}

					map<string,Domain*>::iterator p
						= hitGeneMap.find(thisDom->getName());
					if (p == hitGeneMap.end()) {
						continue;
					}
					Domain *hitDom = (*p).second;
//cout << "Hit2-dom1:" << *thisDom << " " << *hitDom << endl;

					/* assign domnum */
					thisDom->placeOn(hitDom);

					p = hitGeneMap.find(otherDom->getName());
					if (p == hitGeneMap.end()) {
						continue;
					}
					hitDom = (*p).second;
//cout << "Hit2-dom2:" << *otherDom << " " << *hitDom << endl;
//cout << "Hit2:" << **hp << endl;

					/* assign domnum */
					otherDom->placeOn(hitDom);

//cout << "Hit2-2:" << **hp << endl;
					homSetMap.add(*hp);

				/* homology within added organisms */
/*
					if (homDataRead->checkAddedSpec(
							otherDom->getSpec())) {
						if (hitGeneMap.find(otherDom->getName())
								== hitGeneMap.end()) {
							continue;
						}
						homSetMap.add(*hp);
					}
*/

/*
					if (addedData.find(otherDom->getName()) != addedData.end()) {
cout << "ADDED::" << otherDom->getName() << endl;
						homSetMap.add(*hp);
					}
*/
				}
			}

//cerr << "Creating pairdist\n";
			for (HomSetMap::iterator hp = homSetMap.begin();
					hp != homSetMap.end(); hp++) {
				PairDist *pdist = PairDist::create((*hp).second);
				distSet->push_back(pdist);
//cout <<"homdata:" << *((*hp).second) << endl;
//cout <<"pdist:" << *pdist << endl;
     			}

//cerr << "Creating distmat\n";
			distMat = new DistMat(distSet, SimValue::best, opt.distscale);
//cerr << "Done\n";

			if (opt.corrDist) {
				if (DEBUG::verbose_flag) {
					cerr << "Correcting distances..." << endl;
				}
				if (opt.um_tolerance > 0) {
					distMat->set_UMtolerance(opt.um_tolerance);
				}
				distMat->correctDistance(specSetInst, opt.corrDist);
				if (DEBUG::verbose_flag) {
					cerr << "Done" << endl;
				}
			}

			/** print matrix mode **/
			if (printmat_mode) {
				distMat->printMat();
				cout << "//\n";
				delete distMat;
				continue;
			}
			if (distMat->size() == 0) {
				// empty distmat
				if (treenum != 0) {
					cout << "--" << endl;
				}
				
				cout << "Cluster " << clInfo->getName() << endl;
				if (opt.outputStyle == ClusterTree::TREEDETAIL) {
					tree->printTreeDetail();
				} else if (opt.outputStyle == ClusterTree::NEWICK) {
					tree->printNewick(false);
				} else if (opt.outputStyle == ClusterTree::NEWICK_DIST) {
					tree->printNewick(true);
				} else {
					tree->printTree();
				}
				delete distMat;
				treenum++;
				continue;
			}
		}
//cerr << "OK2\n";
		list<ClusterNode*> *roots;
		if (! noUpdateMode) {
			if (SimValue::getReprType() == 's') {
				distMat->setDistCutoff(opt.missdist);
			} else {
				distMat->setDistCutoff(0, opt.missdist);
			}

//			distMat->sort();
			distMat->createIndices();

			Hcluster hclust(distMat, opt.cutoff, opt.missdist, opt.phylocutratio);
			hclust.setSpSet( &addedSpSet );
			hclust.execute();

			roots = ClusterNode::listRootNodes();
		} else {
			ClusterTree *clustTree = ClusterTree::convClusterTree(tree);
			clustTree->phyloCut(opt.phylocutratio);
			roots = ClusterNode::listRootNodes();
		}
		if (opt.taxMapOut) {
			/* add species not found in SpecTree into specSetInst->ignore_specSet */
			specSetInst->addUndefinedSpecies(spnum_in_sptree);
		}

		list<ClusterNode*>::iterator p = roots->begin();

		if (treenum == 0) {
			ClusterTree clustTree(*p);
			ostringstream header;
			header << "Cluster " << clInfo->getName();
			cout << header.str();
			if (opt.outputScore) {
				clustTree.outputScore(scoreFp, header.str().c_str());
			}
			cout << endl;
		} else {
			cout << "--" << endl;
		}

		int rnum = 0;
		while (p != roots->end()) {
			if (specFlagInclude(addedSpSet, (*p)->getSpSet())) {
//				cout << "OK\n";
			} else {
				ClusterTree clustTree(*p);
				if (rnum > 0) {
					ostringstream header;
					header << "Cluster " << clInfo->getName() << "." << rnum;
					cout << header.str();
					if (opt.outputScore) {
						clustTree.outputScore(scoreFp, header.str().c_str());
					}
					cout << endl;
				}
				clustTree.output(opt.outputStyle);
				if (opt.taxMapOut) {
					taxMap->mapTaxInfo_Cinfo(roots);
				}
				rnum++;
			}
			p++;
		}

		delete roots;
		delete distMat;
//detectMemoryLeaksEnd(std::cout);
		treenum++;
		ClusterNode::reset();
	    }

	    if (opt.taxMapOut) {
		taxMap->mapTaxInfo_output( clInfo->getName() );
	    }
//	ClusterNode::reset();
	PairDist::reset();
	    delete assList;
	    if (opt.readEachCluster) {
	   	 homDataRead->clearHomData();
	    }
//	    resetAll();
	DomainPool::resetToMarkedPos();
	    cout << "//" << endl;
	}
}

