#!/usr/bin/perl

package MBGD::WWW::SeqRegMap;
use MBGD;
use RECOG::RecogCommon;
use Alignment;
require 'libMBGDImg.pl';

$ImageWidth = 800;
$ImageHeight = 300;
$Xmargin = $Ymargin = 10;
$TitleMargin = 100;
$LineLength = $ImageWidth - $Xmargin * 2 - $TitleMargin;

$Font = 'gdMediumBoldFont';
@Colors = ("255_0_0","0_255_0","0_0_255","255_255_0","255_0_255","0_255_255",
   "180_0_0","0_180_0","0_0_180","180_180_0","180_0_180","0_180_180",
   "255_180_0","255_0_180","0_255_180","180_255_0","180_0_255","0_180_255");
@HomColors = (
	{ident=>75, color=>"0_0_150"},
	{ident=>50, color=>"0_100_255"},
	{ident=>25, color=>"0_255_255"},
	{ident=>0, color=>"200_240_255"},
);

$BarHeight = 14;
$BarHeight2 = 6;
$BarSkip2 = 12;
$BarSkip = 20;
$TickHeight1 = 5;
$TickHeight2 = 10;

$ColorStrong = "255_0_0";
$ColorMedium = "255_100_100";
$ColorWeak   = "255_160_160";
$ColorCOG = "0_0_255";
$ColorDom = "255_0_0";
$Background = "220_230_230";

$ColorBlack = "0_0_0";
$ColorGray120 = "120_120_120";
$ColorGray150 = "150_150_150";
$ColorGray200 = "200_200_200";
$ColorRed = "255_0_0";
$ColorPink = "255_180_180";
$ColorBlue = "0_0_255";

$DomURL = "/htbin/hcluster?prog=hcluster";
$MotURL = "/htbin/searchGeneByMotif.pl";
$SimURL = "/htbin/align";
$GeneURL = "/htbin/MBGD_gene_info_frame.pl";

$TRUNC_ALLOWED = 40;

sub new {
	my($class, $name, %opt) = @_;
	my($this) = {};
	my($gene);
	$this->{name} = $name;
	if ($name) {
	 	$gene = MBGD::Gene->get($name);
		$this->{seqlen} = $gene->length;
	}
	$this->{commands} = [];
	$this->{ypos} = $Ymargin;
	if ($opt{uInfo}) {
		$this->{uInfo} = $opt{uInfo};
		$this->{tabid} = $this->{uInfo}->loadTableIDs('current');
	}
    	if ($opt{tabid}) {
		$this->{tabid} = $opt{tabid};
        }
	if ($opt{clustid}) {
		$this->{clustid} = $opt{clustid};
	}
	$this->{ImageWidth} = $opt{ImageWidth} || $ImageWidth;
	$this->{ImageHeight} = $opt{ImageHeight} || $ImageHeight;
	$this->{LineLength} = $opt{LineLength} || $LineLength;
	$this->{Xmargin} = $opt{Xmargin} || $Xmargin;
	$this->{Ymargin} = $opt{Ymargin} || $Ymargin;
	$this->{TitleMargin} = $opt{TitleMargin} || $TitleMargin;

	push(@{$this->{commands}}, "Background $Background");
	bless $this, $class;
}
sub drawAllHomology {
	my($this, %opt) = @_;
	my $display = 'homlegend,scale,domain,homology';
	$this->drawAll($display, %opt);
}
sub drawAllMotif {
	my($this, %opt) = @_;
	my $display = 'scale,domain,motif';
	$this->drawAll($display, %opt);
}
sub drawAllMultAlign {
	my($this, %opt) = @_;
	my $display = "multalign";
	$this->drawAll($display, %opt);
}
sub drawAll {
	my($this, $display, %opt) = @_;
	$this->setTitleMargin;
	foreach $d (split(/,/, $display)) {
		if ($d eq 'scale') {
			$this->drawScale(%opt);
		} elsif ($d eq 'domain') {
			$this->drawDomain(%opt);
		} elsif ($d eq 'motif') {
			$this->drawDomainMotifs(%opt);
		} elsif ($d eq 'homlegend') {
			$this->drawHomLegend(%opt);
		} elsif ($d eq 'homology') {
			$this->drawHomology(%opt);
		} elsif ($d eq 'multalign') {
			$this->drawMultAlign(%opt);
		}
	}
}

sub setTitleMargin {
	my($this, $length) = @_;
	if (! $length) {
		$length = length($this->{name});
	}
	$length = $length * 7 + 10;
	$this->{TitleMargin} = $length if ($length > $this->{TitleMargin});
	$this->{LineLength} = $this->{ImageWidth} - $this->{Xmargin} * 2 - $this->{TitleMargin};
}
sub drawHomLegend {
	my($this, %opt) = @_;
	my($y1) = $this->{ypos} - int($BarHeight2/2);
	my($y2) = $y1 + $BarHeight2;
	my($y0) = int(($y1+$y2)/2);
	my($xL,$xR) = $this->getLineCoord(0,$this->{seqlen},$this->{seqlen});
	my($xspan) = int(($xR - $xL) / @HomColors * 0.5);
	my($i);
	my($x1, $x2, $x0);
	$x1 = int(($xL + $xR) / 2);
	if ($opt{mode} eq 'clmap') {
		# do nothing
	} else {
		foreach $hc (@HomColors) {
			$x2 = $x1 + int($xspan/1.8);
#			$x0 = int(($x1+$x2)/2);          # fill ǥ顼ȤʤȤ
			$x0 = $x1+1;
			my $xstr = int(($x2+$xspan/8));
			my $ystr = int($y1 - $BarHeight2/2);
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $ColorBlack",
				"fill $x0 $y0 $hc->{color}",
				qq{string $Font $xstr $ystr "$hc->{ident}%" black});
			$x1 += $xspan;
		}
	}
	$this->{ypos} += int($BarSkip * 1.2);
}
sub drawScale {
	my($this, %opt) = @_;
	my($x1,$x2,$x0);
	$this->{ypos} += $BarHeight;
	my($y0) = $this->{ypos};
	my($yh1) = $y0 - $TickHeight1;
	my($yh2) = $y0 - $TickHeight2;
	my($x11,$ystr);
	if ($opt{mode} eq 'clmap') {
		# do nothing
	} else {
		($x1,$x2) = $this->getLineCoord(0,$this->{seqlen},$this->{seqlen});
		push(@{$this->{commands}}, "line $x1 $y0 $x2 $y0 black");
		if ($this->{seqlen} < 200) {
			$tickInt = 10;
		} else {
			$tickInt = 50;
		}
		for (my $i = 0; $i < $this->{seqlen}; $i += $tickInt) {
			$x0 = $this->getCoord($i, $this->{seqlen});
			if ($i % 100 == 0) {
				$yh = $yh2;
			} else {
				$yh = $yh1;
			}
			push(@{$this->{commands}}, "line $x0 $y0 $x0 $yh black");
			$ystr = $yh-10;
			$x11 = $x0-length("$i")*3;
			$flag = 1;
			if ($this->{seqlen} > 1500) {
				$flag = 0 if ($i % 500 != 0);
			} elsif ($this->{seqlen} < 200) {
				$flag = 0 if ($i % 50 != 0);
			} else {
				$flag = 0 if ($i % 100 != 0);
			}
			if ($flag) {
				push(@{$this->{commands}},
					qq{string $Font $x11 $ystr "$i" black});
			}
		}
	}
	$this->{ypos} += $BarHeight;
}
sub drawDomain {
	my($this, %opt) = @_;
	my($i);
	my($y1,$y2);
	my $tabid = $this->{tabid};
	my($cltab_db) = MBGD::ClustTab::DB->new($tabid);
	@Dom = $cltab_db->getDomains($this->{name});

	$y1 = $this->{ypos};
	$y2 = $y1 + $BarHeight;
	$y0 = ($y1+$y2)/2;
	push(@{$this->{commands}},
		qq{string $Font $this->{Xmargin} $y1 "$this->{name}" black});
	foreach $d (@Dom) {
		my($x1,$x2) = $this->getLineCoord($d->{from},$d->{to},$this->{seqlen});
		my $x11 = $x1 + 10;
		my $x0 = ($x1+$x2)/2;
		$x0 = $x1+1 if ($x1 == $x0);
		if (! defined $DomNum{$d->{clustid}}) {
			$DomNum{$d->{clustid}} = $i++;
		}
		my $domid = $DomNum{$d->{clustid}};
		if ($opt{mode} eq 'clmap') {
			my $domurl = $DomURL . qq{&clustid=O$d->{clustid}};
			$domurl .= "&tabid=$tabid" if ($tabid);
			push(@{$this->{clmap}},
			    qq{<AREA SHAPE="RECT" COORDS="$x1 $y1 $x2 $y2" } .
			    	qq{HREF="$domurl" TARGET=_top>});
		} else {
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $ColorBlack",
				"fill $x0 $y0 $Colors[$domid]",
				qq{string $Font $x11 $y1 "O$d->{clustid}" white});
		}
	}
	$this->{ypos} += $BarSkip2;
}

sub drawDomainMotifs {
	my($this, %opt) = @_;
	my(@Mot) = MBGD::ProtMotif->get($this->{name});
	my($MotColor);
	my($x1,$x2,$x0,$x11);
	my($y1,$y2,$y0);
	my(@motList, %motInfo);
	$y1 = $this->{ypos};
	foreach $mot (@Mot) {
		push(@motList, "$mot->{motlib}:$mot->{motid}");
	}
	foreach $mot (MBGD::Motif->get(\@motList)) {
		$motInfo{"$mot->{motlib}:$mot->{motid}"} = $mot;
	}
	foreach $mot (sort {$a->{eval}<=>$b->{eval}} @Mot) {
		my $motname = "$mot->{motlib}:$mot->{motid}";
		my $motlength = $motInfo{$motname}->{length};
		my($partialL, $partialR);
		if ($mot->{from2} > $TRUNC_ALLOWED) {
			$partialL = 1;
		}
		if ($mot->{to2} < $motlength - $TRUNC_ALLOWED + 1) {
			$partialR = 1;
		}

		my($x1,$x2) = $this->getLineCoord($mot->{from1},$mot->{to1}, $this->{seqlen});
		$this->{ypos} += $BarSkip;
		$x11 = $x1 + 10;
		$y1 = $this->{ypos};
		$y2 = $y1+$BarHeight;
#		$x0 = ($x1+$x2)/2;
		$x0 = $x1+1;
		$y0 = ($y1+$y2)/2;
		($bgcol,$fgcol) = &makeDomColor($mot->{motlib}, $mot->{eval});

		if ($opt{mode} eq 'clmap') {
			my $moturl = $MotURL . "?motlib=$mot->{motlib}&motid=$mot->{motid}&count=1";
			push(@{$this->{clmap}},
			    qq{<AREA SHAPE="RECT" COORDS="$x1 $y1 $x2 $y2"} .
			    	qq{HREF="$moturl" TARGET=_top>});
		} else {
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $ColorBlack",
				"fill $x0 $y0 $bgcol");
			if ($partialL) {
				push(@{$this->{commands}},
					&makeBreak($x1, $y1, $y2, 'L'));
			}
			if ($partialR) {
				push(@{$this->{commands}},
					&makeBreak($x2, $y1, $y2, 'R'));
			}
			push(@{$this->{commands}},
				"string $Font $x11 $y1 " .
				qq{"$mot->{motlib}:$mot->{motname}" $fgcol});
		}
	}
}
sub drawHomology {
	my($this, %opt) = @_;

	my(%Param);
    if ($this->{uInfo}) {
        %Param = $this->{uInfo}->getHomolParamHash;
    }

    #
    my($species) = $Param{'species'};
    if ($MBGD::WWW::SeqRegMap::MODE =~ /^RECOG$/i) {
        $species = RECOG::RecogCommon::getSpeciesByTabid($this->{tabid});
    }
    else {
        my($sta) = main::MBGD_SpecTableIsDefaultSpecies($species);
        if ($sta != 0) {
            my(%spec_hash);
            foreach my$sp (split(/[\,\|]/, $species)) {
                $spec_hash{"$sp"} = 1;
            }
            my($sp, $name) = split(/\:/, $this->{'name'});
            $spec_hash{"$sp"} = 1;

            $species = join(',', keys(%spec_hash));
        }
    }
###	my(@splist) = split(/[\,\|]/, $species);
###	my(@Hom) = MBGD::Homology->select({genes=>$this->{name}, species=>\@splist, with_length=>1});

	my(@Hom) = &getHomologyData(genes=>$this->{name}, species=>$species, with_length=>1);

#	my($maxlen);
#	foreach $hom (@Hom) {
#		my($len) = length($hom->{spname2});
#		$maxlen = $len if ($len > $maxlen);
#	}
#	$TitleMargin = $maxlen * 7 + 10;

	my $HomColor = "255_255_120";
	my($cnt);

	$this->{ypos} += $BarSkip;
	foreach $hom (sort {$b->{score}<=>$a->{score}} @Hom) {
#print STDERR "hh>>>$hom->{spname1},$hom->{fro1},$hom->{to1};$hom->{spname2},$hom->{from2},$hom->{to2}\n";
		my($partialL, $partialR);
		$this->{ypos} += $BarSkip2;
 		my($y1,$y2) = ($this->{ypos}, $this->{ypos}+$BarHeight2);
		my($x1,$x2) = $this->getLineCoord(
			$hom->{from1},$hom->{to1}, $this->{seqlen});
#		my $x0 = ($x1+$x2)/2;
		my $x0 = $x1+1;
		my $y0 = ($y1+$y2)/2;
		if ($hom->{from2} > $TRUNC_ALLOWED) {
			$partialL = 1;
		}
		if ($hom->{to2} < $hom->{len2} - $TRUNC_ALLOWED + 1) {
			$partialR = 1;
		}

		if ($opt{mode} eq 'clmap') {
			my $simurl = $SimURL . qq{?seq1=$hom->{spname1}&seq2=$hom->{spname2}};
			my $geneurl = $GeneURL . qq{?name=$hom->{spname2}};
            if ($MBGD::WWW::SeqRegMap::MODE =~ /^RECOG$/i) {
                $simurl  .= "&tabid=$this->{'tabid'}";
                $geneurl .= "&tabid=$this->{'tabid'}";
            }
			my $xt1 = $this->{Xmargin}, $xt2 = $this->{Xmargin} + $this->{TitleMargin};
			push(@{$this->{clmap}},
			    qq{<AREA SHAPE="RECT" COORDS="$xt1 $y1 $xt2 $y2" } .
			    	qq{HREF="$geneurl" TARGET=_top>});
			push(@{$this->{clmap}},
			    qq{<AREA SHAPE="RECT" COORDS="$x1 $y1 $x2 $y2" } .
			    	qq{HREF="$simurl" TARGET=_top>});
		} else {
			$HomColor = &makeHomColor($hom->{ident});
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $ColorBlack",
				"fill $x0 $y0 $HomColor");
			if ($partialL) {
				push(@{$this->{commands}},
					&makeBreak($x1, $y1, $y2, 'L'));
			}
			if ($partialR) {
				push(@{$this->{commands}},
					&makeBreak($x2, $y1, $y2, 'R'));
			}
			push(@{$this->{commands}},
				qq{string $Font $this->{Xmargin} $y1 "$hom->{spname2}" black});
#			last if (++$cnt>1000);
		}
	}
}

###	replacement of MBGD::Homology->select(
##		{genes=>$genename, species=>\@splist, with_length=>1});

sub getHomologyData {
	my(%opt) = @_;
	my($species) = $opt{species};
	my($gene) = $opt{genes};
	my($cmd) = "$main::CMD_select_homlist -SPEC=$species -ORIGOUT $gene ";
	my(@HomOut);

	open(P, "$cmd|") || die "Command failed: $cmd\n";
	while(<P>) {
		chomp;
		my($name1,$name2,$from1,$to1,$from2,$to2,$ident,$eval,$pam,$score,$cov)= split(/ /);
		push(@HomOut, {spname1=>$name1,spname2=>$name2,
				from1=>$from1,to1=>$to1,from2=>$from2,to2=>$to2,
				ident=>$ident,eval=>$eval,pam=>$pam,score=>$score,cov=>$cov});
	}
	close(P);
	@HomOut;
##	return  MBGD::Homology->select(\%opt);

}

sub drawMultAlign {
	my($this, %opt) = @_;
	my($alifile);
	if ($opt{alifile}) {
		$alifile = $opt{alifile};
	} else {
		$alifile = $this->{uInfo}->getpath('maliseq.aln');
		open(F,$alifile);
		my($line) = <F>;
		if ($line =~ /^>/) {
			## assuming fasta format
		} else {
			## assuming clustal format
			$alifile = "$main::CMD_clv2fs $alifile |";
		}
	}
	$ali = AlignmentFile->read($alifile);
	$this->setTitleMargin( $ali->getMaxNameLen );
	$aliReg = $ali->getAlignRegions;

	my $tabid = $this->{tabid};
	my($cltab_db) = MBGD::ClustTab::DB->new($tabid);
	$ClustHash = $cltab_db->getCluster($this->{clustid}, asHash=>1);
	my($alilen) = $ali->{length};
	foreach my $aliseq (@{$aliReg}) {
		my($spname) = $aliseq->{seq}->getname;
		my($sp,$name) = split(/[:_]/, $spname, 2);
		my($clstInfo) = $ClustHash->{$sp}->{$name};
#print STDERR $ClustHash->{$sp}->{$name},"<<<##$ClustHash,$sp,$name<\n";

 		my($y1,$y2) = ($this->{ypos}, $this->{ypos}+$BarHeight2);
		push(@{$this->{commands}},
			qq{string $Font $this->{Xmargin} $y1 "$spname" black});
		$this->{ypos} += $BarSkip2;
		my($y0) = ($y1+$y2)/2;
		my($color);
		foreach $reg (@{$aliseq->{regions}}) {
			my($x1,$x2) = $this->getLineCoord($reg->{from}, $reg->{to}, $alilen);
			my($x0) = ($x1+$x2)/2;
#print STDERR "$clstInfo->{from1},$clstInfo->{to1}; $reg->{fromIdx},$reg->{toIdx}; $spname\n";
			if ($clstInfo->{from1}-1 <= $reg->{fromIdx} && $reg->{toIdx} <= $clstInfo->{to1}) {
				$color = $ColorPink;
			} else {
				$color = $ColorGray200;
			}
#print STDERR "$spname; $x1,$x2,$y1,$y2; $x0,$y0\n" if ($spname =~ /physo:PHYSODRAFT_358116/);
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $ColorBlack",
				"fill $x0 $y0 $color");
		}
	}
	my($cons) = $ali->getConsensus;
	$this->{ypos} += $BarSkip2;
 	my($y1,$y2) = ($this->{ypos}, $this->{ypos}+$BarHeight2);
	my($y0) = ($y1+$y2)/2;
	my($from,$to,$p);
	$color = $ColorBlue;
	push(@{$this->{commands}},
			qq{string $Font $this->{Xmargin} $y1 "Conserved" black});
	foreach my $c (@{$cons}) {
		if ($c =~ /[A-Z]/) {
			if (! $from) {
				$from = $p;
				$to = $p + 1;
			} else {
				$to++;
			}
		} elsif ($from && $to) {
			my($x1,$x2) = $this->getLineCoord($from, $to, $alilen);
			my($x0) = ($x1+$x2)/2;
			push(@{$this->{commands}},
				"rect $x1 $y1 $x2 $y2 $color",
				"fill $x0 $y0 $color");
			$from = $to = 0;
		}
		$p++;
	}
	if ($from && $to) {
		my($x1,$x2) = $this->getLineCoord($from, $to, $alilen);
		my($x0) = ($x1+$x2)/2;
		push(@{$this->{commands}},
			"rect $x1 $y1 $x2 $y2 $color",
			"fill $x0 $y0 $color");
	}

#	foreach $cm (@{$this->{commands}}) {
#		print ">$cm<\n";
#	}
}


sub drawImage {
	my($this, %opt) = @_;
	$this->{ImageHeight} = $this->{ypos} + $BarSkip + $this->{Ymargin};
	unshift(@{$this->{commands}}, "ImageSize $this->{ImageWidth} $this->{ImageHeight}");
	my $image = &MBGDImg_Draw(@{$this->{commands}});
	if ($opt{with_header}) {
		print "Content-type: image/gif\n\n";
	}
	print $image->gif;
}

sub getLineCoord {
	my($this, $pos1,$pos2,$seqlen) = @_;
	($this->getCoord($pos1,$seqlen), $this->getCoord($pos2,$seqlen));
}
sub getCoord {
	my($this, $pos, $seqlen) = @_;
	int ($this->{Xmargin} + $this->{TitleMargin} + $pos * $this->{LineLength} / $seqlen);
}


sub make_clmap {
	my($this) = @_;
	join("\n", @{$this->{clmap}}) . "\n";
}
sub makeHomColor {
	my($ident) = @_;
	my($HomColor);
	foreach $hc (@HomColors) {
		if ($ident >= $hc->{ident}) {
			$HomColor = $hc->{color};
			last;
		}
	}
	$HomColor;
}
sub makeDomColor{
	my($motlib, $eval) = @_;
	my($bgcol, $fgcol);
	if ($motlib =~ /^COG/i) {
		$bgcol = $ColorCOG;
	} else {
		$bgcol = $ColorDom;
	}
	$fgcol = 'white';
	if ($eval < 1e-15) {
		#strong
	} elsif ($eval < 1e-5) {
		#medium
		$bgcol =~ s/0/100/g;
	} else {
		#weak
		$bgcol =~ s/0/160/g;
		$fgcol = '70_70_70';
	}
	($bgcol, $fgcol);
}
sub makeBreak{
	my($x1, $y1, $y2, $side) = @_;
	my($x2, $y3);
	$y3 = ($y1 + $y2) / 2;
	if ($side eq 'L') {
		$x2 = $x1 + 5;
	} elsif ($side eq 'R') {
		$x2 = $x1 - 5;
	}
	"polygon 3 $x1 $y1 $x1 $y2 $x2 $y3 $Background";
}
