#!/usr/bin/perl -s
use strict;
use DirHandle;
use FileHandle;
use MBGD;
use MBGD::DB;
use RECOG::DomClustCommon;
use RECOG::RecogCommon;

$| = 1;

###############################################################################
package RECOG::PhylopatCluster;

                                                # DB $B%"%/%;%9%(%i!<(B
$PhylopatCluster::ERRNO_DB_ACCESS             = "P0001";
                                                # $B%U%!%$%k%"%/%;%9%(%i!<(B
$PhylopatCluster::ERRNO_FILE_ACCESS           = "P0002";
                                                # fork() $B%(%i!<(B
$PhylopatCluster::ERRNO_FORK                  = "P0003";
                                                # $BBP1~$9$k%/%i%9%?!<$,L5$$(B
$PhylopatCluster::ERRNO_NO_CLUST_TAB          = "P0101";
                                                # $B%5%]!<%H30$N%b!<%I(B
$PhylopatCluster::ERRNO_NG_MODE               = "P0102";
                                                # $B%*%W%7%g%s0[>o(B
$PhylopatCluster::ERRNO_NG_OPTIONS            = "P0103";

%PhylopatCluster::ERRMSG = ( $PhylopatCluster::ERRNO_DB_ACCESS
                                => "DB access error.",
                             $PhylopatCluster::ERRNO_FILE_ACCESS
                                => "File access error.",
                             $PhylopatCluster::ERRNO_FORK
                                => "Fork error.",
                             $PhylopatCluster::ERRNO_NO_CLUST_TAB
                                => "No cluster table.",
                             $PhylopatCluster::ERRNO_NG_MODE
                                => "NG clustering mode.",
                             $PhylopatCluster::ERRNO_NG_OPTIONS
                                => "Ng options.",
                            );

###############################################################################
#
$PhylopatCluster::STA_STR_running  = 'running';
$PhylopatCluster::STA_STR_finished = 'finish';
$PhylopatCluster::STA_STR_error    = 'error';

#
$PhylopatCluster::TYPE_CLUST_RES_ADD  = 0;
$PhylopatCluster::TYPE_CLUST_RES_DOM  = 1;

###############################################################################
# phylopat$B%/%i%9%?%j%s%07k2L(B
$PhylopatCluster::TAB_index  = "cluster_tables_idx_phylopat";

# phylopat$B%/%i%9%?%j%s%0$NF~NO(B
$PhylopatCluster::TAB_src_index  = "cluster_tables_idx";

# $B3F(Bphylopat$B%/%i%9%?%j%s%07k2L$r3JG<$9$k%F!<%V%kL>>N(B
# '%s' $B$K$O!"%/%i%9%?!<%F!<%V%k<1JL;R$rKd$a9~$`(B
$PhylopatCluster::TAB_result = "cluster_result_phylopat_%s";

#
$PhylopatCluster::SQL_create_tab_index  = "create table if not exists $PhylopatCluster::TAB_index (
    clust_tab_id varchar(50) NOT NULL,
    status       int         NOT NULL default '0',
    progress     int         NOT NULL default '0',
    spec         text        NOT NULL,
    cmd          text        NOT NULL,
    cdate        timestamp   NOT NULL default CURRENT_TIMESTAMP on update CURRENT_TIMESTAMP,
    PRIMARY KEY  (clust_tab_id)
)";

#
$PhylopatCluster::SQL_create_tab_result = "create table if not exists $PhylopatCluster::TAB_result (
    clust_id    int NOT NULL auto_increment,
    type        smallint,
    data        mediumtext,
    PRIMARY KEY (clust_id)
)";

#
$PhylopatCluster::SQL_truncate_tab_result = "truncate table $PhylopatCluster::TAB_result";

# 'ignore 2 lines' $B$G7k2L%U%!%$%k$N%X%C%@#29T$rFI$_<N$F$F$$$k(B
$PhylopatCluster::SQL_load_data_result = "load data infile '%s' into table $PhylopatCluster::TAB_result
    fields terminated by '\t'
    ignore 2 lines
    (clust_id, type, data)";

#
$PhylopatCluster::SQL_insert_tab_index  = "insert $PhylopatCluster::TAB_index "
                                        . "(clust_tab_id, status, spec, cmd) "
                                        . "values "
                                        . "('%s', %d, '%s', '%s')";

#
$PhylopatCluster::SQL_select_tab_src_index    = "select * from $PhylopatCluster::TAB_src_index";
$PhylopatCluster::SQL_select_tab_src_index_id = $PhylopatCluster::SQL_select_tab_src_index
                                              . " where clusterID='%s' %s";

#
$PhylopatCluster::SQL_select_tab_index    = "select * from $PhylopatCluster::TAB_index";
$PhylopatCluster::SQL_select_tab_index_id = $PhylopatCluster::SQL_select_tab_index
                                          . " where clust_tab_id='%s' %s";

#
$PhylopatCluster::SQL_update_tab_index  = "update $PhylopatCluster::TAB_index "
                                        . "set %s "
                                        . "where clust_tab_id='%s'";

#
$PhylopatCluster::SQL_count_tab_result_dom  = "select count(*) as n_dom "
                                            . "from $PhylopatCluster::TAB_result "
                                            . "where type=%d ";
#
$PhylopatCluster::SQL_select_tab_result  = "select * "
                                         . "from $PhylopatCluster::TAB_result "
                                         . "order by clust_id";

#
$PhylopatCluster::TAB_clust_idx = "cluster_tables_idx";
$PhylopatCluster::SQL_select_clust_idx  = "select * "
                                        . "from $PhylopatCluster::TAB_clust_idx "
                                        . "where clusterID='%s'";

$PhylopatCluster::TAB_clust_tab = "cluster_result_%s";
$PhylopatCluster::SQL_select_clust_tab  = "select * "
                                        . "from $PhylopatCluster::TAB_clust_tab "
                                        . "order by clustid, subclustid";

#
$PhylopatCluster::NOT_EXISTS     = 0;
$PhylopatCluster::EXISTS         = 1;

#
$PhylopatCluster::STA_ERROR      = -99;
$PhylopatCluster::STA_FINISHED   =  -1;
$PhylopatCluster::STA_READY      =   0;

###############################################################################
#
sub new {
    my($class) = shift;
    my($dbname) = shift;
    my($self) = {};

    bless($self, $class);

    $self->setDbname($dbname);
    $self->clearMsg();

    #
    my($tab) = $PhylopatCluster::TAB_index;
    my($db) = $self->getDb();
    my($sta) = $db->exist_table($tab);
    if (! $sta) {
        my($sql) = $PhylopatCluster::SQL_create_tab_index;
        $db->execute($sql);
    }

    return $self;
}

###############################################################################
#
sub createId {
    my($self) = shift;

    my($newId) = sprintf("%d_%05d", time(), $$);

    return $newId;
}

###############################################################################
#
sub clearMsg {
    my($self) = shift;

    $self->{'MSG'} = [];

    return;
}

###############################################################################
#
sub addMsg {
    my($self) = shift;
    my($msg) = shift;

    push(@{$self->{'MSG'}}, $msg);

    return;
}

###############################################################################
#
sub getMsg {
    my($self) = shift;
    my($msg) = shift;

    return @{$self->{'MSG'}};
}

###############################################################################
#
sub makeResultFileName {
    my($self) = shift;
    my($clustTabId) = shift;

    my($filename) = sprintf("%s/work/tmp_phylopat_out_%s", $ENV{'MBGD_HOME'}, $clustTabId);

    return $filename;
}

###############################################################################
#
sub setDbname {
    my($self) = shift;
    my($dbname) = shift;

    $self->{'DBNAME'} = $dbname;
    $self->connectDb($dbname);

    return;
}

###############################################################################
#
sub getDbname {
    my($self) = shift;

    return $self->{'DBNAME'};
}

###############################################################################
#
sub connectDb {
    my($self) = shift;
    my($dbname) = shift || $self->{'DBNAME'};

    if (defined($self->{'DB'})) {
        $self->disconnectDb();
    }
    $self->{'DB'} = MBGD::DB->new($dbname);

    return $self->{'DB'};
}

###############################################################################
#
sub disconnectDb {
    my($self) = shift;

    if (defined($self->{'DB'})) {
        $self->{'DB'}->disconnect();
        $self->{'DB'} = undef();
    }

    return;
}

###############################################################################
#
sub getDb {
    my($self) = shift;

    return $self->{'DB'};
}

###############################################################################
#
sub existsClustTab {
    my($self) = shift;
    my($clust_tab_id) = shift;

    my($db) = $self->getDb();

    my($tab) = 'cluster_tables_idx';
    my($opt) = {};
    $opt->{'columns'} = '*';
    $opt->{'where'} = "clusterID='$clust_tab_id'";
    my($refRes) = $db->select_fetch($tab, $opt);
    if ($refRes->{'ROWS'} != 0) {
        return 1;
    }

    return 0;
}

###############################################################################
# $B%/%i%9%?%j%s%0%3%^%s%I%*%W%7%g%sJ8;zNs$N9=C[(B
#
# $refHash->{'min_exist'}
# $refHash->{'cutoff'}
# $refHash->{'missdist_ratio'}
# $refHash->{'SRC_CLUST_TAB_ID'}    # $B$b$H$H$J$k%/%i%9%?!<%F!<%V%k<1JL;R(B
sub makeCmdOptStr {
    my($self) = shift;
    my($refHash) = shift;
    my($minExist, $cutoff, $missdistRatio, $probcut, $disttype);

    #
    $minExist = $refHash->{'min_exist'};
    $minExist = 5 if (! $minExist);

    #
    $cutoff = $refHash->{'cutoff'};
    $cutoff = 0.1 if (! $cutoff);

    #
    $missdistRatio = $refHash->{'missdist_ratio'};
    $missdistRatio = 1 if ($missdistRatio < 1);

    #
    $probcut = $refHash->{'probcut'};
    $probcut = 1 if (!defined($probcut));

    #
    $disttype = $refHash->{'disttype'};
    $disttype = 'N' if (!defined($disttype));

    #
    my($cmdOpt) = join(' ', "-MIN_EXIST=$minExist",
                            "-CUTOFF=$cutoff",
                            "-MISSDIST_RATIO=$missdistRatio",
                            "-PROBCUT=$probcut",
                            "-DISTTYPE=$disttype",
                            "-SCALE=500",   # $B8GDj!J@h@8$h$j;X<($"$j!K(B
                    );

    # $B0J2<$N%*%W%7%g%s$O!"%/%i%9%?%j%s%07k2L$K1F6A$rM?$($J$$(B
    #   $B%/%i%9%?%j%s%07k2L$r<1JL$9$k$?$a$KIU2C$9$k(B
    if ($refHash->{'SRC_CLUST_TAB_ID'}) {
        $cmdOpt .= " -SRC_CLUST_TAB_ID=$refHash->{'SRC_CLUST_TAB_ID'}";
    }
    if ($refHash->{'mode_cluster'}) {
        $cmdOpt .= " -mode_cluster=$refHash->{'mode_cluster'}";
    }

    return $cmdOpt;
}

###############################################################################
# $B%/%i%9%?%j%s%0%3%^%s%IJ8;zNs$N9=C[(B
# $B%/%i%9%?!<(BID$B$d%/%i%9%?%j%s%0>r7o$,4^$^$l$k$?$a!"(B
# $B%/%i%9%?%j%s%07k2L3JG<$N%-!<$H$7$FMxMQ2DG=$G$"$k!#(B
#
# $refHash->{'min_exist'}
# $refHash->{'cutoff'}
# $refHash->{'missdist_ratio'}
# $refHash->{'SRC_CLUST_TAB_ID'}    # $B$b$H$H$J$k%/%i%9%?!<%F!<%V%k<1JL;R(B
sub makeCmdStr {
    my($self) = shift;
    my($refHash) = shift;

    #
    my($cmdOpt) = $self->makeCmdOptStr($refHash);
    my($cmd) = "$main::CMD_phylopatCluster $cmdOpt";

    return $cmd;
}

###############################################################################
#
sub parseCmdOpts {
    my($self) = shift;
    my($cmd) = shift;
    my($ref) = {};

    foreach my$opt (split(/\s+/, $cmd)) {
#        next if ($opt =~ /^\-SRC_CLUST_TAB_ID\=/i);

        if ($opt =~ /^\-(.+)/) {
            my($k, $v) = split(/\=/, $opt);
            $k = uc($k);
            if ($k =~ /\-(MIN_EXIST|CUTOFF|MISSDIST_RATIO|PROBCUT|SCALE)$/i) {
                # $B?tCM9`L\$NM-8z7e$r9g$o$;$k!J(B1.0 == 1 $B$G$"$k!K(B
                $v = sprintf("%.3f", $v);
            }
            $ref->{"$k"} = $v;
        }
    }

    return $ref;
}

###############################################################################
#
sub isSameCmdOpts {
    my($self) = shift;
    my($cmd0) = shift;
    my($cmd) = shift;

    my($refCmd0)  = $self->parseCmdOpts($cmd0);
    my(@keyList0) = keys(%{$refCmd0});

    my($refCmd)  = $self->parseCmdOpts($cmd);
    my(@keyList) = keys(%{$refCmd});

    if (scalar(@keyList0) != scalar(@keyList)) {
        # $B%*%W%7%g%s$N?t$,0c$&(B
        return 0;
    }

    foreach my$k (@keyList0) {
        if ($refCmd0->{"$k"} ne $refCmd->{"$k"}) {
            # $B%*%W%7%g%s$NCM$,0c$&(B
            return 0;
        }
    }

    return 1;
}

###############################################################################
#
sub getSrcClustTabInfo {
    my($self) = shift;
    my($srcClustTabId) = shift;

    my($db) = $self->getDb();
    my($sql) = sprintf($PhylopatCluster::SQL_select_tab_src_index_id, $srcClustTabId);
#print STDERR "SQL :: $sql\n";
    my($sth) = $db->execute($sql);
    if ($sth->rows() == 0) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        return undef();
    }

    my($ref) = $sth->fetchrow_hashref();
    return $ref;
}

###############################################################################
#
sub getClustTabInfo {
    my($self) = shift;
    my($clustTabId) = shift;

    my($db) = $self->getDb();
    my($sql) = sprintf($PhylopatCluster::SQL_select_tab_index_id, $clustTabId);
    my($sth) = $db->execute($sql);
    if ($sth->rows() == 0) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        return undef();
    }

    my($ref) = $sth->fetchrow_hashref();
    return $ref;
}

###############################################################################
#
sub getClustTabInfoByArgs {
    my($self) = shift;
    my($refArgs) = shift;

    #
    my($db) = $self->getDb();
    my($cmd0) = $self->makeCmdStr($refArgs);

    my($sql) = $PhylopatCluster::SQL_select_tab_index;
    my($sth) = $db->execute($sql);
    while(my$ref = $sth->fetchrow_hashref()) {
        my($staCmd) = $self->isSameCmdOpts($cmd0, $ref->{'cmd'});
        if ($staCmd) {
            return $ref;
        }
    }

    # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
    return undef();
}

###############################################################################
#
sub getIdByArgs {
    my($self) = shift;
    my($refArgs) = shift;

    #
    my($ref) = $self->getClustTabInfoByArgs($refArgs);
    if (! $ref) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        return $ref;
    }

    return $ref->{'clust_tab_id'};
}

###############################################################################
#
sub exists {
    my($self) = shift;
    my($clustTabId) = shift;

    my($ref) = $self->getClustTabInfo($clustTabId);
    if (! $ref) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        return $PhylopatCluster::NOT_EXISTS;
    }
    return $PhylopatCluster::EXISTS;
}

###############################################################################
#
sub existsByArgs {
    my($self) = shift;
    my($refArgs) = shift;

    my($clustTabId) = $self->getIdByArgs($refArgs);
    if (! $clustTabId) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        return $PhylopatCluster::NOT_EXISTS;
    }

    return $self->exists($clustTabId);
}

###############################################################################
#
sub getStatusByClusterId {
    my($self) = shift;
    my($clustTabId) = shift;

    my($ref) = $self->getClustTabInfo($clustTabId);
    if (! $ref) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        $self->addMsg(sprintf("ERROR :: No such cluster [ID:%s]", $clustTabId));
        return $PhylopatCluster::STA_ERROR;
    }

    my($sta) = $ref->{'status'};

    return $sta;
}

###############################################################################
#
sub getStatusByArgs {
    my($self) = shift;
    my($refArgs) = shift;

    #
    my($ref) = $self->getClustTabInfoByArgs($refArgs);
    if (! $ref) {
        # $B3:Ev$9$k(B JOB $B$,B8:_$7$J$$(B
        my($cmdOpt) = $self->makeCmdOptStr($refArgs);
        $self->addMsg(sprintf("ERROR :: No such cluster [CmdOpts:%s]", $cmdOpt));
        return $PhylopatCluster::STA_ERROR;
    }

    #
    my($sta) = $self->getStatusById($ref->{'id'});

    return $sta;
}

###############################################################################
#
sub createClusterResultTable {
    my($self) = shift;
    my($clustTabId) = shift;
    my($spec) = shift;
    my($cmd) = shift;

    my($db) = $self->getDb();
    my($sql);
    my($tab);
    my($sta);

    #
    $sql = sprintf($PhylopatCluster::SQL_insert_tab_index, $clustTabId, $PhylopatCluster::STA_READY, $spec, $cmd);
    $db->execute($sql);


    #
    $tab = sprintf($PhylopatCluster::TAB_result, $clustTabId);
    $sta = $db->exist_table($tab);
    if (! $sta) {
        $sql = sprintf($PhylopatCluster::SQL_create_tab_result, $clustTabId);
        $db->execute($sql);
    }

    return;
}

###############################################################################
# $B%9%F!<%?%9$N99?7(B
sub updateClusterResultStatus {
    my($self) = shift;
    my($clustTabId) = shift;
    my($sta) = shift;

    #
    my($db) = $self->getDb();
    my($tab);
    my($sql);

    #
    $tab = $PhylopatCluster::TAB_index;
    $sql = sprintf($PhylopatCluster::SQL_update_tab_index, sprintf("status='%s'", $sta), $clustTabId);
    $db->execute($sql);

    return;
}

###############################################################################
# progress$B$N99?7(B
sub updateClusterResultProgress {
    my($self) = shift;
    my($clustTabId) = shift;
    my($progress) = shift;

    #
    my($db) = $self->getDb();
    my($tab);
    my($sql);

    #
    $tab = $PhylopatCluster::TAB_index;
    $sql = sprintf($PhylopatCluster::SQL_update_tab_index, sprintf("progress='%s'", $progress), $clustTabId);
    $db->execute($sql);

    return;
}

###############################################################################
# $B%G!<%?EPO?8e!"%9%F!<%?%9$r99?7$9$k(B
sub loadClusterResultTable {
    my($self) = shift;
    my($clustTabId) = shift;
    my($fileResult) = shift;

    my($db) = $self->getDb();
    my($sql);
    my($tabName);
    my($sta);

    #
    $tabName = sprintf($PhylopatCluster::TAB_result, $clustTabId);
    $sta = $db->exist_table($tabName);
    if ($sta) {
        # $B$9$G$K(B MySQL $B$K%F!<%V%k$,B8:_$9$k!#?7$7$$7k2L$r3JG<$9$k$?$a!"8E$$%G!<%?$r:o=|!#(B
        $sql = sprintf($PhylopatCluster::SQL_truncate_tab_result, $clustTabId);
        $db->execute($sql);
    }

    #
    if (-e "$fileResult") {
        $sql = sprintf($PhylopatCluster::SQL_load_data_result, $fileResult, $clustTabId);
        $db->execute($sql);
        unlink("$fileResult");
        $sta = $PhylopatCluster::STA_FINISHED;
    }
    else {
        # $B7k2L%U%!%$%k$,B8:_$7$J$$(B
        $self->addMsg("ERROR :: No result file for $clustTabId");
        $sta = $PhylopatCluster::STA_ERROR;
    }

    return $sta;
}

###############################################################################
#
sub _execCmd {
    my($self) = shift;
    my($clustTabId) = shift;
    my($refArgs) = shift;
    my($fileIn) = shift;
    my($fileOut) = shift;

    # $B%/%i%9%?%j%s%0<B9T(B
    my($cmd) = $self->makeCmdStr($refArgs);
#    system("$cmd $fileIn >> $fileOut");

    my($db) = $self->getDb();
    my($sta) = '';
    my($progress) = 0;
    my($max_read);
    my($fhe) = FileHandle->new("$cmd $fileIn 2>&1 1>> $fileOut |");
    while(my$line = $fhe->getline()) {
        $line =~ s#[\r\n]*$##;

        if ($line =~ /^Done\s+\((\d+)\)/i) {
            $max_read = $1;
        }
        elsif ($line =~ /^phylopat_names\s+(\d+)/i) {
            my($cutoff) = $refArgs->{'cutoff'};
            $cutoff = 0.1 if (!$cutoff);
            $max_read = int($1 * $1 / 5 * $cutoff);
        }
        elsif ($line =~ /^read\s+(\d+)/i) {
            my($n) = $1;
            $sta = 'READING';
            $progress = 20;
            if (0 < $max_read) {
                my($p) = int($n / $max_read * 30);
                $p = 29 if (30 < $p);
                $progress += $p;
            }
        }
        elsif ($line =~ /^Sorting/i) {
            $sta = 'SORTING';
            $progress = 50;
        }
        elsif ($line =~ /^indexing\s+(\d+)/i) {
            my($n) = $1;
            $sta = 'INDEXING';
            $progress = 60;
            if (0 < $max_read) {
                $progress += int($n / $max_read * 30);
            }
        }
        elsif ($line =~ /^clustering\s+(\d+)/i) {
            $sta = 'CLUSTERING';
            $progress = 90;
        }
        elsif ($line =~ /^Done/i) {
            next;
        }
        else {
            next;
        }

        # progress $B$N99?7(B
        $self->updateClusterResultProgress($clustTabId, $progress);
    }

    #
    $progress = 100;
    $self->updateClusterResultProgress($clustTabId, $progress);

    return;
}

###############################################################################
#
sub _exec {
    my($self) = shift;
    my($refArgs) = shift;
    my($clustTabId) = shift;
    my($fileIn) = shift;
    my($fileOut) = shift;

    #
    my($sta) = $self->exists($clustTabId);
    if ($sta == $PhylopatCluster::EXISTS) {
        $sta = $self->getStatusByClusterId($clustTabId);
        if ($PhylopatCluster::STA_READY < $sta) {
            # $B<B9TCf(B
            return;
        }
    }
    else {
        #
        my($refSrcClustTab) = $self->getSrcClustTabInfo($refArgs->{'SRC_CLUST_TAB_ID'});
        my($spec) = ($refSrcClustTab->{'cmd'} =~ /\-spec\=(\S+)/i);
        my($cmd)  = $self->makeCmdStr($refArgs);
        $self->createClusterResultTable($clustTabId, $spec, $cmd);
    }

    # fork() $B$9$kA0$K(B MySQL $B$H$N%3%M%/%7%g%s$r@ZCG(B
    $self->disconnectDb();

##########
if (2 < $main::N_FORK) {
die("Too many fork().");
}
$main::N_FORK++;
##########

    print "";   # flush
    my($pid) = fork();
    $self->connectDb();    # fork() $B8e$K(B MySQL $B$H$N%3%M%/%7%g%s$r:F@\B3(B
    if ($pid) {
        # $B%/%i%9%?%j%s%0%9%F!<%?%9$r<B9TCf!J(B$pid$B!K$K99?7(B
        $self->updateClusterResultStatus($clustTabId, $pid);
    }
    elsif (defined($pid)) {
        close(STDIN);
        close(STDOUT);
        close(STDERR);

        # SRC_CLUST_TAB_ID $B$r$b$H$K%/%i%9%?%j%s%07k2L$r<hF@(B
        my($srcClustTabId) = $self->{'SRC_CLUST_TAB_ID'};
        if ($refArgs->{'mode_cluster'} =~ /^cluster/i) {
            $self->makeFilePhylopatInClust($fileIn, $clustTabId, $srcClustTabId);
        }
        elsif ($refArgs->{'mode_cluster'} =~ /^subcluster/i) {
            $self->makeFilePhylopatInSubClust($fileIn, $clustTabId, $srcClustTabId);
        }
        else {
            my($eid) = $PhylopatCluster::NG_MODE;
            my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
            printErrMsgExit($eid, $emsg, $srcClustTabId);
        }

        my($fhOut) = new FileHandle(">$fileOut");
        if (!$fhOut) {
            my($eid) = $PhylopatCluster::ERRNO_FILE_ACCESS;
            my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
            printErrMsgExit($eid, $emsg, $srcClustTabId);
        }

        $self->writePhylopatOutHead($fhOut);
        $fhOut->close();

        # $B%/%i%9%?%j%s%0<B9T(B
        $self->_execCmd($clustTabId, $refArgs, $fileIn, $fileOut);

        # $B%/%i%9%?%j%s%07k2L$r(B MySQL $B$KEPO?(B
        my($sta) = $self->loadClusterResultTable($clustTabId, $fileOut);

        # $B%/%i%9%?%j%s%0%9%F!<%?%9$r=*N;$K99?7(B
        $self->updateClusterResultStatus($clustTabId, $sta);
        unlink($fileIn);
        unlink($fileOut);

        exit(0);
    }
    else {
        my($eid) = $PhylopatCluster::ERRNO_FORK;
        my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
        printErrMsgExit($eid, $emsg);
    }

    return;
}

###############################################################################
#
sub writePhylopatInHead {
    my($self) = shift;
    my($fh) = shift;

    $fh->print(join("\t", 'ClusterID', 'Size', @{$self->{'SPECIES'}}, 'FuncCat', 'Gene'), "\n");

    return;
}

###############################################################################
#
sub writePhylopatOutHead {
    my($self) = shift;
    my($fh) = shift;

    $fh->print ('#SRC_CLUST_TAB_ID', "\t", $self->{'SRC_CLUST_TAB_ID'}, "\n");
    $fh->print('#ClusterID', "\t", join(' ', @{$self->{'SPECIES'}}), "\n");

    return;
}

###############################################################################
# $B%/%i%9%?C10L$G=PNO!J%5%V%/%i%9%?$r$R$H$^$H$a$K$9$k!K(B
sub writePhylopatInDataClust {
    my($self) = shift;
    my($fh) = shift;
    my($clustId) = shift;
    my($refClust) = shift;

    my($n) = 0;
    my(@orfList) = ();
    my(@subclustIdList) = keys(%{$refClust});
    foreach my$sp (@{$self->{'SPECIES'}}) {
        my($orfListSp) = '';
        foreach my$subclustId (@subclustIdList) {
            my($refSubClust) = $refClust->{"$subclustId"};
            if (exists($refSubClust->{"$sp"})) {
                $n += scalar(@{$refSubClust->{"$sp"}});
                $orfListSp .= ' ' if ($orfListSp);
                $orfListSp .= join(' ', @{$refSubClust->{"$sp"}});
            }
        }
        push(@orfList, $orfListSp);
    }

    $fh->print(join("\t", $clustId, $n, @orfList, '', ''), "\n");

    return;
}

###############################################################################
# $B%5%V%/%i%9%?C10L$G=PNO(B
sub writePhylopatInDataSubClust {
    my($self) = shift;
    my($fh) = shift;
    my($clustId) = shift;
    my($refClust) = shift;

    #
    my(@subclustIdList) = keys(%{$refClust});
    foreach my$subclustId (@subclustIdList) {
        my($refSubClust) = $refClust->{"$subclustId"};
        my($n) = 0;
        my(@orfList) = ();
        foreach my$sp (@{$self->{'SPECIES'}}) {
            my($orfListSp) = '';
            if (exists($refSubClust->{"$sp"})) {
                $n += scalar(@{$refSubClust->{"$sp"}});
                $orfListSp .= ' ' if ($orfListSp);
                $orfListSp .= join(' ', @{$refSubClust->{"$sp"}});
            }
            push(@orfList, $orfListSp);
        }

        #
        $fh->print(join("\t", join('_', $clustId, $subclustId), $n, @orfList, '', ''), "\n");
    }

    return;
}

###############################################################################
# MySQL $B$K3JG<$5$l$F$$$k%/%i%9%?%j%s%07k2L$N<h$j=P$7!#(B
# Cluster/SubCluster $B$4$H$K(BGene $B$r3JG<$9$k!#(B
#     $self->{'DATA_PHYLOPAT_IN_clust'}->{"$clustId"}->{"$subClustId"}->[];
#
sub selectPhylopatIn {
    my($self) = shift;
    my($srcClustTabId) = shift;

    #
    $self->{'DATA_PHYLOPAT_IN'}    = {};

    #
    my($db) = $self->getDb();
    my($sql) = sprintf($PhylopatCluster::SQL_select_clust_idx, $srcClustTabId);
    my($sth) = $db->execute($sql);
    if ($sth->rows() == 0) {
        # $B3:Ev$9$k%/%i%9%?%j%s%07k2L$,B8:_$7$J$$(B
        my($eid) = $PhylopatCluster::ERRNO_NO_CLUST_TAB;
        my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
        printErrMsgExit($eid, $emsg, $srcClustTabId);
    }

    #
    my($sql) = sprintf($PhylopatCluster::SQL_select_clust_tab, $srcClustTabId);
    my($sth) = $db->execute($sql);

    #
    my($refClust) = $self->{'DATA_PHYLOPAT_IN'};
    while(my$ref = $sth->fetchrow_hashref()) {
        #
        my($clustId) = $ref->{'clustid'};
        if (! exists($refClust->{"$clustId"})) {
            $refClust->{"$clustId"} = {};
        }

        #
        my($subClustId) = $ref->{'subclustid'};
        if (!exists($refClust->{"$clustId"}->{"$subClustId"})) {
            $refClust->{"$clustId"}->{"$subClustId"} = {};
        }

        #
        my($refSubClust) = $refClust->{"$clustId"}->{"$subClustId"};
        foreach my$d (split(/\s+/, $ref->{'name'})) {
            next if ($d !~ /\:/);

            my($sp, $orf) = split(/\:/, $d);
            if (! exists($refSubClust->{"$sp"})) {
                $refSubClust->{"$sp"} = [];
            }
            push(@{$refSubClust->{"$sp"}}, $d);
        }
    }

    return;
}

###############################################################################
#
sub selectSpecies {
    my($self) = shift;
    my($srcClustTabId) = shift;

    #
    my($db) = $self->getDb();
    my($sql) = sprintf($PhylopatCluster::SQL_select_clust_idx, $srcClustTabId);
    my($sth) = $db->execute($sql);
    if ($sth->rows() == 0) {
        # $B3:Ev$9$k%/%i%9%?%j%s%07k2L$,B8:_$7$J$$(B
        my($eid) = $PhylopatCluster::ERRNO_NO_CLUST_TAB;
        my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
        printErrMsgExit($eid, $emsg, $srcClustTabId);
    }

    # $BBP>]@8J*<o$N<h$j=P$7(B
    my(@listIgrp, @listOgrp);
    my($refIdx) = $sth->fetchrow_hashref();
    my($spec) = ($refIdx->{'cmd'} =~ /\-SPEC\=(\S+)/);
    my($ogrp) = ($refIdx->{'cmd'} =~ /\-Ooutgroup\=(\S+)/);
    my(%hashSpec, %hashOgrp);
    foreach my$sp (split(/,/, $spec)) {
        $hashSpec{"$sp"} = 1;
    }
    foreach my$sp (split(/,/, $ogrp)) {
        $hashOgrp{"$sp"} = 1;
    }
    foreach my$sp (keys(%hashSpec)) {
        if (exists($hashOgrp{"$sp"})) {
            push(@listOgrp, $sp);
        }
        else {
            push(@listIgrp, $sp);
        }
    }

    # Taxonomy $B$GJB$YBX$((B
    my($tax) = MBGD::Taxonomy->new();
    @listIgrp = $tax->sortByTaxonomy(@listIgrp);
    @listOgrp = $tax->sortByTaxonomy(@listOgrp);

    #
    $self->{'SPECIES'}     = [ @listIgrp, @listOgrp ];
    $self->{'SPECIES_IN'}  = [ @listIgrp ];
    $self->{'SPECIES_OUT'} = [ @listOgrp ];

    return;
}

###############################################################################
#
sub _makeFilePhylopatIn {
    my($self) = shift;
    my($fileIn) = shift;
    my($clustTabId) = shift;
    my($srcClustTabId) = shift;
    my($nameFunction) = shift;

    #
    my($max_progress) = 10;

    #
    $self->selectSpecies($srcClustTabId);

    #
    $self->selectPhylopatIn($srcClustTabId);

    #
    my($fhIn) = new FileHandle(">$fileIn");
    if (!$fhIn) {
        my($eid) = $PhylopatCluster::ERRNO_FILE_ACCESS;
        my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
        printErrMsgExit($eid, $emsg, $srcClustTabId);
    }

    $self->writePhylopatInHead($fhIn);

    #
    my(@clustIdList) = keys(%{$self->{'DATA_PHYLOPAT_IN'}});
    my($n) = scalar(@clustIdList);
    my($i) = 1;
    foreach my$clustId (@clustIdList) {
        my($refClust) = $self->{'DATA_PHYLOPAT_IN'}->{"$clustId"};
        $self->$nameFunction($fhIn, $clustId, $refClust);
        if ($i % 1000 == 0) {
            $self->updateClusterResultProgress($clustTabId, int($i / $n * $max_progress));
        }

        $i++;
    }
    $fhIn->close();

    # progress $B$N99?7(B
    $self->updateClusterResultProgress($clustTabId, $max_progress);

    return;
}

###############################################################################
#
sub makeFilePhylopatInClust {
    my($self) = shift;
    my($fileIn) = shift;
    my($clustTabId) = shift;
    my($srcClustTabId) = shift;

    #
    my($nameFunction) = 'writePhylopatInDataClust';
    $self->_makeFilePhylopatIn($fileIn, $clustTabId, $srcClustTabId, $nameFunction);

    return;
}

###############################################################################
#
sub makeFilePhylopatInSubClust {
    my($self) = shift;
    my($fileIn) = shift;
    my($clustTabId) = shift;
    my($srcClustTabId) = shift;

    #
    my($nameFunction) = 'writePhylopatInDataSubClust';
    $self->_makeFilePhylopatIn($fileIn, $clustTabId, $srcClustTabId, $nameFunction);

    return;
}

###############################################################################
#
sub execByArgs {
    my($self) = shift;
    my($refArgs) = shift;

    if (!$refArgs->{'SRC_CLUST_TAB_ID'}) {
        my($eid) = $PhylopatCluster::ERRNO_NO_CLUST_TAB;
        my($emsg) = $PhylopatCluster::ERRMSG{"$eid"};
        printErrMsgExit($eid, $emsg);
    }
    if (!$refArgs->{'mode_cluster'}) {
        $refArgs->{'mode_cluster'} = 'cluster';
    }

    #
    $self->{'SRC_CLUST_TAB_ID'} = $refArgs->{'SRC_CLUST_TAB_ID'};

    #
    my($newClustTabId) = $self->createId();

    #
    my($fileIn)  = sprintf("%s/work/tmp_phylopat_in_%s_%s",  $ENV{'MBGD_HOME'},
                                                             $newClustTabId,
                                                             $refArgs->{'mode_cluster'});

    #
    my($fileOut) = sprintf("%s/work/tmp_phylopat_out_%s_%s", $ENV{'MBGD_HOME'},
                                                             $newClustTabId,
                                                             $refArgs->{'mode_cluster'});

    #
    $self->_exec($refArgs, $newClustTabId, $fileIn, $fileOut);

    return;
}

###############################################################################
#
sub printStatus {
    my($self) = shift;
    my($clustTabId) = shift;

    #
    my($refClustTabInfo) = $self->getClustTabInfo($clustTabId);

    #
    my($species) = $refClustTabInfo->{'spec'};
    my($nSpecies) = scalar(split(/,/, $species));
    my($nFields) = $nSpecies + 2;
    my($status) = $PhylopatCluster::STA_STR_running;
    if ($refClustTabInfo->{'status'} == -1) {
        $status = $PhylopatCluster::STA_STR_finished
    }
    my($progress) = $refClustTabInfo->{'progress'};
    $progress = 99 if (100 <= $progress);

    my($status2);

    printf("#FORMAT_VER=1\n");
    printf("#N_FIELDS=%d\n",     $nFields);
    printf("#N_SPECIES=%d\n",    $nSpecies);
    printf("#SPECIES=%s\n",      $species);
    printf("#CLUSTER_ID=%s\n",   $clustTabId);
    printf("#STATUS=%s\n",       $status);
    printf("#STATUS2=%s\n",      $status2);
    printf("#PROGRESS=%s\n",     $progress);

    my($http_host_ref) = RECOG::RecogCommon::get_http_host();
    printf("#EXEC_SERVER=%s\n",  $http_host_ref->{'NAME'});
    printf("#EXEC_PORT=%s\n",    $http_host_ref->{'PORT'});
    printf("#EXEC_DATE=%s\n",    $refClustTabInfo->{'cdate'});

    return;
}

###############################################################################
#
sub printStatusHtml {
    my($self) = shift;
    my($clustTabId) = shift;

    print "Content-type: text/plain\n";
    print "\n";

    $self->printStatus($clustTabId);

    return;
}

###############################################################################
#
sub printResult {
    my($self) = shift;
    my($clustTabId) = shift;

    $self->printStatus($clustTabId);


    my($db) = $self->getDb();

    my($sql) = sprintf($PhylopatCluster::SQL_count_tab_result_dom, $clustTabId,
                                                                   $PhylopatCluster::TYPE_CLUST_RES_DOM);
    my($sth) = $db->execute($sql);
    my($ref) = $sth->fetchrow_hashref();
    my($nClusterDom) = $ref->{'n_dom'};

    my($sql) = sprintf($PhylopatCluster::SQL_select_tab_result, $clustTabId);
    my($sth) = $db->execute($sql);
    my($nCluster) = $sth->rows();

    printf("#N_CLUSTERS=%d\n",     $nCluster);
    printf("#N_CLUSTERS_DOM=%d\n", $nClusterDom);
    printf("#START_DATA\n");

    # $B%/%i%9%?%j%s%07k2L$N=PNO(B
    while(my$ref = $sth->fetchrow_hashref()) {
        print 'UPGMA', "\t", $ref->{'clust_id'}, "\n";
        my($tree) = $ref->{'data'};
        $tree =~ s#(.{1,80}[^\d\.\_])#\1\n#g;
        print $tree;
        if ($tree !~ /\n$/) {
            print "\n";
        }
    }

    return;
}

###############################################################################
#
sub printResultHtml {
    my($self) = shift;
    my($clustTabId) = shift;

    print "Content-type: text/plain\n";
    print "\n";

    $self->printResult($clustTabId);

    return;
}

###############################################################################
1;
###############################################################################
