#!/usr/bin/perl -s
use strict;
use warnings;
use DirHandle;
use File::Copy;
use File::Path;
use FileHandle;
use IO::File;
use MBGD::Genome;
use MBGD::Taxonomy;
use RECOG::RecogProject;
require 'InfoSpec.pl';
package RecogBuild;

###############################################################################
#   print_menu_bar_head()
#   get_date_taxontab()
#   print_menu_taxonomy()
#   print_menu_species()
#   print_menu_species_set()
#   print_menu_build_database()
#   print_menu_check_database()
#   print_menu_publish()
#   print_menu_bar_foot()
#   print_menu_bar()
#   print_menu_build_head()
#   print_menu_build_foot()
#   print_menu_build()
#   print_body_build()
#   print_table_species_ent()
#   print_table_species()
#   check_spidset_name()
#   exists_spidset_name()
#   print_spid_list()
#   get_new_spid()
#   load_user_genome()
#   save_user_genome()
#   load_user_chromosome_list()
#   save_user_chromosome()
#   check_files_species()
#   check_nodownload_species()
#   selectUpdSpecInfo()
#   add_user_genome_win()
#   create_user_chromosome()
#   remove_user_chromosome()
#   remove_user_genome()
#   set_building_status()
#   extract_sequences()
#   exec_blast_dp()
#   insert_blast_dp_res()
#   build_bldp()
#   build_homology_all()
#   check_build_status()
#   print_html_header_update()
#   print_html_running_update()
#   print_html_finished_update()
###############################################################################

###############################################################################
#
sub print_menu_bar_head {
    print <<EOB;
<div class="menu">
<h1>Build data</h1>
<ol>
EOB

    return;
}

###############################################################################
#
sub get_date_taxontab {
    my($date_taxonomy) = '';
    my($file_taxon) = "$ENV{'MBGD_HOME'}/etc/taxon.tab";
    if (-e $file_taxon) {
        my(@sta) = stat($file_taxon);
        my($sec, $min, $hour, $mday, $mon,, $year) = localtime($sta[9]);
        $year += 1900;
        $mon++;
        $date_taxonomy = sprintf("%04d/%02d/%02d", $year, $mon, $mday);
    }
    my($file_taxon_work) = "$ENV{'MBGD_HOME'}/database/taxonomy/taxon.tab";
    if (-e $file_taxon_work) {
        my(@sta) = stat($file_taxon_work);
        my($sec, $min, $hour, $mday, $mon,, $year) = localtime($sta[9]);
        $year += 1900;
        $mon++;
        my($date_taxonomy_work) = sprintf("%04d/%02d/%02d", $year, $mon, $mday);
        if ($date_taxonomy lt $date_taxonomy_work) {
            $date_taxonomy = $date_taxonomy_work;
        }
    }

    return $date_taxonomy;
}

###############################################################################
#
sub print_menu_taxonomy {
    my($date_taxonomy) = get_date_taxontab();

    print <<EOB;
<li onclick="disp_menu(''); "> Taxonomy
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <a title="$date_taxonomy" href="javascript:void(0); " onclick="download_taxdump_win(); return false; ">Download taxdump
      </a>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_species {
    my($date_list)  = "YYYY/MM/DD";
    my($date_new)   = "YYYY/MM/DD";
    my($date_mysql) = "YYYY/MM/DD";

    print <<EOB;
<li onclick="disp_menu(''); ">Species
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <a title="$date_list"
         href="javascript:void(0); "
         onclick="list_species_win(); return false; "
        >List(Modify/Delete)
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <a title="$date_new"
         href="javascript:void(0); "
         onclick="new_species_win(); return false;
        ">New species
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <a title="$date_mysql"
         href="javascript:void(0); "
         onclick="update_mysql_species_win(); return false; "
        >Update mysql
      </a>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_species_set {
    my($date_upd)   = "YYYY/MM/DD";
    my($date_spset) = "YTYYY/MM/DD";

    print <<EOB;
<li onclick="disp_menu(''); ">Species set
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <a title="$date_upd"
         href="javascript:void(0); "
         onclick="upd_taxtree_win(); return false; "
        >Update Taxonomy tree
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <a title="$date_spset"
         href="javascript:void(0); "
         onclick="list_species_set_win(); return false; "
        >List species set
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <a href="javascript:void(0); "
         onclick="select_species_set_win(); return false; "
        >Select species set
      </a>
      <div id="selected_species_set">
      </div>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_build_database {
    my($sta_chkbox_extract)  = 'checked';
    my($sta_chkbox_blast_dp) = 'checked';
    my($sta_chkbox_motif)    = 'checked';
    my($sta_chkbox_mysql)    = 'checked';
    my($sta_chkbox_database) = 'checked';
    my($sta_chkbox_misc)     = 'checked';

    print <<EOB;
<li onclick="disp_menu(''); ">
  <a href="javascript:vaoid(); " 
     onclick="build_homology_all_win(); "
    >Build homology database
  </a>
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_extract">
      <a title="$sta_chkbox_extract"
         href="javascript:void(0); "
         onclick="get_sequences_win(); return false; "
        >Extract sequences
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_blast_dp">
      <a title="$sta_chkbox_blast_dp"
         href="javascript:void(0); "
         onclick="exec_blast_win(); "
        >BLAST/DP alignment
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_motif">
      <a title="$sta_chkbox_motif"
         href="javascript:void(0); "
         onclick="exec_motif_win(); "
        >Motif search
      </a>
    </li>

    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_mysql">
      <a title="$sta_chkbox_mysql"
         href="javascript:void(0); "
         onclick="update_mysql_homology_win(); "
        >Update mysql
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_database">
      <a title="$sta_chkbox_database"
         href="javascript:void(0); "
         onclick="build_homology_db_win(); "
        >Build homology DB
      </a>
    </li>
    <li onclick="disp_menu(''); ">
      <input type="checkbox" name="" value="" checked="$sta_chkbox_misc">
      <a title="$sta_chkbox_misc"
         href="javascript:void(0); "
         onclick="build_misc_db_win(); "
        >Build misc DB
      </a>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_check_database {
    my($date_check_db) = '';

    print <<EOB;
<li onclick="disp_menu(''); ">Check database
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <a title="$date_check_db"
         href="javascript:void(0); "
         onclick="publish_db_win(); "
        >Publish DB
      </a>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_publish {
    my($date_publish_db) = '';

    print <<EOB;
<li onclick="disp_menu(''); ">Publish database
  <ol type="i">
    <li onclick="disp_menu(''); ">
      <a title="$date_publish_db"
         href="javascript:void(0); "
         onclick="publish_db_win(); "
        >Publish DB
      </a>
    </li>
  </ol>
</li>
EOB

    return;
}

###############################################################################
#
sub print_menu_bar_foot {
    print <<EOB;
</ol>
</div>
EOB

    return;
}

###############################################################################
#
sub print_menu_bar {
    my(@args) = shift;

    print_menu_bar_head();

    print_menu_taxonomy();
    print_menu_species();
    print_menu_species_set();
    print_menu_build_database();
    print_menu_check_database();
    print_menu_publish();

    print_menu_bar_foot();

    return;
}

###############################################################################
#
sub print_menu_build_head {

    print <<EOB;
<html>
<head>
<link rel="stylesheet" type="text/css" href="/css/mbgd.css" />
<link rel="stylesheet" type="text/css" href="/css/recog_build.css" />
<script type="text/javascript" src="/js/recog_common.js"></script>
<script type="text/javascript" src="/js/recog_build.js"></script>
<!--
src="http://muffinresearch.co.uk/code/javascript/getElementsByClassName/getElementsByClassName.js"
-->
<script type="text/javascript" src="/js/getElementsByClassName.js"></script>
</head>
EOB

    return;
}

###############################################################################
#
sub print_menu_build_foot {
    print <<EOB;
</html>
EOB

    return;
}

###############################################################################
#
sub print_menu_build {
    print_menu_build_head();

    print_menu_bar();

    print_menu_build_foot();

    return;
}

###############################################################################
#
sub print_body_build {
    print <<EOB;
<div id="body">
</div>
EOB

    return;
}

###############################################################################
#
sub print_table_species_ent {
    my($sp) = shift;
    my($sel_spid_ref) = shift;

    my($genome) = MBGD::Genome->get([$sp]);
    my($spid)   = $genome->{'spid'};
    my($name)   = $genome->{'orgname'};
    my($strain) = $genome->{'strain'};

    my($sta_checked) = '';
    if ($sel_spid_ref->{"$spid"}) {
        $sta_checked = 'checked';
    }

    my($html) = '';

    $html .= <<EOB;
<span class="spidset_ent" title="$name"><label><input type="checkbox" name="spidset_ent" value="$spid" $sta_checked>$sp</label></span>
EOB

    return $html;
}

###############################################################################
#
sub print_table_species {
    my($sel_spid_ref) = shift || {};

    my($html) = '';

    my($dir_taxon) = "$ENV{'MBGD_HOME'}/database.work/";
    my($file_taxon) = "$dir_taxon/tax";
    if (!-e $file_taxon) {
        return 'No tax';
    }
    my($tax) = MBGD::Taxonomy->new($dir_taxon);


    $html .= "<table width=\"100%\" border>\n";

    my(@Superkingdoms) = ( 'Bacteria', 'Archaea', 'Eukaryota' );
    foreach my$superkingdom ( @Superkingdoms ) {
        $html .= "<tr><td>$superkingdom<br>\n";
        $html .= "        <table width=\"100%\" border>\n";

        my(@Phylum)  = ();
        my(@Class)   = ();

        $tax->find_tree(\@Phylum, {
                        'match_rank' => 'superkingdom',
                        'match_name' => $superkingdom,
                        'get_rank'   => 'phylum'});

        foreach my$phylum (@Phylum) {
            my(@spnames);
            if ($phylum =~ /^Proteobacteria$/i) {
                $tax->find_tree(\@Class, { 'match_rank' => 'phylum',
                                           'match_name' => $phylum,
                                           'get_rank'   => 'class',
                                            });
                my($n) = scalar(@Class);
                $html .= "<tr><td rowspan=\"$n\">$phylum</td>";
                for(my$i = 0; $i < $n; $i++) {
                    my$class = $Class[$i];
                    $html .= "<tr>" if ($i != 0);
                    $html .= "<td>$class</td>";

                    @spnames = $tax->get_species( { 'match_rank' => 'class',
                                                    'match_name' => $class,
                                                    });

                    $html .= "<td>";
                    foreach my$sp (@spnames) {
                        $html .= print_table_species_ent($sp, $sel_spid_ref);
                    }
                    $html .= "</td>";
                    $html .= "</tr>\n";
                }
            }
            else {
                $html .= "<tr><td colspan=\"2\">$phylum</td>";
                @spnames = $tax->get_species( { 'match_rank' => 'phylum',
                                                'match_name' => $phylum,
                                                });

                $html .= "<td>";
                foreach my$sp (@spnames) {
                    $html .= print_table_species_ent($sp, $sel_spid_ref);
                }
                $html .= "</td>";
                $html .= "</tr>\n";
            }
        }

        $html .= "        </table>\n";
        $html .= "    </td>\n";
        $html .= "</tr>\n";


    }
    $html .= "</table>\n";

    return $html;
}

###############################################################################
#
sub check_spidset_name {
    my($name) = shift;

    if ($name =~ /^\s*$/) {
        return 'Too short';
    }

    if ($name =~ /[^A-Za-z0-9_]/) {
        return 'Invalid character';
    }

    my($recog_project_ref) = RECOG::RecogProject->new();
    foreach my$spidset_name ($recog_project_ref->get_setname()) {
        if ($name =~ /^$spidset_name$/i) {
            return 'Used';
        }
    }
    return;
}

###############################################################################
#
sub exists_spidset_name {
    my($name) = shift;

    my($recog_project_ref) = RECOG::RecogProject->new();
    foreach my$spidset_name ($recog_project_ref->get_setname()) {
        if ($name =~ /^$spidset_name$/i) {
            return;
        }
    }
    return 'Not found';
}

###############################################################################
#
sub print_spid_list {
    #
    my($recog_project_ref) = RECOG::RecogProject->new();
    print <<EOB;
<h1>Species set</h1>
<form name="form_spid_set" method="post" action="">
<input type="hidden" name="spid_set_name" value="">
<input type="button" name="" value="NEW set" onclick="add_spid_set();">
<input type="button" name="" value="Set default" onclick="set_default_spidset();">
<table border>
<tr><th></th><th></th><th>default</th><th>name</th><th>species</th>
</tr>
EOB

    #
    my(@setname_list) = $recog_project_ref->get_setname();
    my($defalut_spidset) = $recog_project_ref->get_default_spidset();
    foreach my$setname (@setname_list) {
        if ($defalut_spidset =~ /^\s*$/) {
            $defalut_spidset = $setname;
        }

        my(@spid_list) = $recog_project_ref->get_spid($setname);
        my($sta_default_spidset) = '';
        if ($setname eq $defalut_spidset) {
            $sta_default_spidset = 'checked';
        }
        print <<EOB;
<tr>
    <td><input type="button" name="" value="DEL" onclick="del_spidset('$setname'); "></td>
    <td><input type="button" name="" value="Edit" onclick="mod_spidset_win('$setname'); "></td>
    <td align="center"><input type="radio" name="default_spidset" value="$setname" $sta_default_spidset></td>
    <td>$setname</td>
    <td>
EOB

        #
        foreach my$spid (@spid_list) {
            my($sp) = main::spid2sp($spid);
            
            my($genome) = MBGD::Genome->get([$sp]);
            my($name)   = $genome->{'orgname'};
            print <<EOB;
<span title="$name">$sp</span>
EOB
        }

        print <<EOB;
</td>
</tr>
EOB

    }
    print <<EOB;
</table>
</form>
EOB

    return;
}

###############################################################################
#
sub get_new_spid {

    my($spid);
    for (my$i = 1; $i <= 99999; $i++) {
        my($tmp_spid) = sprintf("gu%05d", $i);
        my($dir) = sprintf("%s/species/%s", $ENV{'RECOG_HOME'}, $tmp_spid);
        if (! -e $dir) {
            $spid = $tmp_spid;
            last;
        }
    }

    return $spid;
}

###############################################################################
#
sub load_user_genome {
    my($spid) = shift;
    my($ug_ref) = {};

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($filename) = "$dir/genome.txt";
    my($fh) = FileHandle->new("$filename");
    if (!$fh) {
        return;
    }
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        $line =~ s#[\r\n]*$##;
        my($key, $val) = split(/\t/, $line);
        $ug_ref->{"$key"} = $val;
    }
    $fh->close();

    return $ug_ref;
}

###############################################################################
#
sub save_user_genome {
    my($spid) = shift;
    my($ug_ref) = shift;

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($filename) = "$dir/genome.txt";
    my($fh) = FileHandle->new(">$filename");
    if (!$fh) {
        return;
    }

    #
    foreach my$key ('sp', 'orgname', 'abbrev', 'strain', 'taxid') {
        my($val) = $ug_ref->{"$key"};
        $fh->print($key, "\t", $val, "\n");
    }

    $fh->close();

    return $ug_ref;
}

###############################################################################
#
sub load_user_chromosome_list {
    my($spid) = shift;
    my(@user_chr_list);

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($dir_chr) = "$dir/data";
    my($dh) = DirHandle->new("$dir_chr");
    if (!$dh) {
        return @user_chr_list;
    }

    #
    my(@uc_id_list);
    while (my$file=$dh->read()) {
        if ($file =~ /^(\d+)\.gene$/) {
            push(@uc_id_list, $1);
        }
    }

    #
    foreach my$uc_id (sort {$a<=>$b} @uc_id_list) {
        my($uc_ref) = {};
        $uc_ref->{'id'} = $uc_id;
        $uc_ref->{'seqno'} = $uc_id;
        $uc_ref->{'length'} = '';

        my($filename) = "$dir_chr/$uc_id.chromosome.txt";
        my($fh) = FileHandle->new("$filename");
        if (!$fh) {
            print STDERR "DBG :: WARNING :: Can not open $filename($!)\n";
            next;
        }

        while (my$line=$fh->getline()) {
            next if ($line =~ /^\s*$/);
            next if ($line =~ /^\s*#/); 

            $line =~ s#[\r\n]*$##;
            my($key, $val) = split(/\t/, $line);
            $uc_ref->{"$key"} = $val;
       }
        $fh->close();

        push(@user_chr_list, $uc_ref);
    }

    return @user_chr_list;
}

###############################################################################
#
sub save_user_chromosome {
    my($spid) = shift;
    my($uc_id) = shift;
    my($uc_ref) = shift;

    if ($uc_id < 1) {
        return;
    }

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($filename) = "$dir/data/$uc_id.chromosome.txt";
    my($fh_chr) = FileHandle->new(">$filename");
    if (!$fh_chr) {
        return;
    }

    my($key, $val);

    #
    $key = 'spid';
    $val = $spid;
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'name';
    $val = $uc_ref->{"name_uc_$uc_id"};
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'type';
    $val = $uc_ref->{"type_uc_$uc_id"};
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'shape';
    $val = $uc_ref->{"shape_uc_$uc_id"};
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'seqno';
    $val = $uc_id;
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'accession';
    $val = join('_', $spid, $uc_id);
    $fh_chr->print($key, "\t", $val, "\n");

    #
    $key = 'file_gene';
    $val = $uc_ref->{"file_gene_table_uc_$uc_id"};
    if ($val) {
        my($filename) = "$dir/data/$uc_id.gene";
        my($fh) = FileHandle->new(">$filename");
        $fh->print(<$val>);
        $fh->close();
    }

    #
    $key = 'file_protein';
    $val = $uc_ref->{"file_protein_uc_$uc_id"};
    if ($val) {
        my($filename) = "$dir/data/$uc_id.protseq";
        my($fh) = FileHandle->new(">$filename");
        $fh->print(<$val>);
        $fh->close();
    }

    #
    $key = 'file_dna';
    $val = $uc_ref->{"file_dna_uc_$uc_id"};
    if ($val) {
        my($filename) = "$dir/data/$uc_id.chrseq";
        my($fh) = FileHandle->new(">$filename");
        $fh->print(<$val>);
        $fh->close();
    }

    #
    $key = 'file_gbk';
    $val = $uc_ref->{"file_gbk_uc_$uc_id"};
    if ($val) {
        my($fh_gbk) = $val;

        # $B%U%!%$%k%5%$%:$N3NG'(B
        seek($fh_gbk, 0, 2);
        my($size_gbk) = tell($fh_gbk);
        seek($fh_gbk, 0, 0);
        if (1000000000 < $size_gbk) {
            $uc_ref->{'MASSAGE'} .= "<br>" if ($uc_ref->{'MASSAGE'} ne '');
            $uc_ref->{'MASSAGE'} .= "So big. Can not save chromosome file.";
        }

        #
        my($multi_locus) = $uc_ref->{"multi_locus_uc_$uc_id"};
        my($file_gbk) = "$dir/data/$uc_id.ncbiGbk";
        File::Copy::cp($fh_gbk, $file_gbk);

        #
        $filename = "$dir/data/$uc_id.gene";
        my($fh_tab) = FileHandle->new(">$filename");

        $filename = "$dir/data/$uc_id.protseq";
        my($fh_protseq) = FileHandle->new(">$filename");

        $filename = "$dir/data/$uc_id.chrseq";
        my($fh_chrseq) = FileHandle->new(">$filename");

        #
        my($ug_id) = int($spid =~ /(\d+)/);
        my($genetab_ref) = MbgdUserGenomeCommon::readNcbiGbk($file_gbk, $ug_id, $uc_id, $multi_locus);
        my($name, $gene, $from, $to, $dir, $desc, $seq);
        foreach my$ent (@{$genetab_ref->{'LIST'}}) {
            next if ($ent->{'PSEUDO'});

            $name = $ent->{'ID'};
            $gene = $ent->{'GENE_NAME'} || '';
            $from = $ent->{'FROM'};
            $to   = $ent->{'TO'};
            $dir  = $ent->{'DIR'};
#            if ($multi_locus) {
#                $from = 0;
#                $to   = 0;
#                $dir  = 0;
#            }
            $desc = $ent->{'DESC'} || '';
            $seq  = $ent->{'SEQ'};
            $seq =~ s#(.{1,60})#$1\n#g if ($seq);

            #
            $fh_tab->print(join("\t", $name, $gene, $from, $to, $dir, 'CDS', $desc) . "\n");

            #
            $fh_protseq->print(">$spid:$name\n");
            $fh_protseq->print($seq);
        }

        if (!$multi_locus) {
            $fh_chr->print('length', "\t", $genetab_ref->{'length'}, "\n");
            $fh_chrseq->print($genetab_ref->{'CHR_SEQ'});
        }

        $fh_tab->close();
        $fh_protseq->close();
        $fh_chrseq->close();
    }

    #
    $fh_chr->close();

    return;
}

###############################################################################
sub check_files_species {
    my($spid) = shift;

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'RECOG_HOME'}, $spid);

    #
    my(@file_list);
    push(@file_list, "$dir");
    push(@file_list, "$dir/genome.txt");
    push(@file_list, "$dir/data");
    push(@file_list, "$dir/data/filelist");
    foreach my$file ("$dir/genome.txt") {
        if (! -e $file) {
            return;
        }
    }

    #
    if ($spid =~ /^gu\d+$/) {
        return 1;
    }

    #
    my($file_list) = "$dir/data/filelist";
    my($fh) = FileHandle->new($file_list);
    if (!$fh) {
        print STDERR "ERROR :: Can not open $file_list($!)\n";
        return;
    }

    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;

        my($file) = "$dir/data/$line";
        if (! -e $file) {
            $fh->close();
            return;
        }
    }
    $fh->close();

    return 1;
}

###############################################################################
#
sub check_nodownload_species {
    my(@spid_list) = @_;

    #
    my(%spid_hash);
    foreach my$spid (@spid_list) {
        my($sta) = RecogBuild::check_files_species($spid);
        if (!$sta) {
            $spid_hash{"$spid"} = 1;
        }
    }
    my(@no_download_spid) = sort keys(%spid_hash);

    return @no_download_spid;
}

###############################################################################
# $B99?7=hM}$NBP>]$H$J$k(B spec $B%G!<%?$rC5$9(B
sub selectUpdSpecInfo {
    my($updDate);
    my($date, $time);
    my($year, $mon, $day);
    my($hour, $min, $sec);
    my($refRes);
    my($dirname) = "$ENV{'MBGD_HOME'}/species";
    my($dh);
    my($spid);
    my(%specHash);
    my(@specList);

    # spid.tab $B%U%!%$%k$KDj5A$5$l$F$$$k(B spid $B$r<h$j=P$9(B
    my($infoSpec) = main::getInfoSpecTab($main::FILE_spidtab);
    my(@spidList) = sort(keys(%{$infoSpec->{'SPID2DIR'}}));

    #
    foreach $spid (@spidList) {
        my($file);

        # gene $B%G!<%?$N99?7F|IU(B
        $file = "$main::DIR_species/$spid/gm/updated";
        my($updDate) = &main::getFileTimestamp($file);
        print STDERR "Date(last update) : $updDate\n";

        # $B%G!<%?%U%!%$%k$N99?7F|IU$r<hF@(B
	my($specDate);
        foreach my $file ( <$dirname/$spid/gm/data/*.{gene,chromosome.txt,chrseq,protseq,geneseq}>, 
				"$dirname/$spid/gm/genome.txt" ) {
        	$specDate = &main::getFileTimestamp($file);
		last if ($specDate gt $updDate);	# updated data is found
	}
        print STDERR "Date($file) : $specDate\n";

        #
        if ($specDate lt $updDate) {
            # $B%G!<%?%U%!%$%k$NF|IU$,8E$$(B
            # $B4{$KEPO?:Q$_$N%G!<%?$H8+$J$9(B
            print STDERR "SKIP :: $spid data file is older than gene table.\n";
            next;
        }

        #
        push(@specList, $spid);
    }

    return @specList;
}

###############################################################################
#
sub build_extract_sequences {

    #
    my($procname) = "m0400NewProtseq";
    my(@fStat) = stat("$main::DIR_work/.$procname.end");
    my($sec, $min, $hour, $day, $mon, $year)  = localtime($fStat[9]);
    $year += 1900;
    $mon++;
    my($date) = sprintf("%04d%02d%02d", $year, $mon, $day);

    #
    my(@spec_list) = selectUpdSpecInfo();
    my($splist) = join(',', @spec_list);

    #
    $procname = "m0490SelectUpdSeq";
    my($cmd) = "$main::DIR_build/$procname.pl -nolog -DEBUG='$main::DEBUG' -SPEC=$splist -DATE=$date";
    system("$cmd");

    return;
}

###############################################################################
#
sub add_user_genome_win {
    my($formOpt) = shift;
    my($user) = $ENV{'REMOTE_USER'};

    my($mode) = 'Add';
    my($spid) = $formOpt->{'spid'} || '';
    my($ug_ref) = {};
    if ($spid) {
        $ug_ref = RecogBuild::load_user_genome($spid);
        $mode = 'Edit';
    }
    my($sp)      = $ug_ref->{'sp'}       || '';
    my($orgname) = $ug_ref->{'orgname'}  || '';
    my($abbrev)  = $ug_ref->{'abbrev'}   || '';
    my($strain)  = $ug_ref->{'strain'}   || '';
    my($taxid)   = $ug_ref->{'taxid'}    || '';
    my($msg)     = $formOpt->{'MESSAGE'} || '';

    #
    print "Content-type: text/html\n";
    print "\n";

    print <<EOB;
<?xml version="1.0" encoding="EUC-JP">
<html>
<head>
<link rel="stylesheet" type="text/css" href="/css/mbgd.css" />
<link rel="stylesheet" type="text/css" href="/css/recog_build.css" />
<script type="text/javascript" src="/js/mymbgd.js"></script>
<script type="text/javascript" src="/js/recog_common.js"></script>
<script type="text/javascript" src="/js/recog_build.js"></script>
<script>
this.top.opener.location.reload();
</script>
<script>
function set_more_chromosome(sta) {
    var doc = this.document;

    doc.frm_add_user_genome.target = '';
    doc.frm_add_user_genome.more_chromosome.value = sta;

    return;
}

function delete_user_chromosome(ucid) {
    var doc = this.document;

    var sta = confirm('Delete from this chromosome?');
    if (!sta) {
        return;
    }

    doc.frm_add_user_genome.del_chromosome.value = ucid;
    doc.frm_add_user_genome.target = '';
    doc.frm_add_user_genome.action = '/htbin/recog_build/mod_user_genome.cgi';

    return;
}
function view_user_chromosome(ucid) {
    var doc = this.document;

    doc.frm_add_user_genome.target = 'view_chr' + ucid;
    doc.frm_add_user_genome.action = '/htbin/recog_build/viewGeneInfoFile.cgi';
    doc.frm_add_user_genome.view_chromosome.value = ucid;

    return;
}
</script>
</head>
<body>
<h1>$mode user genome</h1>
$msg
<form name="frm_add_user_genome" method="post" enctype="multipart/form-data">
<input type="hidden" name="spid" value="$spid">
<input type="hidden" name="del_chromosome" value="">
<input type="hidden" name="view_chromosome" value="">
<fieldset><legend>Genome</legend>
<table width="100%" border>
<tr><th width="40%">Fullname<br>(e.g. Escherichia coli)</th>
    <td><input class="" type="text" name="orgname" value="$orgname" size="60"></td>
</tr>

<tr><th>Abbreviation Name<br>(e.g. E.coli)</th>
    <td><input class="" type="text" name="abbrev" value="$abbrev" size="60">
        <input type="button" name="btn_set_abbrev" value="Set" onclick="setShortName(orgname, abbrev);">
    </td>
</tr>

<tr><th>species code<br>(<a href="/dist/spid.tab.dist" target="spid_tab_dist">e.g. eco</a>)</th>
    <td><input class="" type="text" name="sp" value="$sp" size="60"></td>
</tr>

<tr><th>Strain</th>
    <td><input class="" type="text" name="strain" value="$strain" size="60"></td>
</tr>

<tr><th>Taxonomy ID</th>
    <td><input class="" type="text" name="taxid" value="$taxid" size="60">
        <input type="button" name="btn_search_taxid" value="Search the NCBI Taxonomy DB" onclick="searchTaxonomyId(orgname, abbrev);"><br>
        e.g. 511145(=Escherichia coli K-12 MG1655), 32644(=Unknown)
    </td>
</tr>

</table>
</fieldset>
EOB

    #
    my($modNewUserChr) = '';

    #
    my($dirUc) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my(@uc_id_list);
    my(@uc_list) = RecogBuild::load_user_chromosome_list($spid);
    my($n_uc) = scalar(@uc_list);
    foreach my$uc_ref (@uc_list) {
        my($uc_id) = $uc_ref->{'id'};

        push(@uc_id_list, $uc_id);

        my($sta_circular) = 'checked';
        my($sta_linear)   = '';
        if ($uc_ref->{"shape"} =~ /linear/) {
            $sta_circular = '';
            $sta_linear   = 'checked';
        }

        #
        my($fileError)   = "$dirUc/data/$uc_id.error.txt";
        my($fileGbk)     = "$dirUc/data/$uc_id.ncbiGbk";
        my($fileGeneTab) = "$dirUc/data/$uc_id.gene";
        my($fileProtein) = "$dirUc/data/$uc_id.protseq";
        my($fileDna)     = "$dirUc/data/$uc_id.chrseq";

        #
        my($sta_file_gbk) = '';
#        if ($modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_genbank/i) {
            FileHandle->new(">>$fileGbk"); # touch
#        }
        if (-e "$fileGbk") {
            if (-s $fileGbk) {
                $sta_file_gbk = 'Uploaded';
            }
            else {
                $sta_file_gbk = 'Not uploaded';
            }
            $sta_file_gbk .= '<br>';
        }

        #
        my($sta_file_gene_table) = '';
#        if ($modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_genetab/i) {
            FileHandle->new(">>$fileGeneTab"); # touch
#        }
        if (-e "$fileGeneTab") {
            if (-s $fileGeneTab) {
                $sta_file_gene_table = 'Uploaded';
            }
            else {
                $sta_file_gene_table = 'Not uploaded';
            }
            $sta_file_gene_table .= '<br>';
        }

        #
        my($sta_file_protein) = '';
#        if ($modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_proteinseq/i) {
            FileHandle->new(">>$fileProtein"); # touch
#        }
        if (-e $fileProtein) {
            if (-s $fileProtein) {
                $sta_file_protein = 'Uploaded';
            }
            else {
                $sta_file_protein = 'Not uploaded';
            }
            $sta_file_protein .= '<br>';
        }

        #
        my($sta_file_dna) = '';
#        if ($modNewUserChr =~ /($MbgdUserGenomeCommon::MODE_USERCHR_genetab_proteinseq|$MbgdUserGenomeCommon::MODE_USERCHR_dnaseq)/i) {
            FileHandle->new(">>$fileDna"); # touch
#        }
        if (-e $fileDna) {
            if (-s $fileDna) {
                $sta_file_dna = 'Uploaded';
            }
            else {
                $sta_file_dna = 'Not uploaded';
            }
            $sta_file_dna .= '<br>';
        }


        my($uc_type);
        print qq{<fieldset><legend>Chromosome-$uc_id</legend>\n};
        print qq{<input type="submit" name="btn_del_chromosome" value="Delete this chromosome" onclick="delete_user_chromosome($uc_id);">\n};

        if (($sta_file_gbk        =~ /^Uploaded/i)
         || ($sta_file_gene_table =~ /^Uploaded/i)) {
            print qq{<input type="submit" name="btn_view_chromosome" value="View this chromosome" onclick="view_user_chromosome($uc_id);">\n};
        }

        print qq{<table width="100%" border>\n};
        print qq{<tr><th width="40%" colspan="2">Chromosome name</th>\n};
        print qq{    <td><input class="" type="text" name="name_uc_$uc_id" value="$uc_ref->{'name'}">\n};
        print qq{    </td>\n};
        print qq{</tr>\n};

        if ($sta_file_gbk) {
            print qq{<tr><th nowrap colspan=2>Data in GenBank Format</th>\n};
            print qq{    <td nowrap>$sta_file_gbk\n};
            print qq{        <label><input type="checkbox" name="multi_locus_uc_$uc_id" value="yes">multi-locus</label>\n};
            print qq{        <input type="file" name="file_gbk_uc_$uc_id" value="" size="30">\n};
            print qq{    </td>\n};
            print qq{</tr>\n};
        }

        #
        my($n_row) = 2;
        my($html_shape_len) = '';
        if ($modNewUserChr =~ /^$MbgdUserGenomeCommon::MODE_USERCHR_metagenome\:/) {
            $n_row = 0;
        }
        else {
            $html_shape_len = qq{
    <th nowrap>Chromosome Shape</th>
    <td nowrap>
        <input type="radio" name="shape_uc_$uc_id" value="circular" $sta_circular>Circular
        <input type="radio" name="shape_uc_$uc_id" value="linear" $sta_linear>Linear
    </td>
</tr>

<tr><th nowrap>Chromosome Length</th>
    <td nowrap>
        <input type="text" name="length_uc_$uc_id" value="$uc_ref->{'length'}" size="20">bp</td>
</tr>};


        }
        if ($sta_file_gene_table || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_genetab/i) {
            if ($modNewUserChr !~ /$MbgdUserGenomeCommon::MODE_USERCHR_metagenome/) {
                $n_row++;
            }
        }
        if ($sta_file_protein    || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_proteinseq/i) {
            $n_row++;
        }
        if ($sta_file_dna        || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_dnaseq/i) {
            $n_row++;
        }

        my($n_col) = 1;
        if ($n_row != 1) {
            print <<EOB;
<tr><th rowspan="$n_row" nowrap>Separated Files</th>
$html_shape_len
EOB
        }
        else {
            $n_col = 2;
        }

        if ($sta_file_gene_table || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_genetab/i) {
            if ($modNewUserChr !~ /$MbgdUserGenomeCommon::MODE_USERCHR_metagenome/) {

                print <<EOB;
<tr><th nowrap>Gene Information<br> (a tab-delimited file)</th>
    <td nowrap>$sta_file_gene_table
        <input type="file" name="file_gene_table_uc_$uc_id" value="" size="30">
    </td>
</tr>
EOB
            }
        }

        if ($sta_file_protein || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_proteinseq/i) {
            print <<EOB;
<tr><th colspan="$n_col" nowrap>Protein Sequences <br>(FASTA format)</th>
    <td colspan="$n_col" nowrap>$sta_file_protein
        <input type="file" name="file_protein_uc_$uc_id" value="" size="30"></td>
</tr>
EOB
        }

        if ($sta_file_dna || $modNewUserChr =~ /$MbgdUserGenomeCommon::MODE_USERCHR_dnaseq/i) {
            print <<EOB;
<tr><th nowrap>DNA Sequences <br>(FASTA format)</th>
    <td nowrap>$sta_file_dna
        <input type="file" name="file_dna_uc_$uc_id" value="" size="30"></td>
</tr>
EOB
        }
        print "</table>\n";
        print "</fieldset>\n";
    }
    my($uc_id_list) = join(',', @uc_id_list);
    print qq{<input type="hidden" name="uc_id_list" value="$uc_id_list">};

    #
    if ($spid ne '') {
        print <<EOB;
<!--
<select name="chromosome_data_type">
<optgroup label="Complete Data">
<option value="complete:genbank">GenBank</option>
<option value="complete:genetab_proteinseq">GeneTab + ProteinSeq</option>
</optgroup>

<optgroup label="Predict">
<option value="predict:dnaseq">DnaSeq</option>
</optgroup>
<optgroup label="metagenome">
<option value="metagenome:proteinseq">ProteinSeq</option>
</optgroup>
</select>
-->
<input type="submit" name="btn_more_user_chromosome" value="More chromosome" onclick="set_more_chromosome(1);">
<input type="hidden" name="more_chromosome" value="0">
EOB
    }

    print qq{<input type="button" name="" value="Upload data" onclick="update_user_genome();">\n};
    print qq{<input type="button" name="btn_close" value="Close" onclick="window.close();">\n};
    print qq{</form>};
    print qq{</body>};
    print qq{</html>};

    return;
}

###############################################################################
#
sub create_user_chromosome {
    my($spid) = shift;

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($dir_chr) = "$dir/data";
    my($uc_id);
    for ($uc_id = 1; ; $uc_id++) {
        my($file_chr)  = "$dir_chr/$uc_id.chromosome.txt";
        my($file_gene) = "$dir_chr/$uc_id.gene";
        if (! -e $file_chr) {
            FileHandle->new(">$file_chr");
            FileHandle->new(">$file_gene");
            last;
        }
    }

    return $uc_id;
}

###############################################################################
#
sub remove_user_chromosome {
    my($spid) = shift;
    my($uc_id) = shift;

    #
    my($dir) = sprintf("%s/species/%s/gm", $ENV{'MBGD_HOME'}, $spid);
    my($dir_chr) = "$dir/data";

    #
    my($dh) = DirHandle->new("$dir_chr");
    if (!$dh) {
        return;
    }

    foreach my$file ($dh->read()) {
        next if ($file !~ /^$uc_id\./);

        my($path) = "$dir_chr/$file";
        unlink($path);
    }

    return;
}

###############################################################################
#
sub remove_user_genome {
    my($spid) = shift;
    my($sp) = shift;

    #
    foreach my$name ($spid, $sp) {
        next if (!$name);

        my($dir) = sprintf("%s/species/%s", $ENV{'MBGD_HOME'}, $name);
        File::Path::rmtree($dir);
    }

    return;
}

###############################################################################
#
sub set_building_status {
    my($file_status) = shift;
    my($file_log) = shift;
    my($status) = shift;

    #
    my($fh_s) = FileHandle->new(">$file_status");
    if (!$fh_s) {
        print STDERR "ERROR :: Can not open $file_status($!)\n";
        return;
    }
    $fh_s->print($status);
    $fh_s->close();

    #
    my($fh_l) = FileHandle->new(">>$file_log");
    if (!$fh_l) {
        print STDERR "ERROR :: Can not open $file_log($!)\n";
        return;
    }
    $fh_l->print('#' x 60, "\n");
    $fh_l->print($status);
    $fh_l->close();

    return;
}

###############################################################################
#
sub extract_sequences {
    my($file_status) = shift;
    my($file_log) = shift;
    my($opt) = shift;
    my($dir_build) = "$ENV{'MBGD_HOME'}/build";
    my($dir_work)  = "$ENV{'MBGD_HOME'}/work";

    #
    my($infoSpec) = main::getInfoSpecTab($main::FILE_spidtab);
    my(@spid_list) = sort(keys(%{$infoSpec->{'SPID2DIR'}}));

    #
    my($status) = 'Extracting sequences';
    set_building_status($file_status, $file_log, $status);
    my($opt_spec) = "-SPEC=" . join(',', @spid_list);
    my($cmd) = "$dir_build/m0490SelectUpdSeq.pl $opt_spec";
    my($sta) = system("$cmd >> $file_log 2>&1");

    return $sta;
}

###############################################################################
#
sub exec_blast_dp {
    my($file_status) = shift;
    my($file_log) = shift;
    my($opt) = shift;
    my($dir_build) = "$ENV{'MBGD_HOME'}/build";
    my($dir_work)  = "$ENV{'MBGD_HOME'}/work";

    my($status) = '';
    set_building_status($file_status, $file_log, $status);
    my($cmd) = "$dir_build/m0500ExecSearch.pl";
    my($sta) = system("$cmd >> $file_log 2>&1");

    return $sta;
}

###############################################################################
#
sub insert_blast_dp_res {
    my($file_status) = shift;
    my($file_log) = shift;
    my($opt) = shift;
    my($dir_build) = "$ENV{'MBGD_HOME'}/build";
    my($dir_work)  = "$ENV{'MBGD_HOME'}/work";

    my($status) = '';
    set_building_status($file_status, $file_log, $status);
    my($cmd) = "$dir_build/m1000InsertSearchRes.pl";
    my($sta) = system("$cmd >> $file_log 2>&1");

    return $sta;
}

###############################################################################
#
sub build_bldp {
    my($file_status) = shift;
    my($file_log) = shift;
    my($opt) = shift;
    my($dir_build) = "$ENV{'MBGD_HOME'}/build";
    my($dir_work)  = "$ENV{'MBGD_HOME'}/work";

    my($status) = '';
    set_building_status($file_status, $file_log, $status);
    my($cmd) = "$dir_build/m1100BuildData.pl -all";
    my($sta) = system("$cmd >> $file_log 2>&1");

    return $sta;
}

###############################################################################
#
sub __build_homology_all {
    my($file_status) = shift;
    my($file_log) = shift;
    my($opt) = shift;

    #
    my($sta);

    #
    $sta = extract_sequences($file_status, $file_log, $opt);
    if ($sta) {
        return $sta;
    }

    #
    $sta = exec_blast_dp($file_status, $file_log, $opt);
    if ($sta) {
        return $sta;
    }

    #
    $sta = insert_blast_dp_res($file_status, $file_log, $opt);
    if ($sta) {
        return $sta;
    }

    #
    $sta = build_bldp($file_status, $file_log, $opt);
    if ($sta) {
        return $sta;
    }

    return;
}

###############################################################################
#
sub check_build_status {
    my($formOpt) = shift;

    #
    print "Content-type:text/html\n";
    print "\n";

    print <<EOB;
<html>
<head>
<link rel="stylesheet" type="text/css" href="/css/mbgd.css" />
<link rel="stylesheet" type="text/css" href="/css/recog_build.css" />
<script type="text/javascript" src="/js/common.js"></script>
<script type="text/javascript" src="/js/mbgd.js"></script>
<script type="text/javascript" src="/js/organism_selection.js"></script>
<script type="text/javascript" src="/js/create_tax.js"></script>
<script type="text/javascript" src="/js/recog_common.js"></script>
<script type="text/javascript" src="/js/recog_build.js"></script>
</head>
EOB

    #
    my($pid) = $formOpt->{'pid'};
    my($sta) = kill(0 => $pid);
    if (!$sta) {
        # Can not found $pid
        print "<body>\n";
        print "<h1>Done.</h1>\n";
        print "<a href=\"/htbin/recog_build/view_build_log.cgi?pid=$pid\" target=\"view_build_log\">View build log.</a>\n";
        print "</body>\n";
        print "</html>\n";
        return;
    }

    # FOUND $pid ===> Update status

    #
    my($file_status) = "$ENV{'MBGD_HOME'}/work/build_status.$pid";
    my($fh) = FileHandle->new("$file_status");
    my($status) = 'running';
    if ($fh) {
        $status = $fh->getline();
        $fh->close();
    }

    #
    my($tm) = $formOpt->{'tm'};
    my($d) = int($tm * 1.1);
    if ($d < 1) {
        $d = 1;
    }
    $tm += $d;
    if ($tm < 5) {
        $tm = 5;
    }
    elsif (60 < $tm) {
        $tm = 60;
    }

    #
    print <<EOB;
<html>
<head>
<script>
function reload_status_submit() {
    var doc = this.document;
    doc.frm_check_status.submit();

    return;
}
function reload_status(tm) {
    setTimeout("reload_status_submit();", tm * 1000);
}
</script>
</head>
<body onload="reload_status($tm);">
<h1>$status</h1>
<form name="frm_check_status" method="post" action="/htbin/recog_build/check_build_status.cgi">
<input type="hidden" name="pid" value="$pid">
<input type="hidden" name="tm" value="$tm">
</form>
</body>
</html>
EOB

    return;
}

###############################################################################
#
sub print_html_header_update {
    my($cgi) = shift;
    my($pid) = shift;
    my($t) = shift || 0;

    my($html_reload) = '';
    if (0 < $t) {
        my($uri) = qq{$cgi?pid=$pid&t=$t};
        $html_reload = qq{<meta http-equiv="refresh" content="$t;URL=$uri">};
    }

    print "Content-type: text/html\n";
    print "\n";

    print <<EOB;
<?xml version="1.0" encoding="EUC-JP">
<html>
<head>
$html_reload
<link rel="stylesheet" type="text/css" href="/css/mbgd.css" />
<link rel="stylesheet" type="text/css" href="/css/recog_build.css" />
<script type="text/javascript" src="/js/recog_common.js"></script>
<script type="text/javascript" src="/js/recog_build.js"></script>
</head>
EOB

    return;
}

###############################################################################
sub print_html_running_update {
    my($name) = shift;
    my($cgi) = shift;
    my($pid) = shift;
    my($t) = shift;
    my($file_status) = shift;

    if ($t < 5) {
        $t = 5;
    }

    my($html_status) = '';
    if ($file_status && -e "$file_status") {
        my($fh) = IO::File->new("$file_status");
        $html_status = join('<br>', $fh->getlines());
        $fh->close();
    }

    print_html_header_update($cgi, $pid, $t);

    print <<EOB;
<body>
<h1>$name</h1>
<h2>Running...</h2>
$html_status

<form method="post" action="/htbin/recog_build/stop_build.cgi">
<input type="hidden" name="pid" value="$pid">
<!--
<input type="submit" name="btn_stop" value="STOP">
-->
</form>
</body>
</html>
EOB

    return;
}

###############################################################################
sub print_html_finished_update {
    my($name) = shift;
    my($msg) = shift;

    print_html_header_update();

    my($html_msg) = '';
    if ($msg) {
        $html_msg = '<h3>Found error(s)</h3>' . $msg;
    }

    print <<EOB;
<h1>Update $name</h1>
<h2>Done.</h2>
$html_msg
</body>
</html>
EOB

    return;
}

###############################################################################
sub build_get_protseq {
    my($force_rebuild, @spid_list) = @_;

    if ($force_rebuild) {
    	unlink("$ENV{'MBGD_HOME'}/work/.m0450GetProteinSeq.end");
    }
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m0450GetProteinSeq.pl @spid_list";
    my($ret) = system("$cmd");

    return $ret;
}

###############################################################################
sub build_select_upd_seq {
    my($force_rebuild, @spid_list) = @_;

    if ($force_rebuild) {
    	unlink("$ENV{'MBGD_HOME'}/work/.m0490SelectUpdSeq.end");
    }
    my($spList) = join(',', @spid_list);
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m0490SelectUpdSeq.pl -SPEC=$spList";
    my($ret) = system("$cmd");

    return $ret;
}

###############################################################################
sub build_exec_homsearch {
    my($force_rebuild) = @_;
    if ($force_rebuild) {
    	unlink(<$ENV{'MBGD_HOME'}/work/tmphom.pid>);
    	unlink(<$ENV{'MBGD_HOME'}/work/.m05*>);
    }
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m0500ExecSearch.pl";
    my($ret) = system("$cmd");

    return $ret;
}

###############################################################################
sub build_insert_homsearch {
    my($force_rebuild) = @_;
    if ($force_rebuild) {
    	unlink(<$ENV{'MBGD_HOME'}/work/.m10*>);
    }
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m1000InsertSearchRes.pl";
    my($ret) = system("$cmd");

    return $ret;
}

###############################################################################
sub build_homology_db {
    my($force_rebuild) = @_;
    if ($force_rebuild) {
    	unlink(<$ENV{'MBGD_HOME'}/work/.m11*>);
    }
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m1100BuildData.pl -all";
    my($ret) = system("$cmd");

    return $ret;
}



###############################################################################
sub write_build_status {
    my($file_status) = shift;
    my($status) = shift;

    my($fh) = IO::File->new(">$file_status") || return;
    $fh->print($status);
    $fh->close();

    return;
}

###############################################################################
sub build_homology_all {
    my($force_rebuild) = shift;
    my($file_status) = shift;
    my($file_log) = shift;
    my(@spid_list) = @_;

    my($ret);

    #
# skip the step for extracting sequences
#    write_build_status($file_status, '');
#    $ret = build_get_protseq($force_rebuild, @spid_list);
#    if ($ret >> 8 != 0) {
#        return;
#    }
#
#    #
#    write_build_status($file_status, '');
#    $ret = build_select_upd_seq($force_rebuild, @spid_list);
#    if ($ret >> 8 != 0) {
#        return;
#    }

    #
    write_build_status($file_status, 'Executing BLAST/DP');
    $ret = build_exec_homsearch($force_rebuild);
    if ($ret >> 8 != 0) {
        return;
    }

    #
    write_build_status($file_status, 'Inserting homology results');
    $ret = build_insert_homsearch($force_rebuild);
    if ($ret >> 8 != 0) {
        return;
    }

    #
    write_build_status($file_status, 'Building homology DB');
    $ret = build_homology_db($force_rebuild);
    if ($ret >> 8 != 0) {
        return;
    }

    return $ret;
}

###############################################################################
sub update_base_cluster {
    my($proj_ref) = shift;
    my($user) = shift;

    #
    my($dir) = "$ENV{'MBGD_HOME'}/MBGD.tmp/base_cluster";
    File::Path::mkpath("$dir", 0, 0750);

    #
    my($cmd) = "$main::CMD_dbInsert_base_cluster -DBNAME=$main::DBNAME_RECOGWK -CLEAR";
    system("$cmd");

    #
    my(@proj_id_list) = $proj_ref->get_project_id_list();
    foreach my$proj_id (@proj_id_list) {
        my($dir_proj) = "$dir/$proj_id.d";
        File::Path::mkpath("$dir_proj", 0, 0750);

        my($ref) = $proj_ref->get_project($user, $proj_id);
        my($base_cluster_id)     = $ref->{'BASE_CLUSTER'}->{'ID'};
        my($base_cluster_server) = $ref->{'BASE_CLUSTER'}->{'server'};
        my($base_cluster_cgi);
        if ($base_cluster_server eq $main::URL_PUBLIC_SERVER) {
            $base_cluster_cgi = $base_cluster_server . '/htbin/getData';
        }
        else {
            $base_cluster_cgi = $base_cluster_server . '/htbin/RECOG/getData';
        }
        if ($base_cluster_id) {
            my($cmd);
	    my($file_base_cluster) = "$dir_proj/$base_cluster_id";
            # download base cluster
            my($url) = "$base_cluster_cgi"
                     . '?' . 'table=base_cluster'
                     . '&' . "base_cluster_id=$base_cluster_id";
            $cmd = "$main::CMD_curl -s '$url'> $file_base_cluster";
            system("$cmd");
	}
	foreach my $file (<$dir_proj/*>) {
	    my($fh) = FileHandle->new($file) || next;
	    my($l) = $fh->getline();
	    next if ($l !~ /^#spec/);
	    my($file_base_cluster) = $file;
	
            #
            $cmd = "$main::CMD_dbInsert_base_cluster -DBNAME=$main::DBNAME_RECOGWK -PROJECT_ID=$proj_id $file_base_cluster";
            system("$cmd");
        }
    }

    return 1; # return TRUE
}

###############################################################################
1;#
###############################################################################
