#include "Domain.h"
#include <sstream>
#include <algorithm>
#include <functional>

//class GeneName
string GeneName::getSpec(string name) {
	int idx = name.find_first_of(":");
	return name.substr(0, idx);
}

//class SeqReg
SeqReg::SeqReg (int _from, int _to) : from(_from), to(_to) {
}
void SeqReg::setData(int _from, int _to) {
	from = _from; to = _to;
}
void SeqReg::copyData(SeqReg *reg) {
	from = reg->from; to = reg->to;
}
bool SeqReg::overlapCheck(SeqReg *reg) {
	int len1 = getLength();
	int len2 = reg->getLength();
	int minlen, maxlen, ovlen;
	double ovlpratio = 0.6, ovlpratio2 = 0.3;
	const int minovlp = 50;
	const int MINLEN_LIMIT = 10;

	if (len1 < len2) {
		minlen = len1; maxlen = len2;
	} else {
		minlen = len2; maxlen = len1;
	}
	minlen = int(minlen * ovlpratio);
	maxlen = int(maxlen * ovlpratio2);
	if (maxlen < minovlp) maxlen = minovlp;
	if (minlen < MINLEN_LIMIT) minlen = MINLEN_LIMIT;
	ovlen = (minlen < maxlen) ? minlen : maxlen;

	return overlap(reg, ovlen);
}
bool SeqReg::overlapCheckStrict(SeqReg *reg) {
	int ovlen = 10;
	if (reg->getLength() < 50) ovlen = reg->getLength() / 5;
	return overlap(reg, ovlen);
}
bool SeqReg::overlap(SeqReg *reg, int ovlen) {
	return ( from+ ovlen - 1 <= reg->to && reg->from + ovlen - 1 <= to );
}
int SeqReg::overlapLen(SeqReg *reg) {
	return(std::min(to, reg->to) - std::max(from, reg->from) + 1);
}
void SeqReg::transform(SeqReg* reg1, SeqReg *reg2) {
	transform(reg1, reg2, this);
}
void SeqReg::transform(SeqReg* reg1, SeqReg *reg2, SeqReg *newreg) {
	newreg->from = transformPos(from, reg1, reg2);
	newreg->to = transformPos(to, reg1, reg2);
}
int SeqReg::transformPos(int pos, SeqReg* reg1, SeqReg *reg2) {
	if (pos < reg1->from) {
		return pos - reg1->from + reg2->from;
	} else if (pos > reg1->to) {
		return pos - reg1->to + reg2->to;
	} else {
		return (pos - reg1->from) *
			(reg2->to - reg2->from) / (reg1->to - reg1->from)
			+ reg2->from;
	}
}
void SeqReg::limitRegion(SeqReg* reg) {
	if (from < reg->from) {
		from = reg->from;
	}
	if (to >= reg->to) {
		to = reg->to;
	}
}

//class DomInfo
class CompareDomain : public binary_function<Domain*, Domain*, bool> {
public:
	bool operator()(Domain* a, Domain* b) {
		int cmp = a->getName().compare(b->getName());
		if (cmp < 0) {
			return true;
		} else if (cmp == 0) {
			return (a->getDomNum() < b->getDomNum());
		} else {
		}
		return false;
	}
};

DomInfo::DomInfo() : domList(), geneDomMap() {

}
void DomInfo::addDomain(Domain *dom){
	domList.push_back(dom);
}
void DomInfo::makeIndex() {
	sort(domList.begin(), domList.end(), CompareDomain() );
	string sp;

	for (int i = 0; i < domList.size(); i++) {
		GeneDomMap::iterator gInfoIter = geneDomMap.find(domList[i]->getName());
		sp = domList[i]->getSpec();
		if (! findSp(sp)) {
			spSet.insert(sp);
		}
		if ( gInfoIter == geneDomMap.end() ) {
			geneDomMap.insert(pair<string,int>(domList[i]->getName(), i));
		} else {
			
		}
	}
}
void DomInfo::setSp(string sp) {
	if (! findSp(sp)) {
		spSet.insert(sp);
	}
}
bool DomInfo::findSp(string sp) {
	return (spSet.find(sp) != spSet.end());
}
list<Domain *>* DomInfo::getOvlpDomains(Domain *dom) {
	list<Domain*>* ovlpDomList = new list<Domain*>();
	GeneDomMap::iterator gInfoIter = geneDomMap.find( dom->getName() );
	int lastIdx = domList.size();
	if (gInfoIter != geneDomMap.end()) {
		int idx = (*gInfoIter).second;
		const int MINOVLP=20; 	/*TEMPORAL*/
		int hitnum = 0, maxOvlp = MINOVLP /*minovlp*/, maxIdx = -1;
		while (idx < lastIdx &&
				domList[idx]->getName() == dom->getName()) {
/*
if (dom->getName() == "mta:MOTH_2429") {
	cout << "DDOM>>" << *dom << "; (" << idx << ") " << *(domList[idx]) << endl;
}
*/
			if (dom->overlapCheck(domList[idx])) {
				ovlpDomList->push_back( domList[idx] );
				hitnum++;
			} else if (hitnum == 0) {
				if (dom->overlapLen(domList[idx]) > maxOvlp) {
					maxIdx = idx;
				}
			}
			idx++;
		}
		/** TEMPORAL!! **/
		/* add maximally overlapped domain even if no domain satisfy the overlap condtion */
		if (hitnum == 0 && maxIdx >= 0) {
			ovlpDomList->push_back( domList[maxIdx] );
		}
	}
	return ovlpDomList;
}
void DomInfo::printDomains() {
	for (int i = 0; i < domList.size(); i++) {
		const string& domName = domList[i]->getDomName();
		int spos = domName.find(":");
		string spname = domName.substr(0, spos); 
		string genename = domName.substr(spos+1);
		cout << spname << " " << genename << " " << domList[i]->getLength() << endl;
	}
}

//class Domain

Domain::Domain(string& _name, int _from, int _to, int _domnum) :
	name(_name), SeqReg(_from, _to), domnum(_domnum), base_dom(NULL) {
}
void Domain::setData(string& _name, int _from, int _to, int _domnum) {
	name = _name; SeqReg::setData(_from, _to); domnum = _domnum;
	base_dom = NULL;
}
void Domain::setData(Domain *_base_dom, int _from, int _to, int _domnum) {
	base_dom = _base_dom;
	name = base_dom->getDomName();
	SeqReg::setData(_from, _to);
	domnum = _domnum;
}
void Domain::copyData(Domain *dom) {
	name = dom->name;
	SeqReg::copyData(dom);
	domnum = dom->domnum;
	base_dom = dom->base_dom; clustid = dom->clustid;
}
string Domain::toString() {
	ostringstream os;
//	os  << name << ":" << from << ":" << to;
	os  << name << ":" << domnum;
	return os.str();
}
Domain *Domain::getOrigDomain(Domain *in) {
	if (base_dom) {
		Domain *orig_dom = base_dom->getOrigDomain(in);
		string domname = orig_dom->getName();
		Domain *retdom = in;
		if (! in) {
			retdom = DomainPool::create();
		}
		retdom->setData(domname, orig_dom->from + from - 1,
			orig_dom->from + to - 1, orig_dom->domnum);
		return retdom;
	} else {
		if (in) {
			in->copyData(this);
		}
		return this;
	}
}
string Domain::getOrigName() {
	if (base_dom) {
		return base_dom->getOrigName();
	} else {
		return name;
	}
}
string Domain::getDomName() const {
	if (domnum > 0) {
		ostringstream ss;
		ss << name << "(" << domnum << ")";
		return ss.str();
	} else {
		return name;
	}
}
void Domain::placeOn(Domain* dom) {
	limitRegion(dom);
	translate(- (dom->from - 1) );
	base_dom = dom;
	name = base_dom->getDomName();
}
bool Domain::overlapAll(Domain *dom) {
	overlap(dom, getLength());
}
bool Domain::overlap(Domain *dom, int ovlen) {
	return ( getName() == dom->getName() && SeqReg::overlap(dom, ovlen) );
}

/*
void Domain::copySubClustIDs(Domain *dom) {
	copy( dom->subids.begin(), dom->subids.end(), subids.begin() );
}
*/


// class DomainPool
MemAlloc<Domain> DomainPool::objPool(100000, true, 10000);

Domain* DomainPool::create() {
	Domain *dom = objPool.allocate();
/*
if (objPool.allocSize() % 1000 == 0) {
cout << "AllocDom: " << objPool.allocSize() << endl;
}
*/
	dom->setClustID( 0 );
	return dom;
}
Domain* DomainPool::create(string& _name, int _from, int _to, int _domnum) {
	Domain *dom = objPool.allocate();
/*
if (objPool.allocSize() % 1000 == 0) {
cout << "AllocDom: " << objPool.allocSize() << endl;
}
*/
	dom->setData(_name, _from, _to, _domnum);
	dom->setClustID( 0 );
	return dom;
}
Domain* DomainPool::create(Domain *base_dom, int from, int to, int domnum) {
	Domain *dom = objPool.allocate();
	if (to == 0)  to = base_dom->getLength();
	dom->setData(base_dom, from, to, domnum);
	dom->setClustID( base_dom->getClustID() );
	return dom;
}
Domain* DomainPool::duplicate(Domain *dom0) {
	Domain *dom = objPool.allocate();
	dom->copyData(dom0);
	return dom;
}
void DomainPool::markCurrent() {
	objPool.markCurrent();
}
void DomainPool::resetToMarkedPos() {
	objPool.resetToMarkedPos();
}

ostream& operator<<(ostream& ostr, const Domain& domain) {
	return ostr << domain.getDomName() << " "
			<< domain.from << " " << domain.to;
/*
	return ostr << domain.name << "(" << domain.domnum << ") "
				<< domain.from << " " << domain.to;
*/
}
