#!/usr/bin/perl -s
use strict;
use File::Path;
use IO::Dir;
use IO::File;
use MBGD::DB;
use NqsAxes;

package GenePropAxes;

###############################################################################
# Functions:
#    get_spec_list()
#    get_gene_list($spec)
#    get_gene_info($spec, $name, $type)
#    get_clustid_list($tabid)
#    get_cluster_info($tabid, $clustid)
#    get_align_path($tabid, $align_name)
#    make_align_path($tabid, $align_name)
#    clear_gene_prop($prop_name, $spec)
#    set_gene_prop($prop_name, $spec, $name, $type, $val)
#    exec_job_list(@cmd_list)
###############################################################################


###############################################################################
# Type of GeneInformation for get_gene_info()
$GenePropAxes::GI_TAB   = 'gene_tab';
$GenePropAxes::GI_NTSEQ = 'ntseq';
$GenePropAxes::GI_AASEQ = 'aaseq';

###############################################################################
#
$GenePropAxes::TYPE_NUMERIC      = 'num';
$GenePropAxes::TYPE_NUMERIC_ENUM = 'num_enum';
$GenePropAxes::TYPE_STRING       = 'str';
$GenePropAxes::TYPE_STRING_ENUM  = 'str_enum';

###############################################################################
#
sub get_db {
    my($dbname) = shift;
    my($opt_ref) = shift;

    if ($opt_ref->{'DBNAME'} ne '') {
        $dbname = $opt_ref->{'DBNAME'};
    }
    my($db) = MBGD::DB->new($dbname);

    return $db;
}

###############################################################################
#
sub get_spec_list {
    my($opt_ref) = shift;

    my($db) = get_db($main::DBNAME_MBGD, $opt_ref->{'DBNAME'});
    my($sql) = "select sp from genome order by sp";
    my($sth) = $db->execute($sql);
    my(@spec_list);
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@spec_list, $ref->{'sp'});
    }

    return @spec_list;
}

###############################################################################
#
sub get_gene_list {
    my($spec) = shift;
    my($opt_ref) = shift;

    my($db) = get_db($main::DBNAME_MBGD, $opt_ref->{'DBNAME'});
    my($sql) = "select name from gene where sp=? order by name";
    my($sth) = $db->prepare($sql);
    $sth->execute($spec);
    my(@name_list);
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@name_list, $ref->{'name'});
    }

    return @name_list;
}

###############################################################################
#
sub get_gene_info {
    my($spec) = shift;
    my($name) = shift;
    my($type) = shift;
    my($opt_ref) = shift;

    my($db) = get_db($main::DBNAME_MBGD, $opt_ref->{'DBNAME'});
    my($sql);
    my($sth);
    if ($type =~ /^$GenePropAxes::GI_TAB$/i) {
        $sql = "select * from gene where sp=? and name=?";
        $sth = $db->prepare($sql);
        $sth->execute($spec, $name);
        if ($sth->rows() != 0) {
            my($ref) = $sth->fetchrow_hashref();
            return $ref;
        }
    }
    elsif ($type =~ /^$GenePropAxes::GI_NTSEQ$/i) {
        $sql = "select gs.seq from gene g, geneseq gs  where g.sp=? and g.name=? and g.ntseq=gs.id";
        $sth = $db->prepare($sql);
        $sth->execute($spec, $name);
        if ($sth->rows() != 0) {
            my($ref) = $sth->fetchrow_hashref();
            return $ref->{'seq'};
        }
    }
    elsif ($type =~ /^$GenePropAxes::GI_AASEQ$/i) {
        $sql = "select ps.seq from gene g, proteinseq ps  where g.sp=? and g.name=? and g.aaseq=ps.id";
        $sth = $db->prepare($sql);
        $sth->execute($spec, $name);
        if ($sth->rows() != 0) {
            my($ref) = $sth->fetchrow_hashref();
            return $ref->{'seq'};
        }
    }

    return;
}

###############################################################################
#
sub get_clustid_list {
    my($tabid) = shift;
    my($opt_ref) = shift;

    my($db) = get_db($main::DBNAME_RECOG, $opt_ref->{'DBNAME'});

    #
    my(@clustid_list);
    my($tab) = "cluster_func_clust2sql_$tabid";
    my($sql) = "select clustid from $tab";
    my($sth) = $db->execute($sql);
#print STDERR "SQL :: $sql\n";
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@clustid_list, $ref->{'clustid'});
    }

    return @clustid_list;
}

###############################################################################
#
sub get_cluster_info {
    my($tabid) = shift;
    my($clustid) = shift;
    my($opt_ref) = shift;

    my($db) = get_db($main::DBNAME_RECOG, $opt_ref->{'DBNAME'});

    #
    my(@clustent_list);
    my($tab) = "cluster_domclust_cache_$tabid";
    my($where) = "clustid=?";
    my($order) = "sp, name";
    my($sql) = "select clustid from $tab where $where order by $order";
    my($sth) = $db->prepare($sql);
    $sth->execute($clustid);
#print STDERR "SQL :: $sql\n";
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@clustent_list, $ref);
    }

    return @clustent_list;
}

###############################################################################
#
sub get_align_path {
    my($tabid) = shift;
    my($align_name) = shift;
    my($opt_ref) = shift;

    my($path) = sprintf("%s/database/clusters/%s/%s", $ENV{'MBGD_HOME'},
                                                      $tabid,
                                                      $align_name);

    return $path;
}

###############################################################################
#
sub make_align_path {
    my($tabid) = shift;
    my($align_name) = shift;
    my($opt_ref) = shift;

    my($path) = get_align_path($tabid, $align_name, $opt_ref);
    if (!-e $path) {
        File::Path::mkpath($path, 0, 0750);
    }

    return $path;
}

###############################################################################
#
sub clear_gene_prop {
    my($prop_name) = shift;
    my($spec) = shift;
    my($opt_ref) = shift;

    #
    close_gene_prop();

    if (!$prop_name) {
        return;
    }
    if (!$spec) {
        return;
    }

    my($dir) = sprintf("%s/species/%s/gm/data", $ENV{'MBGD_HOME'}, $spec);
    File::Path::mkpath($dir, 0, 0750) if (! -e $dir);
    my($dh) = IO::Dir->new($dir);
    if (!$dh) {
        print STDERR "ERROR :: Can not open $dir($!)\n";
        return;
    }

    while (my$file=$dh->read()) {
        if ($file =~ /\.$prop_name$/) {
            print STDERR "DBG :: unlink() :: $dir/$file\n";
            unlink("$dir/$file");
        }
    }

    return;
}

###############################################################################
#
$GenePropAxes::SAVE_filename_gene_prop = '';
$GenePropAxes::SAVE_fh_gene_prop;
sub set_gene_prop {
    my($prop_name) = shift;
    my($spec) = shift;
    my($name) = shift;
    my($type) = shift;
    my($val) = shift;
    my($opt_ref) = shift;

    my($dir) = sprintf("%s/species/%s/gm/data", $ENV{'MBGD_HOME'}, $spec);
    File::Path::mkpath($dir, 0, 0750) if (! -e $dir);

    my($file) = "$spec.$prop_name";
    if ("$dir/$file" ne $GenePropAxes::SAVE_filename_gene_prop) {
        if ($GenePropAxes::SAVE_fh_gene_prop) {
            $GenePropAxes::SAVE_fh_gene_prop->close();
        }
        $GenePropAxes::SAVE_filename_gene_prop = "$dir/$file";
        $GenePropAxes::SAVE_fh_gene_prop = IO::File->new(">>$dir/$file");
        print STDERR "DBG :: open() :: $dir/$file\n";
    }

    #
    my(@dat);
    push(@dat, $spec);
    push(@dat, $name);
    push(@dat, $prop_name);
    push(@dat, $type);
    if ($type =~ /^n/i) {
        push(@dat, $val);
        push(@dat, '\N');
    }
    else {
        push(@dat, '\N');
        push(@dat, $val);
    }
    $GenePropAxes::SAVE_fh_gene_prop->print(join("\t", @dat), "\n");

    return;
}

###############################################################################
#
sub close_gene_prop {
    undef($GenePropAxes::SAVE_filename_gene_prop);
    undef($GenePropAxes::SAVE_fh_gene_prop);

    return;
}

###############################################################################
#
sub exec_job_list {
    my(@job_list) = @_;

    my($nqs_ref);
    if ($main::QSUB) {
        $nqs_ref = NqsAxes->new();
        $nqs_ref->set_max_job($main::NUM_cpu);
    }

    foreach my$job (@job_list) {
        my($file_stdout) = '/dev/null';
        my($file_stderr) = '/dev/null';
        if ($main::QSUB) {
            my(@cmd_list);
            push(@cmd_list, "#\$ -S /bin/bash");
            push(@cmd_list, "#\$ -o $file_stdout");
            push(@cmd_list, "#\$ -e $file_stderr");
            push(@cmd_list, "#\$ -v MBGD_HOME,PERLLIB");
            push(@cmd_list, $job);
            $nqs_ref->qsub_cmd(@cmd_list);
        }
        else {
            system("($job) >$file_stdout 2>$file_stderr");
        }
    }

    #
    if ($main::QSUB) {
        $nqs_ref->wait_all_job();
    }

    return;
}

###############################################################################
#
sub print_options {
    my($ref) = shift;

    #
    my($descr) = '';
    foreach my$param (@{$ref->{'LIST_PARAM'}}) {
        if ($param->{'REQUIRE'} !~ /^T$/i) {
           print "[";
        }
        print '-' . $param->{'KEY'};
        if ($param->{'EXAMPLE'} !~ /^\s*$/) {
           print "='" . $param->{'EXAMPLE'} . "'";
        }
        if ($param->{'REQUIRE'} !~ /^T$/i) {
           print "]";
        }
        print " ";

        #
        $descr .= "    " .  $param->{'KEY'} .  " : " . $param->{'EXPLAIN'};
        if ($param->{'DEFAULT'} !~ /^\s*$/) {
           $descr .= ' [default : ' . $param->{'DEFAULT'} . ']';
        }
        $descr .= "\n";
    }
    print "\n";

    #
    print $ref->{'DESCRIPTION'} . "\n";

    #
    print $descr;

    return;
}

###############################################################################
#
sub print_usage {
    my($ref) = shift;

    print "Usage :: $0 ";

    print_options($ref);

    return;
}

###############################################################################
#
sub print_params {
    my($ref) = shift;

    foreach my$param (@{$ref->{'LIST_PARAM'}}) {
        print $param->{'KEY'};
        if ($param->{'EXAMPLE'} !~ /^\s*$/) {
           print "=" . $param->{'EXAMPLE'};
        }
        if ($param->{'DEFAULT'} !~ /^\s*$/) {
           print " [DEFAULT:" . $param->{'DEFAULT'} . "]";
        }
        print "\n";
    }

    return;
}

###############################################################################
1;#
###############################################################################
