#!/usr/bin/perl -s

####################################################
package Graph;
####################################################
$Graph::Delim = " ";
sub new {
	my($class) = @_;
	my($this) = {};
	bless $this, $class;
	$this->set_edge_weight_tag('weight');
	return $this;
}
sub convGraph {
	my($class, $hash) = @_;
	my($this) = $class->new;
	foreach $a (keys %{$hash}) {
		foreach $b (keys %{$hash->{$a}}) {
			$this->add($a,$b,$hash->{$a}->{$b});
		}
	}
	$this;
}
sub dup {
	my($this) = @_;
	my($newg) = Graph->new;
	foreach my $e ($this->edges) {
		($a,$b) = &get_edge_ends($e);
		$newg->add($a,$b,$this->edge_weight($a,$b));
	}
	foreach my $n ($this->nodes) {
		$newg->set_node_weight($a, $this->node_weight($a));
	}
	$newg;
}
sub set_edge_weight_tag {
	my($this, $tag) = @_;
	$this->{edge_weight_tag} = $tag;
}
sub add {
	my($this,$a,$b,$w) = @_;
	$w = 1 if (! defined $w);
	$this->{dir}->{$a}->{$b} = $w;
	$this->{rev}->{$b}->{$a} = $w;
	$this->{nodes}->{$a} = 1 if (! defined $this->{nodes}->{$a});
	$this->{nodes}->{$b} = 1 if (! defined $this->{nodes}->{$b});
}
sub node_exist {
	my($this, $a) = @_;
	defined $this->{nodes}->{$a};
}
sub edge_exist {
	my($this, $a, $b) = @_;
	defined $this->{dir}->{$a}->{$b};
}
sub edge_data {
	my($this, $a, $b) = @_;
	$this->{dir}->{$a}->{$b};
}
sub set_edge_weight {
	my($this, $a, $b, $w, $tag) = @_;
	if (ref $this->{dir}->{$a}->{$b}) {
		if ($tag) {
			$this->{dir}->{$a}->{$b}->{$tag} = $w;
		} else {
			$this->{dir}->{$a}->{$b}->{$this->{edge_weight_tag}} = $w;
		}
	} else {
		$this->{dir}->{$a}->{$b} = $w;
	}
}
sub edge_weight {
	my($this, $a, $b, $tag) = @_;
	if (ref $this->{dir}->{$a}->{$b}) {
		if ($tag) {
			$this->{dir}->{$a}->{$b}->{$tag};
		} else {
			$this->{dir}->{$a}->{$b}->{$this->{edge_weight_tag}};
		}
	} else {
		$this->{dir}->{$a}->{$b};
	}
}
sub get_edge_ends {
	my($e) = @_;
	return split(/$Graph::Delim/, $e);
}
sub set_node_weight {
	my($this, $a, $wt) = @_;
	$wt = 0 if (! $wt);
	$this->{nodes}->{$a} = $wt;
}
sub node_weight {
	my($this, $a) = @_;
	return $this->{nodes}->{$a};
}
sub delete_node {
	my($this,$a) = @_;
	foreach $bb (keys %{$this->{dir}->{$a}}) {
		delete($this->{rev}->{$bb}->{$a});
	}
	foreach $bb (keys %{$this->{rev}->{$a}}) {
		delete($this->{dir}->{$bb}->{$a});
	}
	delete($this->{dir}->{$a});
	delete($this->{rev}->{$a});
	delete($this->{nodes}->{$a});
}
sub delete_node_shrink {
	my($this,$a) = @_;
	my(@in) = $this->in($a);
	my(@out) = $this->out($a);
	my($prev,$next);
	foreach $prev (@in) {
		foreach $next (@out) {
			$this->add($prev, $next, 1);
		}
	}
	$this->delete_node($a);
}
sub delete_edge {
	my($this, $a, $b) = @_;
	delete($this->{dir}->{$a}->{$b});
	delete($this->{rev}->{$b}->{$a});
}
sub nodes {
	my($this, %opt) = @_;
	if ($opt{sort_by_weight} < 0) {
		sort {$this->{nodes}->{$a}<=>$this->{nodes}->{$b}}
				(keys %{$this->{nodes}});
	} elsif ($opt{sort_by_weight} > 0) {
		sort {$this->{nodes}->{$b}<=>$this->{nodes}->{$a}}
				(keys %{$this->{nodes}});
	} else {
		sort (keys %{$this->{nodes}});
	}
}
sub edges {
	my($this) = @_;
	my(@list);
	foreach $a (sort keys %{$this->{dir}}) {
		foreach $b (sort keys %{$this->{dir}->{$a}}) {
			push(@list, join($Graph::Delim, $a, $b));
		}
	}
	@list;
}
sub print_edges {
	my($this, $outfile) = @_;
	$outfile = ">&STDOUT" if (! $outfile);
	$outfile = ">$outfile" if ($outfile !~ /^>/);
	open(OUT, $outfile) || die;
	print OUT "digraph G {\n";
	foreach $a (sort keys %{$this->{dir}}) {
		foreach $b (sort keys %{$this->{dir}->{$a}}) {
			print OUT qq{"$a" -> "$b";\n};
		}
	}
	print OUT "}\n";
	close(OUT);
}
sub in {
	my($this, $node) = @_;
	keys %{$this->{rev}->{$node}};
}
sub out {
	my($this, $node) = @_;
	keys %{$this->{dir}->{$node}};
}
####################################################
package UndirectedGraph;
####################################################
@ISA = qw(Graph);

sub add {
	my($this,$a,$b,$w) = @_;
	$w = 1 if (! defined $w);
	$this->{dir}->{$a}->{$b} = $this->{dir}->{$b}->{$a} = $w;
	$this->{nodes}->{$a} = 1 if (! defined $this->{nodes}->{$a});
	$this->{nodes}->{$b} = 1 if (! defined $this->{nodes}->{$b});
}
sub edges {
	my($this) = @_;
	my(@list);
	foreach $a (sort keys %{$this->{dir}}) {
		foreach $b (sort keys %{$this->{dir}->{$a}}) {
			next if ( ($a cmp $b) > 0 );
			push(@list, join($Graph::Delim, $a, $b));
		}
	}
	@list;
}
sub delete_edge {
	my($this, $a, $b) = @_;
	delete($this->{dir}->{$a}->{$b});
	delete($this->{dir}->{$b}->{$a});
}
sub print_edges {
	my($this, $outfile) = @_;
	$outfile = ">&STDOUT" if (! $outfile);
	$outfile = ">$outfile" if ($outfile !~ /^>/);
	open(OUT, $outfile) || die;
	print OUT "graph G {\n";
	foreach $a (sort keys %{$this->{dir}}) {
		foreach $b (sort keys %{$this->{dir}->{$a}}) {
			next if ( ($a cmp $b) > 0 );
			print OUT qq{"$a" -- "$b";\n};
		}
	}
	print OUT "}\n";
	close(OUT);
}

####################################################
package Graph::FeedbackSet;
####################################################
# the algorithm by Levy and Low (J. Algorithm 9:470 1988)
####################################################
sub new {
	my($class, $graph, %opt) = @_;
	my($this) = {};
	if (! $opt{replace}) {
		## copy to save the original graph
		$graph = $graph->dup;
	}
	if ($opt{edge}) {
		convert_graph($graph);
	}
	$this->{graph} = $graph;
	bless $this, $class;
}
sub contraction {
	my($this) = @_;
	my($graph) = $this->{graph};
	my(@nodes);
	do {
		while ($this->contraction_check()) {
			## loop
		}
		if (@nodes = $graph->nodes) {
			($n) = sort { $graph->node_weight($a) <=>
					$graph->node_weight($b) } @nodes;
			$graph->delete_node($n);
			$this->{FS2}->{$n} = 1;
		}
	} while ( @nodes = $graph->nodes );
}
sub contraction_check {
	my($this) = @_;
	my($n);
	my($modcnt);
	my($graph) = $this->{graph};
	foreach $n (sort { $graph->node_weight($b)<=>$graph->node_weight($a) }
					$graph->nodes) {
		my @in = $graph->in($n);
		my @out = $graph->out($n);
#print STDERR ">", scalar(@in),' ',scalar(@out),' ', "\n";
		if ($graph->edge_exist($n,$n)) {
			&REM("loop: $n");
			$graph->delete_node($n);
			$this->{FS}->{$n} = 1;
			$modcnt++;
		} elsif (@in == 0 || @out == 0) {
			&REM("in0out0: $n");
			$graph->delete_node($n);
			$modcnt++;
		} elsif (@in == 1) {
			&REM("in1: $n");
			$graph->delete_node_shrink($n);
			$modcnt++;
		} elsif (@out == 1) {
			&REM("out1: $n");
			$graph->delete_node_shrink($n);
			$modcnt++;
		}
		last if ($modcnt);
	}
	$modcnt;
}
sub feedback_set {
	my($this) = @_;
	my @FS1 = keys (%{ $this->{FS} });
	my @FS2 = keys (%{ $this->{FS2} });
	(\@FS1, \@FS2);
}
sub REM {
	my($mes) = @_;
	print "$mes\n" if ($DEBUG);
}
sub convert_graph {
	my($g, $opt) = @_;
	my($num) = 1;
	foreach $e ($g->edges) {
		my($a,$b) = Graph::get_edge_ends($e);
		my($newnode) = "E$a:$b";
		$g->add($a, $newnode);
		$g->add($newnode, $b);
		my $weight = $g->edge_weight($a,$b);
		$g->set_node_weight($newnode, $weight);
		$g->delete_edge($a,$b);
	}
	foreach $n ($g->nodes) {
		if ($n !~ /^E/) {
			$g->delete_node_shrink($n);
		}
	}
}
sub print_feedback_set {
	my($this) = @_;
	my($FS1,$FS2) = $this->feedback_set;
	print "FS1: ", join(' ', @{$FS1}),"\n";
	print "FS2: ", join(' ', @{$FS2}),"\n";
	my @nodes = $this->{graph}->nodes;
	if (@nodes) {
		"Nodes: ", join(' ', @noes), "\n";
	}
}

#######################################################
package Graph::SCC;
#######################################################
# Strongly Connected Component
#######################################################
sub scc {
	my($class, $graph) = @_;

#	my($VisitOut1,$VisitOut2) = ({},{});
#	my($SCC) = {};
#	&dfs($graph->{dir}, {}, $VisitOut1);
#	@nodelist = sort {$VisitOut1->{$b}<=>$VisitOut1->{$a}} $graph->nodes;
#	&dfs($graph->{rev}, {}, $VisitOut2, $SCC, \@nodelist);

	my($dfs) = Graph::DFS->new($graph);
	my ($In1, $Out1) = $dfs->dfs('dir');
	my @nodelist = sort {$Out1->{$b}<=>$Out1->{$a}} $graph->nodes;

	my ($In2, $Out2, $SCC) = $dfs->dfs('rev', \@nodelist);

	my(@newNodes);
	my($newGraph,$newRevGraph) = ({}, {});
	foreach $n (sort {$SCC->{$a}<=>$SCC->{$b}} (keys %{$SCC})) {
##		print STDERR ">>>$n $SCC->{$n}\n";
		push(@{$newNodes[$SCC->{$n}]}, $n);
	}
#	foreach $n (1..$#newNodes) {
#		print STDERR "$n: ", join(' ', @{$newNodes[$n]}),"\n";
#	}
	foreach my $n1 (keys %{$graph}) {
		foreach $n2 ($graph->{$n1}) {
			$newGraph->{ $SCC->{$n1} }->{ $SCC->{$n2} } =
			$newRevGraph->{ $SCC->{$n2} }->{ $SCC->{$n1} } =
				$graph->{$n1}->{$n2};
		}
	}
	($SCC, \@newNodes, $newGraph, $newRevGraph);
}
#######################################################
package Graph::DFS;
#######################################################
sub new {
	my($class, $graph) = @_;
	my($this) = {};
	$this->{graph} = $graph;
	return bless $this, $class;
}
sub dfs {
	my($this, $dir, $nodelist) = @_;
	local($Graph) = $this->{graph}->{$dir};
	local($VisitIn, $VisitOut, $Clust) = ({},{},{});
	local($CountIn, $CountOut);
	my @nodelist;
	local $CID;

	if ($nodelist) {
		@nodelist = @{$nodelist};
	} else {
		@nodelist = keys %{$Graph};
	}

	foreach my $n1 (@nodelist) {
		if ($Clust) {
			next if ($Clust->{$n1});
			$CID++;
		}
		&dfs_sub($n1);
	}
	($VisitIn, $VisitOut, $Clust);
}
sub dfs_sub {
	my($n1) = @_;
	return if ($VisitIn->{$n1});
	$Clust->{$n1} = $CID if ($CID);

	$VisitIn->{$n1} = ++$CountIn;
	foreach my $n2 (keys %{$Graph->{$n1}}) {
		&dfs_sub($n2);
	}
	$VisitOut->{$n1} = ++$CountOut;
}
#######################################################
package Graph::MST;
###################################################
# minimum spanning tree
###################################################
sub kruskal {
	my($class, $graph, %opt) = @_;
	my(@Edges, @Sorted, %Cluster);
	my($Enum, $Nnum);
	my($tree) = UndirectedGraph->new;
	my($Cluster) = DisjointSet->new;
	my($dist);
	foreach my $n ($graph->nodes) {
		$Cluster->add_elem($n);
		$Nnum++;
	}
	foreach my $e ($graph->edges) {
		my($a,$b) = Graph::get_edge_ends($e);
		my $w = $graph->edge_weight($a,$b,$opt{'weight_tag'});
		$w *= -1 if ($opt{'max'});
		push(@Edges, {n1=>$a,n2=>$b,w=>$w});
	}
	@Sorted = sort {$a->{w} <=> $b->{w}} @Edges;
	foreach my $e (@Sorted) {
		next if ($Cluster->find_set($e->{n1})
				eq $Cluster->find_set($e->{n2}));
		$tree->add($e->{n1},$e->{n2}, $e->{w});
		$Cluster->union($e->{n1}, $e->{n2});
		$dist += $e->{w};
	}
	if ($opt{with_dist}) {
		$dist *= -1 if ($opt{max});
		return ($tree, $dist);
	} else {
		return $tree;
	}
}
#######################################################
package DisjointSet;
sub new {
	my($class) = @_;
	return bless {}, $class;
}
sub add_elem {
	my($this, $elem) = @_;
	$this->{parent}->{$elem} = $elem;
	$this->{rank}->{$elem} = 0;
}
sub union {
	my($this, $elem1, $elem2) = @_;
	$root1 = $this->find_set($elem1);
	$root2 = $this->find_set($elem2);
	$this->link($root1, $root2);
}
sub link {
	my($this, $elem1, $elem2) = @_;
	if ($this->{rank}->{$elem1} >= $this->{rank}->{$elem2}) {
		$this->{parent}->{$elem2} = $this->{parent}->{$elem1};
		if ($this->{rank}->{$elem1} == $this->{rank}->{$elem2}) {
			$this->{rank}->{$elem2} ++;
		}
	} else {
		$this->{parent}->{$elem1} = $this->{parent}->{$elem2};
	}
}
sub find_set {
	my($this, $elem)= @_;
	my($parent) = $this->{parent}->{$elem};
	return $parent if ($parent eq $elem);
	my($root) = $this->find_set($parent);
	$this->{parent}->{$elem} = $root;
	return $root;
}

#######################################################
package Graph::SlinkClust;
###################################################
# single linkage clustering (connected components)
###################################################
sub new {
	my($class, $graph) = @_;
	my($this) = {};
	bless $this, $class;
	$this->{graph} = $graph;
	$this->{clust} = {};
	$this;
}
sub slinkclust {
	my($this) = @_;
	my($posclustid) = 1;

	foreach $i ($this->{graph}->nodes) {
		if ($this->traverse($i, $posclustid) >= 0) {
			$posclustid++;
		}
	}
	$this->{clustnum} = $posclustid - 1;
}
sub traverse {
	my($this, $i, $posclustid) = @_;
	my($j);
	return -1 if ($this->{clust}->{$i});
	$this->{clust}->{$i} = $posclustid;
	foreach $j ( $this->{graph}->out($i) ) {
		$this->traverse($j, $posclustid);
	}
	return 0;
}

####################################################
package main;
####################################################
if (__FILE__ eq $0) {
#	$g = UndirectedGraph->new;
	$g = Graph->new;
	while(<>){
		chomp;
		($a,$b,$w) = split;
		$g->add($a,$b,$w);
	}
#	($tree,$dist) = Graph::MST->kruskal($g, with_dist=>1);
#	$tree->print_edges;
#	print "dist=$dist\n";
#
#	if ($edge) {
#		%opt = (edge => 1);
#	}
	
	print "SCC\n";
	($SCC) = Graph::SCC->scc($g);
	foreach $n (sort {$SCC->{$a}<=>$SCC->{$b}} keys %{$SCC}) {
		print "scc>$n: $SCC->{$n}\n";
	}

	$fas = Graph::FeedbackSet->new($g, %opt);
	$fas->contraction;
	if ($print_graph) {
		$g->print_edges;
	} else {
		($FS1,$FS2) = $fas->feedback_set;
		print "FS1: ", join(' ', @{$FS1}),"\n";
		print "FS2: ", join(' ', @{$FS2}),"\n";
		@nodes = $g->nodes;
		if (@nodes) {
			"Nodes: ", join(' ', @noes), "\n";
		}
	}
}

1;


