#!/usr/bin/perl -s
use strict;
use FileHandle;
use IO::Dir;
use IO::File;
use HTTP::Request;
use MBGD::DB;
package RECOG::RecogCommon;

###############################################################################
#
sub set_default_recog_conf {
    my($conf_ref) = shift || {};
    my($is_update) = 0;
    my($key);

    #
    $key = 'max-species-domclust';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 300;
        $is_update++;
    }
    $key = 'max-species-domclust-bidirec';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 300;
        $is_update++;
    }
    $key = 'max-species-domclust-unidirec';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 300;
        $is_update++;
    }

    # URL of COG/KEGG/TIGR
    $key = 'URL_xref_cog';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 'http://www.ncbi.nlm.nih.gov/COG/grace/wiew.cgi?';
        $is_update++;
    }
    $key = 'URL_xref_kegg';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 'http://www.genome.jp/dbget-bin/www_bget?ko+';
        $is_update++;
    }
    $key = 'URL_xref_tigr';
    if (! exists($conf_ref->{"$key"})) {
        $conf_ref->{"$key"} = 'http://cmr.tigr.org/cgi-bin/CMR/HmmReport.cgi?hmm_acc=';
        $is_update++;
    }

    return $is_update;
}

###############################################################################
#
sub read_recog_conf {
    my($fileRecogConf) = shift || "$ENV{'MBGD_HOME'}/etc/recog.conf";
    my($conf_ref) = {};

    #
    my($fh) = FileHandle->new("$fileRecogConf") || die("Can not open $fileRecogConf($!)");
    while(my$line = $fh->getline()) {
        $line =~ s#[\r\n]*$##;
        my($k, $v) = split(/\t/, $line);

        $conf_ref->{"$k"} = $v;
    }
    $fh->close();

    #
    my($sta) = set_default_recog_conf($conf_ref);
    if ($sta) {
        write_recog_conf($fileRecogConf, $conf_ref);
    }

    return $conf_ref;
}

###############################################################################
#
sub write_recog_conf {
    my($fileRecogConf) = shift || "$ENV{'MBGD_HOME'}/etc/recog.conf";
    my($conf_ref) = shift;

    #
    my($fh) = FileHandle->new(">$fileRecogConf") || die("Can not open $fileRecogConf($!)");
    foreach my$key (sort(keys(%{$conf_ref}))) {
        $fh->print($key, "\t", $conf_ref->{"$key"}, "\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub getSpeciesByTabid {
    my($tabid) = shift;

    my($dbname) = $main::DBNAME_RECOG;
    my($db) = MBGD::DB->new($dbname);

    my($tabname) = 'cluster_tables_idx';
    my($sql) = "select * from $tabname where clusterID='$tabid'";
    my($sth) = $db->execute("$sql");
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();
    my($splist) = ($ref->{'cmd'} =~ /\-SPEC\=(\S+)/);

    return $splist;
}

###############################################################################
#
sub getSpeciesByType {
    my($dbname) = shift;
    my($type) = shift;

    my(@spec_list);
    if (!$dbname) {
        $dbname = $main::DBNAME_MBGD;
    }
    my($category) = 'genome';
    my($name) = 'species_' . $type;

    #
    my($db) = MBGD::DB->new($dbname);
    my($tab) = 'attribute';
    my($sql) = "select * from $tab where category=? and name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($category, $name);
    while (my$ref = $sth->fetchrow_hashref()) {
        push(@spec_list, split(',', $ref->{'value'}));
        last;
    }

    return @spec_list;
}

###############################################################################
#
sub get_http_host {
    my($name, $port) = split(/\:/, $ENV{'HTTP_HOST'});
    if (! $port) {
        $port = 80;
    }

    my($ref) = {};
    $ref->{'NAME'} = $name;
    $ref->{'PORT'} = $port;

    return $ref;
}

###############################################################################
#
sub rebuild_domclust_options_cmd {
    my($cmd) = shift;
    my($opt_sel_ref) = shift;
    my($opt_dom_ref) = shift;

    my($ref) = {};
    $ref->{'cmd'} = $cmd;

    my($species)        = ($ref->{'cmd'} =~ /\-SPEC\=(\S+)/);
    my(@spec_list) = split(/,/, $species);
    my($out_group)      = ($ref->{'cmd'} =~ /\-Ooutgroup\=(\S+)/);

    # for select
    if ($ref->{'cmd'} =~ /\-SCORE\=(\S+)/) {
        $opt_sel_ref->{'-SCORE'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-EVAL\=(\S+)/) {
        $opt_sel_ref->{'-EVAL'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-PAM\=(\S+)/) {
        $opt_sel_ref->{'-PAM'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-IDENT\=(\S+)/) {
        $opt_sel_ref->{'-IDENT'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-BESTHIT\=(\S+)/) {
        $opt_sel_ref->{'-BESTHIT'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-RATIOCUT\=(\S+)/) {
        $opt_sel_ref->{'-RATIOCUT'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-di/) {
        $opt_sel_ref->{'-di'} = '';
    }
    else {
        delete($opt_sel_ref->{'-di'});
    }

    # for domclust
    if ($ref->{'cmd'} =~ /\-S/) {
        $opt_dom_ref->{'-S'} = '';
    }
    else {
        delete($opt_dom_ref->{'-S'});
    }
    if ($ref->{'cmd'} =~ /\-d/) {
        $opt_dom_ref->{'-d'} = '';
    }
    else {
        delete($opt_dom_ref->{'-d'});
    }
    if ($ref->{'cmd'} =~ /\-c([\d\.]+)/) {
        $opt_dom_ref->{'-c'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-ci([\d\.]+)/) {
        $opt_dom_ref->{'-ci'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-m([\d\.]+)/) {
        $opt_dom_ref->{'-m'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-mr([\d\.]+)/) {
        $opt_dom_ref->{'-mr'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-C([\d\.]+)/) {
        $opt_dom_ref->{'-C'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-V([\d\.]+)/) {
        $opt_dom_ref->{'-V'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-n([\d\.]+)/) {
        $opt_dom_ref->{'-n'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-ne([\d\.]+)/) {
        $opt_dom_ref->{'-ne'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-p([\d\.]+)/) {
        $opt_dom_ref->{'-p'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-ai([\d\.]+)/) {
        $opt_dom_ref->{'-ai'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-ao([\d\.]+)/) {
        $opt_dom_ref->{'-ao'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-HO/) {
        $opt_dom_ref->{'-HO'} = '';
    }
    elsif ($ref->{'cmd'} =~ /\-H/) {
        $opt_dom_ref->{'-H'} = '';
    }
    if ($ref->{'cmd'} =~ /\-Ooutgroup\=(\S+)/) {
        $opt_dom_ref->{'-Ooutgroup'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-Ohorizweight\=(\S+)/) {
        $opt_dom_ref->{'-Ohorizweight'} = $1;
    }
    if ($ref->{'cmd'} =~ /\-OtaxMapSpec\=(\S+)/) {
        $opt_dom_ref->{'-OtaxMapSpec'} = $1;
    }

    return @spec_list;
}

###############################################################################
#
sub rebuild_domclust_options_tabid {
    my($db) = shift;
    my($tabid) = shift;
    my($opt_sel_ref) = shift;
    my($opt_dom_ref) = shift;

    my($sql) = "select * from $main::TBL_DOMINDEX where clusterID=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid);
    if ($sth->rows() == 0) {
        print STDERR "LOG :: Not found cluster for clusterID=$tabid\n";
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    my(@spec_list) = rebuild_domclust_options_cmd($ref->{'cmd'},
                                                  $opt_sel_ref,
                                                  $opt_dom_ref);

    return @spec_list;
}

###############################################################################
#
sub rebuild_domclust_options {
    my($db) = shift;
    my($tabid) = shift;
    my($opt_sel_ref) = shift;
    my($opt_dom_ref) = shift;

    my(@spec_list) = rebuild_domclust_options_tabid($db, $tabid,
                                                         $opt_sel_ref,
                                                         $opt_dom_ref);

    return @spec_list;
}

###############################################################################
#
sub clear_recog_tables {
    my($db) = shift;

    #
    my($sql);
    my(@tab_list) = $db->list_tables();
    foreach my$tab (@tab_list) {
        if (($tab =~ /^cluster_tables_idx$/) ||
            ($tab =~ /^cluster_tables_idx_phylopat$/) ||
            ($tab =~ /^core_tables_idx$/) ||
            ($tab =~ /^project_core$/) ||
            ($tab =~ /^project_domclust$/) ||
            ($tab =~ /^project_phylopat$/) ||
            ($tab =~ /^cluster_taxmap_index$/)) {
            $sql = "truncate table $tab";
        }
        else {
            $sql = "drop table $tab";
        }
        $db->execute($sql);
    }

    return;
}

###############################################################################
#
sub post_domclust_request {
    my($timeout) = shift;
    my($form_opt) = shift;

    #
    my($public__server) = "http://mbgd.genome.ad.jp/htbin/RECOG/domclust.cgi";
#    my($public__server) = "http://bias.nibb.ac.jp:8043/htbin/RECOG/domclust.cgi";

    #
    my($t_start) = time();

    #
    my($opts) = [ 'output_format' => 'text',
                  'output_type'   => 'complete',
                ];

    my(@key_list) = ('ingroup', 'outgroup', '-c', '-m', '-mr', '-C', '-V', '-n', '-ne', '-p', '-H', '-ai', '-ao');
    foreach my$key (@key_list) {
        if (!exists($form_opt->{"$key"})) {
            next;
        }

        push(@{$opts}, $key => $form_opt->{"$key"});
    }



    #
    for (;;) {
        my($user_agent) = LWP::UserAgent->new();
        my($res) = $user_agent->request(POST $public__server, $opts);

        print "##########\n";
        print 'RES :: ' . $res->as_string() . "\n";
        print "##########\n";
        sleep(10);
    }

    return;
}

###############################################################################
#
sub get_ug_spec_list {
    my(@ug_spec_list);

    my($dir) = "$ENV{'RECOG_HOME'}/species";
    my($dh) = IO::Dir->new("$dir");
    if (!$dh) {
        return;
    }

    while (my$file=$dh->read()) {
        next if ($file !~ /^gu\d+$/);

        my($filename) = "$dir/$file/gm/genome.txt";
        my($fh) = IO::File->new("$filename", 'r');
        if (!$fh) {
            next;
        }
        while (my$line=$fh->getline()) {
            $line =~ s#[\r\n]*$##;
            my($key, $val) = split(/\t/, $line);
            if ($key =~ /^sp$/) {
                push(@ug_spec_list, $val);
                last;
            }
        }
        $fh->close();
    }

    return @ug_spec_list;
}

###############################################################################
#
sub get_file_genome_last {
    my(@ug_spec_list);

    my($dir) = "$ENV{'RECOG_HOME'}/species";
    my($dh) = IO::Dir->new("$dir");
    if (!$dh) {
        return;
    }

    my($file_last);
    while (my$file=$dh->read()) {
        next if ($file !~ /^gu\d+$/);

        my($filename) = "$dir/$file/gm/genome.txt";
        if (!$file_last) {
            $file_last = $filename;
        }
        elsif (-M $filename < -M $file_last) {
            $file_last = $filename;
        }
    }

    return $file_last;
}

###############################################################################
#
sub update_tax_tree {
    my($file_tax) = shift;

    if (!$file_tax) {
        $file_tax = "$ENV{'MBGD_HOME'}/database.work/tax";
    }

    if (-e $file_tax) {
        my($file_genome_last) = get_file_genome_last();
        if (!$file_genome_last) {
            # no user genome
            return;
        }

        if ($file_genome_last
         && -e $file_genome_last
         && (-M $file_tax < -M $file_genome_last)) {
            return;
        }
    }

    #
    my($cmd) = "$ENV{'MBGD_HOME'}/build/m1150UpdateTaxTree.pl -USE_FILE -OUT=$file_tax";
    system("$cmd");

    #
    my($file_tax_work) = "$ENV{'RECOG_HOME'}/etc/tax.work";
    my($cmd) = "$main::CMD_cp -p $file_tax $file_tax_work";
    system("$cmd");

    return;
}

###############################################################################
1;#
###############################################################################
