#!/usr/bin/perl -s
use strict;
use IO::Dir;
use IO::File;
use File::Path qw/rmtree/;
use RecogProjectCommon;
require 'InfoSpec.pl';

package RECOG::RecogProject;

###############################################################################
# method(s)
#   new()
#   _init
#   get_dir_projects
#   validate_project_id
#   validate_project_name
#   get_filename_project
#   exists_project
#   get_project_id_list
#   exists_project_name
#   create_project_id
#   is_system_project
#   load_project
#   save_project
#   create_project
#   add_project
#   set_project_species
#   derive_project
#   add_subproject
#   del_project
#   del_subproject
#   get_dir_users
#   validate_user_name
#   exists_user
#   exists_project_user
#   get_project_user_list
#   get_user_list
#   exists_project_admin
#   permit_project_user
#   has_role_project_user
#   clear_role_project_user
#   set_derive_project_id
#   set_role_project_user
#   unset_role_project_user
#   get_role_project_user
#   add_project_user
#   del_project_user
#   print_html_sorry
###############################################################################


###############################################################################
$RECOG::RecogProject::ID_PRJ_system = 1; # $RecogProjectCommon::ID_PRJ_SYSTEM;

###############################################################################
$RECOG::RecogProject::AUTH_PRJ_view_data = 'auth_prj_view_data';
$RECOG::RecogProject::AUTH_PRJ_add_data  = 'auth_prj_add_data';
$RECOG::RecogProject::AUTH_PRJ_admin     = 'auth_prj_admin';
$RECOG::RecogProject::AUTH_SYS_add_data  = 'auth_sys_add_data';
$RECOG::RecogProject::AUTH_SYS_admin     = 'auth_sys_admin';

@RECOG::RecogProject::AUTH_PRJ = ($RECOG::RecogProject::AUTH_PRJ_view_data,
                                  $RECOG::RecogProject::AUTH_PRJ_add_data,
                                  $RECOG::RecogProject::AUTH_PRJ_admin,
                                  );
@RECOG::RecogProject::AUTH_SYS = ($RECOG::RecogProject::AUTH_SYS_add_data,
                                  $RECOG::RecogProject::AUTH_SYS_admin,
                                  );

###############################################################################
$RECOG::RecogProject::ACT_ADD_PRJ     = 'action_add_project';
$RECOG::RecogProject::ACT_DEL_PRJ     = 'action_del_project';
$RECOG::RecogProject::ACT_ADD_SUBPRJ  = 'action_add_subproject';
$RECOG::RecogProject::ACT_DEL_SUBPRJ  = 'action_del_subproject';
$RECOG::RecogProject::ACT_ADD_USR     = 'action_add_user';
$RECOG::RecogProject::ACT_DEL_USR     = 'action_del_user';
$RECOG::RecogProject::ACT_ADD_PRJ_USR = 'action_add_project_user';
$RECOG::RecogProject::ACT_DEL_PRJ_USR = 'action_del_project_user';

$RECOG::RecogProject::ACT_VIEW_PRJ_DAT = 'action_view_project_data';
$RECOG::RecogProject::ACT_ADD_PRJ_DAT  = 'action_add_project_data';
$RECOG::RecogProject::ACT_DEL_PRJ_DAT  = 'action_del_project_data';
$RECOG::RecogProject::ACT_ADD_SYS_DAT  = 'action_add_system_data';
$RECOG::RecogProject::ACT_DEL_SYS_DAT  = 'action_del_system_data';

###############################################################################
#
sub new {
    my($class) = shift;
    my(@args) = @_;
    my($self) = {};

    bless($self, $class);
    $self->_init();

    return $self;
}

###############################################################################
#
sub _init {
    my($self) = shift;

    my($role);
    $self->{'PERMIT'} = {};

    #
    $role = $RECOG::RecogProject::AUTH_PRJ_view_data;
    $self->{'PERMIT'}->{"$role"} = {};
    foreach my$p ($RECOG::RecogProject::ACT_VIEW_PRJ_DAT) {
        $self->{'PERMIT'}->{"$role"}->{"$p"} = 1;
    }

    #
    $role = $RECOG::RecogProject::AUTH_PRJ_add_data;
    $self->{'PERMIT'}->{"$role"} = {};
    foreach my$p ($RECOG::RecogProject::ACT_VIEW_PRJ_DAT,
                  $RECOG::RecogProject::ACT_ADD_PRJ_DAT,
                  $RECOG::RecogProject::ACT_DEL_PRJ_DAT) {
        $self->{'PERMIT'}->{"$role"}->{"$p"} = 1;
    }

    #
    $role = $RECOG::RecogProject::AUTH_PRJ_admin;
    $self->{'PERMIT'}->{"$role"} = {};
    foreach my$p ($RECOG::RecogProject::ACT_ADD_SUBPRJ,
                  $RECOG::RecogProject::ACT_DEL_SUBPRJ,
                  $RECOG::RecogProject::ACT_ADD_USR,
                  $RECOG::RecogProject::ACT_DEL_USR,
                  $RECOG::RecogProject::ACT_ADD_PRJ_USR,
                  $RECOG::RecogProject::ACT_DEL_PRJ_USR) {
        $self->{'PERMIT'}->{"$role"}->{"$p"} = 1;
    }

    #
    $role = $RECOG::RecogProject::AUTH_SYS_add_data;
    $self->{'PERMIT'}->{"$role"} = {};
    foreach my$p ($RECOG::RecogProject::ACT_ADD_SYS_DAT,
                  $RECOG::RecogProject::ACT_DEL_SYS_DAT) {
        $self->{'PERMIT'}->{"$role"}->{"$p"} = 1;
    }

    #
    $role = $RECOG::RecogProject::AUTH_SYS_admin;
    $self->{'PERMIT'}->{"$role"} = {};
    foreach my$p ($RECOG::RecogProject::ACT_ADD_PRJ,
                  $RECOG::RecogProject::ACT_DEL_PRJ,
                  $RECOG::RecogProject::ACT_ADD_SUBPRJ,
                  $RECOG::RecogProject::ACT_DEL_SUBPRJ,
                  $RECOG::RecogProject::ACT_ADD_USR,
                  $RECOG::RecogProject::ACT_DEL_USR,
                  $RECOG::RecogProject::ACT_ADD_PRJ_USR,
                  $RECOG::RecogProject::ACT_DEL_PRJ_USR) {
        $self->{'PERMIT'}->{"$role"}->{"$p"} = 1;
    }

    return;
}

###############################################################################
#
sub get_dir_projects {
    my($self) = shift;

    my($dir) = "$ENV{'MBGD_HOME'}/etc/projects";

    return $dir;
}

###############################################################################
#
sub validate_project_id {
    my($self) = shift;
    my($id_proj) = shift;

    if ($id_proj !~ /^\d+$/) {
        print STDERR "ERROR :: Invalid project-id [$id_proj]\n";
        foreach my$i (0 .. 5) {
            print STDERR "DBG[$i] :: " . join('#', caller($i)) . "\n";
        }
        return;
    }

    return 1;
}

###############################################################################
#
sub validate_project_name {
    my($self) = shift;
    my($name_proj) = shift;

    if ($name_proj =~ /^\s*$/) {
        print STDERR "ERROR :: No project-name.\n";
        return;
    }

    if ($name_proj =~ /[^a-z0-9\_\.\!\#\%\=\-\+\[\]\{\}]/i) {
        print STDERR "ERROR :: Invalid project-name [$name_proj]\n";
        return;
    }

    return 1;
}

###############################################################################
#
sub get_filename_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    my($dir) = $self->get_dir_projects();
    my($file_project) = "$dir/$id_proj";

    return $file_project;
}

###############################################################################
#
sub exists_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($file_project) = $self->get_filename_project($user, $id_proj);
    if (-e "$file_project") {
        return $file_project;
    }

    return;
}

###############################################################################
#
sub get_project_id_list {
    my($self) = shift;

    my(@id_list);

    my($dir) = $self->get_dir_projects();
    my($dh) = IO::Dir->new("$dir") || return;
    while (my$file=$dh->read()) {
        next if ($file !~ /^\d+$/);

        push(@id_list, $file);
    }

    return @id_list;
}

###############################################################################
#
sub count_project {
    my($self) = shift;

    my(@id_list) = $self->get_project_id_list();
    my($n) = scalar(@id_list);

    return $n;
}

###############################################################################
#
sub exists_project_name {
    my($self) = shift;
    my($user) = shift;
    my($name_proj) = shift;

    my($sta) = $self->validate_project_name($name_proj);
    if (!$sta) {
        return 2;
    }

    #
    my(@id_list) = $self->get_project_id_list();
    foreach my$id_proj (@id_list) {
        my($file_project) = $self->get_filename_project($user, $id_proj);
        my($ref) = $self->load_project($user, $id_proj);
        if ($ref->{'PROP'}->{'name'} eq $name_proj) {
            # Found same project name.
            return 1;
        }
    }

    return;
}

###############################################################################
#
sub create_project_id {
    my($self) = shift;
    my($user) = shift;

    for (my$id_proj = 1; ; $id_proj++) {
        my($file_project) = $self->get_filename_project($user, $id_proj);
        if (!-e "$file_project") {
            IO::File->new(">$file_project");
            return $id_proj
        }
    }

    return;
}

###############################################################################
#
sub is_system_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    if (!defined($id_proj)) {
        print STDERR "WARNIGN!! No project ID :: is_system_project($user, ?) :: " . caller() . "\n";
    }

    #
    if ($id_proj == $RECOG::RecogProject::ID_PRJ_system) {
        return 1;
    }

    return;
}

###############################################################################
#
sub load_project_base_cluster {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($dir_proj) = $self->get_dir_projects();
    my($dir_bc) = "$dir_proj/$id_proj.d/base_cluster";
    my($dh) = IO::Dir->new("$dir_bc");
    if (!$dh) {
        return;
    }
    while (my$file=$dh->read()) {
        next if (($file !~ /^default$/) && (($file !~ /^\d+$/) && ($file !~ /^\d+\_\d+$/)));

        my($fh) = IO::File->new("$dir_bc/$file", 'r');
        if ($fh) {
            my($ref) = {};
            $ref->{'ID'} = $file;
            while (my$line=$fh->getline()) {
                $line =~ s#[\r\n]*$##;
                my($k, $v) = split(/\t/, $line);
                $ref->{"$k"} = $v;
            }
            $fh->close();


            $ref->{'spid_list'} = [];
            if (exists($ref->{'spid'})) {
                push(@{$ref->{'spid_list'}}, split(/\s/, $ref->{'spid'}));
            }
            return $ref;
        }
    }

    return;
}

###############################################################################
#
sub load_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($file_project) = $self->exists_project($user, $id_proj);
    if (!$file_project) {
        print STDERR "ERROR :: Not found project file '$file_project'\n";
        return;
    }
    my($fh) = IO::File->new("$file_project");
    if (!$fh) {
        print STDERR "ERROR :: Can not open project file '$file_project'\n";
        return;
    }

    #
    my($ent) = {};
    $ent->{'ID'} = $id_proj;
    $ent->{'PROP'} = {};
    $ent->{'USER'} = {};
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        $line =~ s#[\r\n]*$##;
        if ($line =~ /^#(.+)/) {
            my($prop) = $1;
            my($k, $v) = split(/\t/, $prop);
            next if ($k =~ /^spid_list$/i);
            $v =~ s#\%([0-9a-z][0-9a-z])#chr(hex($1))#gei;
            $ent->{'PROP'}->{"$k"} = $v;
            next;
        }

        my($u, $r) = split(/\t/, $line);
        my($sta) = $self->exists_user($user, $u);
        if (!$sta) {
            print STDERR "DBG :: NO USER :: $user\n";
            next;
        }

        #
        $ent->{'USER'}->{"$u"} = {};
        my(@role_list) = split(/\,/, lc($r));
        foreach my$role (@role_list) {
            $ent->{'USER'}->{"$u"}->{"$role"} = 1;
        }
    }
    $fh->close();

    $ent->{'PROP'}->{'spid_list'} = [];
    if (exists($ent->{'PROP'}->{'spid'})) {
        push(@{$ent->{'PROP'}->{'spid_list'}}, split(/\s/, $ent->{'PROP'}->{'spid'}));
    }

    #
    $self->{'PROJECTS'}->{'ID'}->{"$id_proj"} = $ent;

    #
    my($name_proj) = $ent->{'PROP'}->{'name'};
    $self->{'PROJECTS'}->{'NAME'}->{"$name_proj"} = $ent;

    my($bc_ref) = $self->load_project_base_cluster($user, $id_proj);
    if ($bc_ref) {
        $ent->{'BASE_CLUSTER'} = $bc_ref;
    }

    return $ent;
}

###############################################################################
#
sub get_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($ref) = $self->load_project($user, $id_proj);

    return $ref;
}

###############################################################################
#
sub get_spid_list {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($ref) = $self->load_project($user, $id_proj);

    my(@spid_list) = @{$ref->{'PROP'}->{'spid_list'}};

    return @spid_list;
}

###############################################################################
#
sub get_spec_list {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($file_spidtab)      = "$ENV{'MBGD_HOME'}/etc/spid.tab";
    my($file_spidtab_dist) = "$file_spidtab.dist";

    my(@spec_list);
    my(@spid_list) = $self->get_spid_list($user, $id_proj);
    foreach my$spid (@spid_list) {
        my($spec) = main::spid2sp($spid, $file_spidtab_dist);
        if (!$spec) {
            $spec = main::spid2sp($spid, $file_spidtab);
        }
        if ($spec) {
            push(@spec_list, $spec);
        }
    }

    return @spec_list;
}

###############################################################################
#
sub get_spid_list_base_cluster {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($ref) = $self->load_project($user, $id_proj);

    my(@spid_list);
    if (exists($ref->{'BASE_CLUSTER'}->{'spid_list'})) {
        @spid_list = @{$ref->{'BASE_CLUSTER'}->{'spid_list'}};
    }

    return @spid_list;
}

###############################################################################
#
sub get_spec_list_base_cluster {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($file_spidtab)      = "$ENV{'MBGD_HOME'}/etc/spid.tab";
    my($file_spidtab_dist) = "$file_spidtab.dist";

    my(@spec_list);
    my(@spid_list) = $self->get_spid_list_base_cluster($user, $id_proj);
    foreach my$spid (@spid_list) {
        my($spec) = main::spid2sp($spid, $file_spidtab_dist);
        if (!$spec) {
            $spec = main::spid2sp($spid, $file_spidtab);
        }
        if ($spec) {
            push(@spec_list, $spec);
        }
    }

    return @spec_list;
}

###############################################################################
#
sub get_spid_list_all {
    my($self) = shift;
    my($user) = shift;

    #
    my(%spid_hash);
    my(@proj_id_list) = $self->get_project_id_list();
    foreach my$proj_id (@proj_id_list) {
        my(@spid_list) = $self->get_spid_list($user, $proj_id);
        foreach my$spid (@spid_list) {
            $spid_hash{"$spid"} = 1;
        }
    }

    my(@spid_list) = sort(keys(%spid_hash));

    return @spid_list;
}

###############################################################################
#
sub get_spid_list_all_base_cluster {
    my($self) = shift;
    my($user) = shift;

    #
    my(%spid_hash);
    my(@proj_id_list) = $self->get_project_id_list();
    foreach my$proj_id (@proj_id_list) {
        my(@spid_list) = $self->get_spid_list($user, $proj_id);
        foreach my$spid (@spid_list) {
            $spid_hash{"$spid"} = 1;
        }

        my(@spid_list) = $self->get_spid_list_base_cluster($user, $proj_id);
        foreach my$spid (@spid_list) {
            $spid_hash{"$spid"} = 1;
        }
    }

    my(@spid_list) = sort(keys(%spid_hash));

    return @spid_list;
}

###############################################################################
#
sub save_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($file_project) = $self->get_filename_project($user, $id_proj);
    my($fh) = IO::File->new(">$file_project.$$");
    if (!$fh) {
        return;
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};

    #
    my(@key_list) = keys(%{$ent->{'PROP'}});
    foreach my$k (sort(@key_list)) {
        next if ($k =~ /^spid_list$/i);

        my($v) = $ent->{'PROP'}->{"$k"};
        $v =~ s#(\W)#sprintf("%%%02X", ord($1))#ge;
        $fh->print('#', $k, "\t", $v, "\n");
    }

    #
    my(@user_list) = $self->get_project_user_list($user, $id_proj);
    foreach my$u (sort(@user_list)) {
        my(@role_list) = $self->get_role_project_user($user, $id_proj, $u);
        my($role) = join(',', @role_list);
        $fh->print($u, "\t", $role, "\n");
    }

    $fh->close();

    #
    unlink("$file_project.bak") if (-e "$file_project.bak");
    rename("$file_project", "$file_project.bak") if (-e "$file_project");
    rename("$file_project.$$", "$file_project");

    return;
}

###############################################################################
#
sub create_project {
    my($self) = shift;
    my($user) = shift;
    my($name_proj) = shift;
    my($opt) = shift;
    my(@species) = @_;

    #
    my($id_proj) = $self->create_project_id($user);

    #
    my($ent) = {};
    $ent->{'PROP'} = {};
    $ent->{'PROP'}->{'name'} = $name_proj;
    $ent->{'PROP'}->{'descr'} = $opt->{'descr'};
    $ent->{'PROP'}->{'derive'} = $opt->{'derive'};
    $ent->{'USER'} = {};
    $self->{'PROJECTS'}->{'ID'}->{"$id_proj"} = $ent;
    $self->{'PROJECTS'}->{'NAME'}->{"$name_proj"} = $ent;

    #
    $self->clear_role_project_user($user, $id_proj, $user);
    $self->set_role_project_user($user, $id_proj, $user, @RECOG::RecogProject::AUTH_PRJ);
    $self->save_project($user, $id_proj);

    return $id_proj;
}

###############################################################################
#
sub add_project {
    my($self) = shift;
    my($user) = shift;
    my($name_proj) = shift;
    my($opt) = shift;
    my(@species) = @_;

    my($sta) = $self->validate_project_name($name_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->permit_project_user($user, $RECOG::RecogProject::ID_PRJ_system,
                                                 $RECOG::RecogProject::ACT_ADD_PRJ);
    if (!$sta) {
        return;
    }

    my($id_proj) = $self->create_project($user, $name_proj, $opt, @species);

    return $id_proj;
}

###############################################################################
#
sub set_project_species {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my(@splist) = @_;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my(@spid_list);
    my($spid);
    my($file_spidtab)      = "$ENV{'MBGD_HOME'}/etc/spid.tab";
    my($file_spidtab_dist) = "$file_spidtab.dist";
    foreach my$sp (@splist) {
        if ($sp =~ /^g[mu]\d+$/) {
            $spid = $sp;
        }
        else {
            $spid = main::sp2spid($sp, $file_spidtab_dist);
            if (!$spid) {
                $spid = main::sp2spid($sp, $file_spidtab);
            }
        }
        if ($spid) {
            push(@spid_list, $spid);
        }
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    $ent->{'PROP'}->{'spid'} = join(' ', @spid_list);

    #
    $self->save_project($user, $id_proj);

    return;
}

###############################################################################
#
sub set_project_descr {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($descr) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    $ent->{'PROP'}->{'descr'} = $descr;

    #
    $self->save_project($user, $id_proj);

    return;
}

################################################################################
#
sub get_project_descr {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    my($descr) = $ent->{'PROP'}->{'descr'};

    return $descr;
}

###############################################################################
#
sub set_project_create_default_cluster {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($sta_create_default_cluster) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    $ent->{'PROP'}->{'create_default_cluster'} = $sta_create_default_cluster;

    #
    $self->save_project($user, $id_proj);

    return;
}

################################################################################
#
sub get_project_create_default_cluster {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    my($sta) = $ent->{'PROP'}->{'create_default_cluster'};

    return $sta;
}

###############################################################################
#
sub derive_project {
    my($self) = shift;
    my($user) = shift;
    my($name_proj) = shift;
    my($opt) = shift;
    my(@species) = @_;

    my($derive_project_id) = $opt->{'derive'};
    my($sta) = $self->validate_project_id($derive_project_id);
    if (!$sta) {
        return;
    }

    my($sta) = $self->validate_project_name($name_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->permit_project_user($user, $derive_project_id,
                                                 $RECOG::RecogProject::ACT_ADD_SUBPRJ);
    if (!$sta) {
        return;
    }

    my($id_proj) = $self->create_project($user, $name_proj, $opt, @species);

    return $id_proj;
}

###############################################################################
#
sub add_subproject {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($id_proj_new) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->validate_project_id($id_proj_new);
    if (!$sta) {
        return;
    }

    my($sta) = $self->permit_project_user($user, $id_proj,
                                                 $RECOG::RecogProject::ACT_ADD_SUBPRJ);
    if (!$sta) {
        return;
    }

    my($sta) = $self->exists_project($user, $id_proj);
    if (!$sta) {
        # NOT exists
        return;
    }

    my($sta) = $self->exists_project($user, $id_proj_new);
    if ($sta) {
        # already exists
        return;
    }

    # create NEW project
    my($file_project) = $self->get_filename_project($user, $id_proj_new);
    my($fh) = IO::File->new(">$file_project");
    $fh->print('#', 'parent_project', "\t", $id_proj, "\n");
    my($role) = join(',', $RECOG::RecogProject::AUTH_PRJ_admin,
                          $RECOG::RecogProject::AUTH_PRJ_add_data,
                          $RECOG::RecogProject::AUTH_PRJ_view_data);
    $fh->print($user, "\t", $role, "\n");
    $fh->close();

    return;
}

###############################################################################
#
sub del_project {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->permit_project_user($user, $RECOG::RecogProject::ID_PRJ_system,
                                                 $RECOG::RecogProject::ACT_DEL_PRJ);
    if (!$sta) {
        return;
    }

    my($sta) = $self->exists_project($user, $id_proj);
    if (!$sta) {
        # NOT exists
        return;
    }

    my($sta) = $self->is_system_project($user, $id_proj);
    if ($sta) {
        # can not delete'system' project.
        return;
    }

    #
    my($file_project) = $self->get_filename_project($user, $id_proj);
    my($file_project_d) = $file_project . '.d';
    print STDERR "pro = $file_project $file_project_d \n";
    unlink($file_project);
    File::Path::rmtree($file_project_d) if (-d $file_project_d);

    return;
}

###############################################################################
#
sub del_subproject {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_name($id_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->permit_project_user($user, $id_proj,
                                                 $RECOG::RecogProject::ACT_DEL_SUBPRJ);
    if (!$sta) {
        return;
    }

    my($sta) = $self->exists_project($user, $id_proj);
    if (!$sta) {
        # NOT exists
        return;
    }

    my($sta) = $self->is_system_project($user, $id_proj);
    if ($sta) {
        # can not delete 'system' project.
        return;
    }

    #
    my($file_project) = $self->get_filename_project($user, $id_proj);
    unlink($file_project);

    return;
}

###############################################################################
#
sub get_dir_users {
    my($self) = shift;

    my($dir) = "$ENV{'MBGD_HOME'}/etc/users";

    return $dir;
}

###############################################################################
#
sub validate_user_name {
    my($self) = shift;
    my($name_user) = shift;

    if ($name_user =~ /^\s*$/) {
        return;
    }

    if ($name_user =~ /[^a-z0-9\_\.\-\@]/i) {
        return;
    }

    return 1;
}

###############################################################################
#
sub exists_user {
    my($self) = shift;
    my($user) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_user_name($name_user);
    if (!$sta) {
        return;
    }

    my($dir) = $self->get_dir_users();
    my($file_user) = "$dir/$name_user";
    if (-e "$file_user") {
        return 1;
    }

    return;
}

###############################################################################
#
sub exists_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        print STDERR " LOG :: Invalid project ID. ($id_proj)\n";
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    if (exists($ent->{'USER'}->{"$name_user"})) {
        print STDERR " LOG :: Found user. ($id_proj :: $name_user)\n";
        return 1;
    }

    return;
}

###############################################################################
#
sub get_project_user_list {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    my(@user_list) = keys(%{$ent->{'USER'}});

    return @user_list;
}

###############################################################################
#
sub get_user_list {
    my($self) = shift;
    my($user) = shift;

    my(@user_list);
    my($dir) = $self->get_dir_users();
    my($dh) = IO::Dir->new("$dir") || return;
    while (my$file=$dh->read()) {
        next if ($file =~ /^\./);

        push(@user_list, $file);
    }

    return @user_list;
}

###############################################################################
#
sub exists_project_admin {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    my($r) = $RECOG::RecogProject::AUTH_PRJ_admin;
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }
    my(@user_list) = $self->get_project_user_list($user, $id_proj);
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    foreach my$u (@user_list) {
        if ($u eq $name_user) {
            next;
        }

        if ($ent->{'USER'}->{"$u"}->{"$r"} != 0) {
            return 1;
        }
    }

    return;
}

###############################################################################
#
sub permit_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($action) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }
    my(@role_list) = $self->get_role_project_user($user, $id_proj, $user);
    foreach my$role (@role_list) {
        if ($self->{'PERMIT'}->{"$role"}->{"$action"}) {
            return 1;
        }
    }

    return;
}

###############################################################################
#
sub has_role_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;
    my($role) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    my($sta) = exists($ent->{'USER'}->{"$name_user"}->{"$role"});
    if ($sta) {
        return 1;
    }

    return;
}

###############################################################################
#
sub clear_role_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    $ent->{'USER'}->{"$name_user"} = {};

    return;
}

###############################################################################
#
sub set_derive_project_id {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($id_proj_derive) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    my($sta) = $self->validate_project_id($id_proj_derive);
    if (!$sta) {
        return;
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    $ent->{'PROP'}->{'derive'} = $id_proj_derive;

    return;
}

###############################################################################
#
sub set_role_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;
    my(@role_list) = @_;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    foreach my$role (@role_list) {
        $ent->{'USER'}->{"$name_user"}->{"$role"} = 1;
    }

    return;
}

###############################################################################
#
sub unset_role_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;
    my(@role_list) = @_;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    foreach my$role (@role_list) {
        delete($ent->{'USER'}->{"$name_user"}->{"$role"});
    }

    return;
}

###############################################################################
#
sub get_role_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    #
    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};

    if (!exists($ent->{'USER'}->{"$name_user"})) {
        return;
    }
    my(@role_list) = keys(%{$ent->{'USER'}->{"$name_user"}});

    return @role_list;
}

###############################################################################
#
sub add_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    if (!exists($ent->{'USER'}->{"$name_user"})) {
        $ent->{'USER'}->{"$name_user"} = {};
    }
    my($p) = $RECOG::RecogProject::AUTH_PRJ_view_data;
    $ent->{'USER'}->{"$name_user"}->{"$p"} = 1;
    $self->save_project($user, $id_proj);

    return;
}

###############################################################################
#
sub del_project_user {
    my($self) = shift;
    my($user) = shift;
    my($id_proj) = shift;
    my($name_user) = shift;

    my($sta) = $self->validate_project_id($id_proj);
    if (!$sta) {
        return;
    }

    if (!exists($self->{'PROJECTS'}->{'ID'}->{"$id_proj"})) {
        # Not loaded
        $self->load_project($user, $id_proj);
    }

    #
    my($ent) = $self->{'PROJECTS'}->{'ID'}->{"$id_proj"};
    delete($ent->{'USER'}->{"$name_user"});
    $self->save_project($user, $id_proj);

    return;
}

###############################################################################
#
sub print_html_sorry {
    my($self) = shift;
    my($user) = shift;
    my($msg) = shift;

    if (!$msg) {
        $msg = "You don't have permission. [$user]";
    }

    print "Content-type: text/html\n";
    print "\n";

    print <<EOB;
<html>
<head>
<link rel="stylesheet" type="text/css" href="/css/mbgd.css" />
</head>
<body>
<h1>Sorry</h1>
$msg<br>
<input type="button" name="btn_close" value="Close" onclick="self.close();">
</body>
</html>
EOB

    return;
}

###############################################################################
if ($0 eq __FILE__) {
}

###############################################################################
1;#
###############################################################################
