#!/usr/bin/perl -s

###############################################################################
use strict;
use warnings;
use File::Basename;
use File::Path;
use FileHandle;
require "InfoSpec.pl";
package SpidSet;

$SpidSet::MODE_blastdp = 'blastdp';


###############################################################################
#   new()
#   _init()
#   clear_spid_set()
#   get_dirname()
#   set_dirname()
#   get_filename()
#   read_spid_set()
#   write_spid_set()
#   read_spid_status()
#   write_spid_status()
#   get_setname()
#   count_spidset()
#   delete_spid_set()
#   clear_spid()
#   add_spid()
#   get_spid_list()
#   get_spec_list()
#   is_member_spid_set()
#   rename_spid_set()
#   clear_spid_status()
#   set_spid_status()
#   get_spid_status()
#   get_all_spid_list()

###############################################################################
#
sub new {
    my($class) = shift;
    my(@args) = @_;
    my($self) = {};

    bless($self, $class);
    $self->_init(@args);

    return $self;
}

###############################################################################
#
sub _init {
    my($self) = shift;
    my($dirname) = shift;

    # ʲ˽
    $self->clear_spid_set();

    #
    if (!$dirname || $dirname =~ /^\s*$/) {
        $dirname = "$ENV{'MBGD_HOME'}/etc/spid_set";
    }

    $self->set_dirname($dirname);

    #
    $self->{'NAME2SPID'} = {};
    $self->{'SPID2NAME'} = {};
    foreach my$file ("$main::FILE_spidtab.dist",
                     "$main::FILE_spidtab") {
        my($spectab_ref) = main::getInfoSpecTab("$file");
        my(@name_list) = keys(%{$spectab_ref->{'NAME2SPID'}});
        foreach my$name (@name_list) {
            my($spid) = $spectab_ref->{'NAME2SPID'}->{"$name"};
            $self->{'NAME2SPID'}->{"$name"} = $spid;
        }
        my(@spid_list) = keys(%{$spectab_ref->{'SPID2NAME'}});
        foreach my$spid (@spid_list) {
            my($name) = $spectab_ref->{'SPID2NAME'}->{"$spid"};
            $self->{'SPID2NAME'}->{"$spid"} = $name;
        }
    }

    #
    $self->read_spid_set();

    return;
}

###############################################################################
#
sub clear_spid_set {
    my($self) = shift;

    $self->{'SPID_SET_ID'}     = {};
    $self->{'SPID_SET_NAME'}   = {};
    $self->{'SPID_SET_STATUS'} = {};

    $self->{'SPEC_SET'}        = {};

    return;
}

###############################################################################
#
sub get_dirname {
    my($self) = shift;

    return $self->{'DIRNAME'};
}

###############################################################################
#
sub set_dirname {
    my($self) = shift;
    my($dir) = shift;

    $self->{'DIRNAME'} = $dir;
    if (! -e $dir) {
        my($sta) = File::Path::mkpath($dir, 0, 0750);
        if (! $sta) {
            print STDERR "ERROR :: Can not mkpath '$dir'. ($!)";
        }
    }

    return;
}

###############################################################################
#
sub get_filename {
    my($self) = shift;

    my($dir) = $self->get_dirname();
    my($filename) = sprintf("%s/spid_set.tab", $dir);

    return $filename;
}

###############################################################################
#
sub read_spid_set {
    my($self) = shift;
    my($filename) = $self->get_filename();

    $self->clear_spid_set();
    my($fh) = FileHandle->new("$filename");
    if (!$fh) {
        print STDERR "WARNING :: Can not open $filename($!)\n";
        return;
    }

    #
    my($n_spidset) = 0;
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        $line =~ s#[\r\n]*$##;
        my($set_id, $set_name, $spid) = split(/\t/, $line);
        next if ($set_name =~ /^\s*$/);
#        next if ($spid  =~ /^\s*$/);

        $self->add_set_name($set_id, $set_name);
        my(@spid_list) = split(/[,\s]+/, $spid);
        $self->clear_spid($set_id);
        foreach my$wk_spid (@spid_list) {
            if ($wk_spid !~ /^g[mu]\d+$/) {
                $wk_spid = $self->{'NAME2SPID'}->{"$wk_spid"};
            }
            next if (!$wk_spid);

            $self->add_spid($set_id, $wk_spid);
        }
        $n_spidset++;
    }
    $fh->close();
    print STDERR "DBG :: read_spid_set($filename)\n" if ($main::DEBUG);

    #
    $self->read_spid_status();

    return;
}

###############################################################################
#
sub write_spid_set {
    my($self) = shift;
    my($filename) = $self->get_filename();
    my($fh) = FileHandle->new(">$filename.$$");
    if (!$fh) {
        print STDERR "WARNING :: Can not open $filename($!)\n";
        return;
    }

    #
    $fh->print("#", join("\t", 'id', 'name', 'spid-list'), "\n");

    my(@set_id_list) = $self->get_setid_list();
    foreach my$set_id (@set_id_list) {
        my($set_name) = $self->get_setname($set_id);
        my(@spid_list) = $self->get_spid_list($set_id);

        $fh->print($set_id, "\t", $set_name, "\t", join(" ", @spid_list), "\n");
    }
    $fh->close();
    unlink("$filename.bak") if (-e "$filename.bak");
    rename("$filename",    "$filename.bak");
    rename("$filename.$$", "$filename");
    print STDERR "DBG :: write_spid_set($filename)\n" if ($main::DEBUG);

    #
    $self->write_spid_status();

    return;
}

###############################################################################
#
sub read_spid_status {
    my($self) = shift;

    my($dir) = $self->get_dirname();
    my($filename) = "$dir/spid.status";
    my($fh) = FileHandle->new("$filename");
    if (!$fh) {
        print STDERR "WARNING :: Can not open $filename($!)\n";
        return;
    }

    #
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*#/);
        next if ($line =~ /^\s*$/);

        $line =~ s#[\r\n]*$##;

        my($mode, $spid1, $spid_list) = split(/\t/, $line);
        my(@spid_list) = split(/\s+/, $spid_list);
        foreach my$spid2 (@spid_list) {
            $self->set_spid_status($mode, $spid1, $spid2);
        }
    }
    $fh->close();
    print STDERR "DBG :: read_spid_status($filename)\n" if ($main::DEBUG);

    return;
}

###############################################################################
#
sub write_spid_status {
    my($self) = shift;

    my($dir) = $self->get_dirname();
    my($filename) = "$dir/spid.status";
    my($fh) = FileHandle->new(">$filename");
    if (!$fh) {
        print STDERR "WARNING :: Can not open $filename($!)\n";
        return;
    }

    #
    $fh->print("#", join("\t", 'mode', 'spid', 'spid-list'), "\n");

    my(@mode_list) = sort(keys(%{$self->{'SPID_SET_STATUS'}}));
    foreach my$mode (@mode_list) {
        my($spid_sta__ref) = $self->{'SPID_SET_STATUS'}->{"$mode"};
        my(@spid1_list) = sort(keys(%{$spid_sta__ref}));
        foreach my$spid1 (@spid1_list) {
            my($spid_sta_spid1_ref) = $spid_sta__ref->{"$spid1"};
            my(@spid_list) = sort(keys(%{$spid_sta_spid1_ref}));
            my($spid_list) = join(" ", @spid_list);
            $fh->print(join("\t", $mode, $spid1, $spid_list), "\n");
        }
    }
    $fh->close();

    return;
}

###############################################################################
#
sub exists_set_id {
    my($self) = shift;
    my($set_id) = shift;

    if (exists($self->{'SPID_SET_ID'}->{"$set_id"})) {
        return $set_id;
    }

    return;
}

###############################################################################
#
sub exists_set_name {
    my($self) = shift;
    my($name) = shift;

    my(@setid_list) = $self->get_setid_list();
    foreach my$set_id (@setid_list) {
        my($set_name) = $self->get_setname($set_id);
        if ($name eq $set_name) {
            return $name;
        }
    }

    return;
}

###############################################################################
#
sub get_setid_list {
    my($self) = shift;

    my(@setid_list) = sort(keys(%{$self->{'SPID_SET_ID'}}));

    return @setid_list;
}

###############################################################################
#
sub add_set_id {
    my($self) = shift;
    my($set_id) = shift;

    if (!$set_id) {
        for (my$i = 1; $i < 999999; $i++) {
            if (!exists($self->{'SPID_SET_ID'}->{"$i"})) {
                $set_id = $i;
                last;
            }
        }
    }

    #
    $self->{'SPID_SET_ID'}->{"$set_id"} = $set_id;
    $self->{'SPID_SET_NAME'}->{"$set_id"} = '';

    return $set_id;
}

###############################################################################
#
sub add_set_name {
    my($self) = shift;
    my($set_id) = shift;
    my($set_name) = shift;

    if (!$set_id) {
        $set_id = $self->add_set_id();
    }
    $self->{'SPID_SET_NAME'}->{"$set_id"} = $set_name;

    return $set_id;
}

###############################################################################
#
sub get_setname {
    my($self) = shift;
    my($set_id) = shift;

    if (!$set_id) {
       print STDERR "WARNING :: No set-id.\n";
    }
    my($set_name) = $self->{'SPID_SET_NAME'}->{"$set_id"};

    return $set_name;
}

###############################################################################
#
sub count_spidset {
    my($self) = shift;

    my(@setid_list) = $self->get_setid_list();
    my($n) = scalar(@setid_list);

    return $n;
}

###############################################################################
#
sub delete_spid_set {
    my($self) = shift;
    my($set_id) = shift;

    delete($self->{'SPID_SET_ID'}->{"$set_id"});
    delete($self->{'SPID_SET_STATUS'}->{"$set_id"});

    return;
}

###############################################################################
#
sub clear_spid {
    my($self) = shift;
    my($set_id) = shift;

    #
    $self->{'SPID_SET_ID'}->{"$set_id"} = {};

    return;
}

###############################################################################
#
sub add_spid {
    my($self) = shift;
    my($set_id) = shift;
    my(@spid_list) = @_;

    if (!exists($self->{'SPID_SET_ID'}->{"$set_id"})) {
        return;
    }

    #
    foreach my$spid (@spid_list) {
        $self->{'SPID_SET_ID'}->{"$set_id"}->{"$spid"} = 1;
    }

    return;
}

###############################################################################
#
sub get_spid_list {
    my($self) = shift;
    my($set_id) = shift;

    if (!exists($self->{'SPID_SET_ID'}->{"$set_id"})) {
        return;
    }

    my(@spid_list) = sort(keys(%{$self->{'SPID_SET_ID'}->{"$set_id"}}));

    return @spid_list;
}

###############################################################################
#
sub get_spec_list {
    my($self) = shift;
    my($set_id) = shift;

    my(@spid_list) = $self->get_spid_list($set_id);
    my(@spec_list);
    foreach my$spid (@spid_list) {
        my($spec) = $self->{'SPID2NAME'}->{"$spid"};
        next if (!$spec);

        push(@spec_list, $spec);
    }

    return @spec_list;
}

###############################################################################
#
sub is_member_spid_set {
    my($self) = shift;
    my($set_id) = shift;
    my($spid) = shift;

    my(@spid_list) = $self->get_spid_list($set_id);
    foreach my$spid2 (@spid_list) {
        if ($spid eq $spid2) {
            return $spid;
        }
    }

    return;
}

###############################################################################
#
sub clear_spid_status {
    my($self) = shift;
    my($mode) = shift;
    my($spid1) = shift;
    my($spid2) = shift;

    delete($self->{'SPID_SET_STATUS'}->{"$mode"}->{"$spid1"}->{"$spid2"});

    return;
}

###############################################################################
#
sub set_spid_status {
    my($self) = shift;
    my($mode) = shift;
    my($spid1) = shift;
    my($spid2) = shift;
    my($sta) = shift;

    $sta = 1 if (!defined($sta));
    $self->{'SPID_SET_STATUS'}->{"$mode"}->{"$spid1"}->{"$spid2"} = $sta;
#    print STDERR "DBG :: set_spid_status('$mode', '$spid1', '$spid2', $sta)\n";

    return;
}

###############################################################################
#
sub get_spid_status {
    my($self) = shift;
    my($mode) = shift;
    my($spid1) = shift;
    my($spid2) = shift;

    my($val) = $self->{'SPID_SET_STATUS'}->{"$mode"}->{"$spid1"}->{"$spid2"};

    return $val;
}

###############################################################################
#
sub get_all_spid_list {
    my($self) = shift;

    my(%spid_hash);
    my(@setid_list) = $self->get_setid_list();
    foreach my$set_id (@setid_list) {
        my(@spid_list) = $self->get_spid_list($set_id);
        foreach my$spid (@spid_list) {
            $spid_hash{"$spid"} = 1;
        }
    }
    my(@all_spid_list) = sort(keys(%spid_hash));

    return @all_spid_list;
}

###############################################################################
1;#
###############################################################################
