#!/usr/bin/perl


package UserInfo;

use MBGD;
use MBGD::Taxonomy;
use Property::HomolParam;
use Property::Preference;
use Property::MalignParam;
use FileHandle;
use File::Path;


sub new {
	my($class, $uid) = @_;
	my($this) = {};

	bless $this, $class;
	$this->setUID($uid);
	return $this;
}

sub setUID {
	my($this, $uid) = @_;
	my($datadir);

	if ($uid =~ /\W/){
		warn "Invalid uid: $uid\n";
		$uid = '';
	}
	if (! $uid) {
		my $time = time();
		$uid = $time . '_' .  $$;
		$datadir = $main::DIR_MbgdUser . "/$uid";

		while (-d $datadir) {
			$uid = ++$time . '_' . $$;
			$datadir = $main::DIR_MbgdUser . "/$uid";
		}
		$this->{newuid} = 1;
	} else {
		$datadir = $main::DIR_MbgdUser . "/$uid";
	}

	$this->{uid} = $uid;
	$this->{datadir} = $datadir;
	$this->{paramfile} = "$datadir/${uid}.param";
	$this->{preffile} = "$datadir/${uid}.pref";
	$this->{malifile} = "$datadir/${uid}.mali";
	$this->{tabidfile} = "$datadir/${uid}.tabid";
	$this->{tabidlistfile} = "$datadir/common.tabid";
	$this->{qcheckfile} = "$datadir/${uid}.clstinf";
	$this->{seqfile} = "$datadir/${uid}.seq";
	$this->{seqsimfile} = "$datadir/${uid}.sim";

	$this->{propfile}->{'Property::Preference'} = $this->{preffile};
	$this->{propfile}->{'Property::MalignParam'} = $this->{malifile};
	$this->{propfile}->{'Property::HomolParam'} = $this->{paramfile};
#	if ($this->{newuid}) {
#		$this->saveHomolParams();
#	}
}
sub is_newuid {
	$_[0]->{newuid};
}
sub uid {
	$_[0]->{uid};
}
sub paramfile {
	$_[0]->{paramfile};
}

sub createUserDir {
	my($this) = @_;
	if (! -e $this->{datadir}) {
		mkpath($this->{datadir});
	}
}
sub getpath {
	my($this, $filename) = @_;
	return "$this->{datadir}/$filename";
}
sub saveParams {
	my($this, $data, $suff) = @_;
	my($paramfile) = $this->{paramfile};
	if (UNIVERSAL::isa($data,"Property::Base")) {
		my $class = ref $data;
		$paramfile = $this->{propfile}->{$class};
		$data = $data->getParams;
	}
	$paramfile .= ".$suff" if ($suff);

	$this->createUserDir();

	my $fh = FileHandle->new(">$paramfile");
	die "Can't open $this->{paramfile} for write\n" if (! $fh);

	foreach $key (keys %{$data}) {
		if ($key =~ /^species$/i) {
			my$dir = "$ENV{'MBGD_HOME'}/database/";
			my$tax = MBGD::Taxonomy->new($taxonfile);
			my$defaultSpec = join('|', sort($tax->get_default_spec()));
			my$userSpec = join('|', sort(split(/\|/, $data->{$key})));
			if ($userSpec eq $defaultSpec) {
				# don't save default species set
				next;
			}
		}
		print $fh "$key\t$data->{$key}\n";
	}
	$fh->close;
}
sub resetHomolParams {
	my($this) = @_;
	$hom = Property::HomolParam->new(reset=>1);
	$this->saveParams($hom->getParams);
}
sub saveHomolParams {
	my($this, $hom, $suff) = @_;
	if (! $hom) {
		# $hom might be an initopt such as 'reset'
		$hom = Property::HomolParam->new();
	}
	my $curr_params = $this->loadParams;	## read current UserParams
	my $hom_params = $hom->getParams;	## get current HomolParams
	foreach $k (keys %{$hom_params}) {	## overwrite UserParams
		$curr_params->{$k} = $hom_params->{$k};
	}
	$this->saveParams($curr_params, $suff);
}

#sub initHomolParams {
#	my $hom = $this->saveHomolParams('reset');
#}

sub loadParams {
	my($this, $suff) = @_;
	my $data = {};
	my($paramfile) = $this->{paramfile};
	$paramfile .= ".$suff" if ($suff);

	if (my $fh = FileHandle->new($paramfile)) {
		while(<$fh>){
			chomp;
			($key, $value) = split(/\t/);
			$data->{$key} = $value;
		}
		$fh->close;
	}
	return $data;
}
sub loadHomolParams {
	my($this, $suff) = @_;
	my $data = $this->loadParams($suff);
	my $hom = Property::HomolParam->new;
	if ($data) {
		## replace the default values with the saved ones
		$hom->setParams($data);
	} else {
		## no parameter file: create a new file
		##  to save the default homology parameters
		$this->saveParams($hom->getParams);
	}
	return $hom;
}
sub getHomolParamHashRef {
	my($this) = @_;
	my $hom = $this->loadHomolParams;
	return {} if (! $hom);
	return $hom->asHashRef();
}
sub getHomolParamHash {
	my($this) = @_;
	my $hom = $this->loadHomolParams;
	return {} if (! $hom);
	return $hom->asHash($mode);
}

sub setHomolParamFromHash {
	my($this, $hash) = @_;
	my $hom = Property::HomolParam->new();
	$hom->setParams($hash);
	$this->saveHomolParams($hom);
}

#sub savePreference {
#	my($this, $pref) = @_;
#	$pref->save($this->{preffile});
#}
#sub loadPreference {
#	my($this) = @_;
#	my $pref = Property::Preference->new;
#	$pref->load($this->{preffile}) if (-f $this->{preffile});
#	$pref;
#}

sub saveProperty {
	my($this, $prop) = @_;
	my($class) = ref($prop);
	$prop->save($this->{propfile}->{$class});
}
sub loadProperty {
	my($this, $class) = @_;
	my $prop = $class->new;
	my $propfile = $this->{propfile}->{$class};
	$prop->load($propfile) if (-f $propfile);
	$prop;
}
sub resetProperty {
	my($this, $class) = @_;
	my $prop = $class->new(reset=>1);
	my $a = $prop->getParams;
	$this->saveParams($prop);
}

sub readQueueCheck {
	my($this) = @_;
	my $fh = FileHandle->new($this->{qcheckfile});
	my $rawinput = <$fh>;
	chomp $rawinput;
	$fh->close;
	$rawinput;
}
sub writeQueueCheck {
	my($this, $out) = @_;
	my $fh = FileHandle->new(">" . $this->{qcheckfile});
	print $fh $out;
	$fh->close;
}

sub saveTableIDs {
	my($this, $tabid) = @_;
	my $tabID = $this->loadTableIDs;
	my %tmpHash;
	foreach $t (@{$tabID->{list}}, $tabid) {
		$tmpHash{$t} = 1;
	}
	$list = join(',', keys %tmpHash);
	$this->createUserDir();

	my $fh = FileHandle->new(">".$this->{tabidfile});
	print $fh "current	$tabid\n";
	$fh->close;

	my $fh = FileHandle->new(">".$this->{tabidlistfile});
	print $fh "list	$list\n";
	$fh->close;
}
sub loadTableIDs {
	my($this, $ret) = @_;
	my($tabID) = {};

	foreach my $filename ($this->{tabidfile}, $this->{tabidlistfile}) {
		next if (! -e $filename);
		my $fh = FileHandle->new($filename);
		return 0 if (! $fh);
		while(<$fh>) {
			chomp;
			my($name, $value) = split(/\t/);
			if ($name eq 'current') {
				$tabID->{current} = $value;
			} elsif ($name eq 'list') {
				@{$tabID->{list}} = split(/,/, $value);
			}
		}
		$fh->close;
	}
	if ($ret eq 'current') {
		return $tabID->{current};
	} elsif ($ret eq 'list') {
		return $tabID->{list};
	} else {
		return $tabID;
	}
}
sub currentTableID {
	my($this) = @_;
	my $tabIDs = $this->loadTableIDs;
	if ($tabIDs) {
		return $tabIDs->{current};
	} else {
		return 'default';
	}
}
sub saveQuerySequence {
	my($this, $seq) = @_;
	$this->createUserDir();
	my $fh = FileHandle->new(">".$this->{seqfile});
	print $fh $seq;
	$fh->close;
}
sub loadQuerySequence {
	my($this) = @_;
	my($seq);
	my $fh = FileHandle->new($this->{seqfile});
	while(<$fh>) {
		$seq .= $_;
	}
	$fh->close;
	$seq;
}
sub getQuerySequenceAsHash {
	my($this) = @_;
	my(%Seq);
	my $seq = $this->loadQuerySequence;
	foreach $ln (split(/\n/, $seq)) {
		if ($ln =~ /^>\s*(\S+)/) {
			$name = $1;
			my($sp,$orf) = split(/:/,$name);
			$orf = uc($orf);
			$name = "$sp:$orf";
		}
		$Seq{$name} .= "$ln\n";
	}
	return %Seq;
}
sub getQuerySequence {
	my($this, $hash, $name) = @_;
	my($sp,$orf) = split(/:/,$name);
	$orf = uc($orf);
	$name = "$sp:$orf";
	$hash->{ $name };
}
sub saveData {
	my($this, $filename, $content) = @_;
	my $fullname = $this->{datadir} . '/' . $filename;
	$this->createUserDir();
	my $fh = FileHandle->new(">$fullname");
	print $fh $content;
	$fh->close;
}
sub loadData {
	my($this, $filename) = @_;
	my($data);
	my $fh = FileHandle->new($filename);
	while(<$fh>) {
		$data .= $_;
	}
	$fh->close;
	$data;
}
sub getDataFile {
	my($this, $name) = @_;
	return $this->{datadir} if (! $name);
	return $this->{$name} if ($this->{$name});
	return $this->{datadir} . "/$name";
}

sub setSpecies {
    my($this) = shift;
    my($spec) = shift;

    #
    my($hom) = $this->loadHomolParams();
    my(%param) = $this->getHomolParamHash;

    my(@species);
    if (@{$spec}) {
        my(%species);
        foreach my$sps (@{$spec}) {
            foreach my$sp (split(/,/, $sps)) {
                $species{"$sp"} = 1;
            }
        }

        # sort by taxonomy
        my($tax) = MBGD::Taxonomy->new($main::ENV{'MYSQL_USER_DATABASE'});
        foreach my$sps ($tax->get_all_spec) {
            push(@species, $sps) if ($species{$sps} == 1);
        }
        $param{'species'} = join('|',@species);
    } elsif ($spec) {
        @species = split(/,/, $spec);
        $param{'species'} = join('|',@species);
    }

    $hom->setParams( \%param );
    $this->saveHomolParams($hom);

    return @species;
}

1;
