#include "ClusterTree.h"
#include "SpecSet.h"
#include "PairDist.h"

class ExecPhyloCut {
	ClusterTree *tree;
	double phylo_cutoff;
	SpecSet *chkSpSet;
public:
	ExecPhyloCut(ClusterTree *_tree, double cutoff, SpecSet *_chkSpSet)
		: tree(_tree), phylo_cutoff(cutoff), chkSpSet(_chkSpSet) {
	}
	void phyloCut() {
		phyloCut(tree->getRoot());
	}
	void phyloCut(ClusterNode *node) {
		ClusterNode *child1 = node->getChild(0);
		ClusterNode *child2 = node->getChild(1);
		if (child1 == NULL || child2 == NULL) {
		} else if (duplicationCheck(child1, child2)) {
/*
cout << "phylopat:" << child1->spSet.toString()<< " " << chkSpSet->toString() << endl;
cout << "phylocut:" << node->getID() << ">" << child1->getID() << " " << child2->getID();
cout << " flags: " <<  specFlagInclude(*chkSpSet, child1->spSet) << " " << specFlagInclude(*chkSpSet, child2->spSet) << endl;
*/

			if (chkSpSet &&
				! specFlagInclude(*chkSpSet, child1->spSet) &&
				! specFlagInclude(*chkSpSet, child2->spSet)) {
				return;
			}

			node->unsetRootFlag();
			child1->setRootFlag();
			child2->setRootFlag();
/*
			phyloCut(child1);
			phyloCut(child2);
*/
		}
	}
	bool duplicationCheck(ClusterNode *node1, ClusterNode *node2) {
/*
		SpecSet spSet1 = node1->spSet;
		SpecSet spSet2 = node2->spSet;
*/
		SpecSetInstances *specSetInst = SpecSetInstances::getInstance();

		/* ignoring taxon-unknown species in metagenome data */
		SpecSet spSet1, spSet2;
		specFlagANDNOT(node1->spSet,specSetInst->unknown_SpecSet, spSet1) ;
		specFlagANDNOT(node2->spSet,specSetInst->unknown_SpecSet, spSet2) ;

		int mchcnt = specFlagANDcnt(spSet1, spSet2);
		int spcnt1 = spSet1.count();
		int spcnt2 = spSet2.count();
		if (spcnt1 == 1 && spcnt2 == 1 && mchcnt == 1) {
			/* single organism cluster */
			return 0;
		}
		return (mchcnt > spcnt1 * phylo_cutoff
				|| mchcnt > spcnt2 * phylo_cutoff);
	}
};

class ExecScoreCut {
	ClusterTree *tree;
	double cutoff;
	SpecSet *newSpSet;
public:
	ExecScoreCut(ClusterTree *_tree, double cutoff, SpecSet *_newSpSet)
		: tree(_tree), cutoff(cutoff), newSpSet(_newSpSet) {
	}
	int scoreCut() {
		return scoreCut(tree->getRoot());
	}
	int scoreCut(ClusterNode *node) {
		ClusterNode *child1 = node->getChild(0);
		ClusterNode *child2 = node->getChild(1);
		if (child1 == NULL || child2 == NULL) {
		} else {
			SimValue simV = node->getSimValData();
			if (simV.worseSimValue(cutoff)) {
				bool newspec_only_1 = specFlagInclude(*newSpSet, child1->getSpSet());
				bool newspec_only_2 = specFlagInclude(*newSpSet, child2->getSpSet());
				if (newspec_only_1 && ! newspec_only_2) {
					node->unsetRootFlag();
					child2->setRootFlag();
					return 1;
				} else if (! newspec_only_1 && newspec_only_2) {
					node->unsetRootFlag();
					child1->setRootFlag();
					return 1;
				}
			}
		}
		return 0;
	}
};

ClusterTree::ClusterTree(ClusterNode *_root, string _name) :
	BaseTree::BaseTree(_root, _name) {
}
void ClusterTree::phyloCut(double cutoff, SpecSet *chkSpSet) {
	ExecPhyloCut exec_phycut(this, cutoff, chkSpSet);
	exec_phycut.phyloCut();
}
int ClusterTree::scoreCut(double cutoff, SpecSet *newSpSet) {
	ExecScoreCut scoreCut(this, cutoff, newSpSet);
	return scoreCut.scoreCut();
}
void ClusterTree::outputScore(FILE *fp, const char *str) {
	getRoot()->outputNodeScore(fp, str);
}

void ClusterTree::output(int outStyle) {
	if ( (OutStyle) outStyle == TREEOUT ) {
		printTree();
	} else if ( (OutStyle) outStyle == TREEDETAIL ) {
		printTreeDetail();
	} else if ( (OutStyle) outStyle == NEWICK ) {
		printNewick( false );
	} else if ( (OutStyle) outStyle == NEWICK_DIST) {
		printNewick( true );
	} else {
		printTree();
	}
}
void ClusterTree::printTree() {
	ClusterNode *root = getRoot();
	if (root->isRoot()) {
		BaseTree::printTree();
	} else {
		printTree(root);
	}
}
void ClusterTree::printTree(ClusterNode *node) {
	if (node->isRoot()) {
		BaseTree bt(node);
		bt.printTree();
	} else {
		printTree(node->getChild(0));
		printTree(node->getChild(1));
	}
}


void ClusterTree::printTreeDetail() {
        printTreeDetail_sub(getRoot(), NULL);
}
void ClusterTree::printTreeDetail_sub(ClusterNode* node, ClusterNode *parent) {
        Edge *child;
        char markchar;
        int pid = -1;

        if (parent != NULL) {
                pid = parent->getID();
        }

        if (node->isLeaf()) {
                /** leaf **/
        	Domain *dom = node->getDomain()->getOrigDomain();
                printf("L %d %d %s ", node->getID(), pid, dom->getName().c_str());
                printf("%d %d %d", dom->getFrom(), dom->getTo(), dom->getDomNum());
                putchar('\n');
        } else {
                printf("I %d %d %s ", node->getID(), pid, node->getName().c_str());

                printf("%.1f %.1f\n", node->getScore(), node->getDist());
        
                printTreeDetail_sub(node->getChild(0), node);
                printTreeDetail_sub(node->getChild(1), node);
        }
}

ClusterTree *ClusterTree::convClusterTree(DistTree *tree) {
	
	ClusterNode *root = convClusterTree_sub(tree->getRoot());
	return (new ClusterTree(root));
}
ClusterNode *ClusterTree::convClusterTree_sub(TreeNode *node) {
	ClusterNode *newn;
	if (node->isLeaf()) {
		Domain *dom = node->getDomain();
		newn = ClusterNode::createLeafNode(dom);
	} else {
		TreeNode *child1 = node->getChild(0);
		TreeNode *child2 = node->getChild(1);
		ClusterNode *n1 = convClusterTree_sub(child1);
		ClusterNode *n2 = convClusterTree_sub(child2);
		PairDist *pdist = PairDist::create(n1, n2, node->getDist(), node->getScore());
		newn = ClusterNode::createIntNode(pdist);
	}
	return newn;
}
