#!/usr/local/bin/perl

package MBGD::ClustTab::Writer;
use MBGD::ClustTab;
use MBGD::ClustTab::PosCluster;
use FileHandle;
use ArgCheck;
use MBGD::FuncCat;
use MBGD::FunctionCategory;
use MBGD::Gene;

require 'libMBGDDBM.pl';
#require 'libMBGDFuncCat.pl';

$DEFAULT_FIELD_SEP = "\t";
$DEFAULT_RECORD_SEP = "\n";
$DEFAULT_GENE_SEP = " ";

#########################################################################
# for saving the text format
#########################################################################
sub new {
	my($class, $clusttab, %origOpt) = @_;
	my($this) = {};
	bless $this, $class;

	$this->{clusttab} = $clusttab;

	if ($origOpt{out} =~ /^\w/) {
		$this->{out} = FileHandle->new(">$origOpt{out}")
			|| die "Can't open $origOpt{out}\n";
	}

	if ($this->can(ValidArgs)) {
		$argCheck = ArgCheck->new( $this->ValidArgs , {DEBUG=>0});
		$opt = $argCheck->check(\%origOpt);
	} else {
		$opt = \%origOpt;
	}

	$this->init();

	while (($k,$v) = each %{$opt}) {
		$this->{$k} = $v;
	}

	$this->init2();

    if ($origOpt{'cluster_color'}) {
        my($dirFuncCat) = "$ENV{'MBGD_HOME'}/database/function";
        $this->{'func_cat'} = new MBGD::FunctionCategory($origOpt{'cluster_color'},
                                                         $dirFuncCat, 1);
    }

	return $this;
}

sub table_begin { $_[0]->{table_begin}; } sub table_end { $_[0]->{table_end}; }
sub record_begin { $_[0]->{record_begin}; } sub record_end { $_[0]->{record_end}; }
sub field_begin { $_[0]->{field_begin}; } sub field_end { $_[0]->{field_end}; }
sub field_h_begin { $_[0]->{field_h_begin}; } sub field_h_end { $_[0]->{field_h_end}; }
sub field_rh_begin { $_[0]->{field_rh_begin}; } sub field_rh_end { $_[0]->{field_rh_end}; }
sub gene_begin { $_[0]->{gene_begin}; } sub gene_end { $_[0]->{gene_end}; }

sub init {
	my($this) = @_;

	$this->{gene_end} = $DEFAULT_GENE_SEP;
	$this->{field_end} = $DEFAULT_FIELD_SEP;
	$this->{field_h_end} = $DEFAULT_FIELD_SEP;
	$this->{field_rh_end} = $DEFAULT_FIELD_SEP;
	$this->{record_end} = $DEFAULT_RECORD_SEP;
}
sub init2 {
	my($this) = @_;
	if ($this->{format}) {
		if (ref $this->{format} ne 'ARRAY') {
			my @array = split(/,/, $this->{format});
			$this->{format} = \@array;
		}
	} else {
		$this->{format} = ['DATA'];
		foreach $f ($this->{clusttab}->list_field) {
			push(@{$this->{format}}, $f);
		}
	}
}

#
sub get_splist {
    my($this) = shift;
	my($splist) = $this->{clusttab}->splist;

    return $splist;
}

sub printStrings {
	my($this, @strings) = @_;
	if ($this->{out}) {
		$this->{out}->print(@strings);
	} else {
		print(@strings);
	}
}

sub print_table {
	my($this) = @_;
	my $clustTab = $this->{clusttab};
	my $clnum = $clustTab->clustnum;
	$this->printStrings($this->table_begin);
	$this->print_table_header;
	for (my $i = 0; $i < $clnum; $i++) {
		my $cluster = $clustTab->get_cluster($i);
		$this->print_row($cluster);
	}
	if ($clnum >= 6) {
		$this->print_table_header;
	}
	$this->printStrings($this->table_end);
}

sub print_table_header {
	my($this) = @_;
	$this->printStrings($this->record_begin,
			$this->field_h_begin,
			"ClusterID",
			$this->field_h_end);

	foreach $item (@{$this->{format}}) {
		if ($item eq 'DATA') {
			$this->print_table_header_for_species;
		} elsif ($this->{clusttab}->has_field($item)) {
			my($name);
			if ($field = MBGD::ClustTab::Fields->getInstance($item)) {
				$name = $field->display_name;
			} else {
				$name = $item;
			}
			$this->printStrings(
				$this->field_h_begin,
				$name,
				$this->field_h_end);
		}
	}

	$this->printStrings( $this->record_end );
}
sub print_table_header_for_species {
	my($this) = @_;
#	my($splist) = $this->{clusttab}->splist;
	my($splist) = $this->get_splist();

#	for (my $i = 0; $i < @{$splist}; $i++) {
#		$this->printStrings(
#			$this->field_h_begin .
#			$splist->[$i] .
#			$this->field_h_end);
#	}
	foreach my$sp (@{$splist}) {
		$this->printStrings(
			$this->field_h_begin .
			$sp .
			$this->field_h_end);
	}
}
sub print_row {
	my($this, $clustdata) = @_;
	$this->printStrings($this->{record_begin});
	$this->print_row_header($clustdata);
	$this->print_all_data($clustdata);
	$this->printStrings($this->record_end);
}
sub print_row_header {
	my($this, $clustdata) = @_;
	$this->printStrings($this->field_h_begin,
		$clustdata->clustid,
		$this->field_h_end);
}
sub print_all_data {
	my($this, $clustdata) = @_;
	my(@format) = @{$this->{format}};

	foreach $item (@{$this->{format}}) {
		if ($item eq 'DATA') {
			$this->print_data($clustdata);
		} elsif ($this->{clusttab}->has_field($item)) {
			$this->print_field($clustdata, $item);
		} else {
#print STDERR "field item not found: $item\n";
		}
	}

    if ($this->{'annotsp'}) {
        if (!$this->{'annotsp_descr'}) {
            my($opt) = {};
            $opt->{'colmns'} = "sp,name,descr";
            my(@gene_list) = MBGD::Gene->get($sp, $opt);
            $this->{'annotsp_descr'} = {};
            foreach my$ref (@gene_list) {
                my($sp)    = $ref->{'sp'};
                my($name)  = $ref->{'name'};
                my($descr) = $ref->{'descr'};
                $this->{'annotsp_descr'}->{"$sp:$name"} = $descr;
            }
        }
        my($data) = $this->{clusttab}->get_spdata_by_sp($clustdata, $sp);
        my($descr) = @{$data};
        $this->printStrings($this->field_begin, $descr, $this->field_end);
    }

    return;
}
sub print_data {
	my($this, $clustdata) = @_;

	my($splist) = $this->get_splist();
	foreach my$sp (@{$splist}) {
		my $data = $this->{clusttab}->get_spdata_by_sp($clustdata, $sp);
		$this->printStrings($this->field_begin);
		$this->print_species_gene_data($sp, $data);
		$this->printStrings($this->field_end);
	}
}
sub print_field {
	my($this, $clustdata, $item) = @_;
	my($field_data) = $clustdata->field($item);

	if ($field = MBGD::ClustTab::Fields->getInstance($item)) {
		$field_data = $field->print($field_data);
	}
	$this->printStrings(
		$this->field_begin,
		$field_data,
		$this->field_end);
}
sub print_species_gene_data {
	my($this, $sp, $data) = @_;

	foreach $gdata (@{$data}) {
		$this->print_gene_data( $sp, $gdata );
	}
}
sub print_gene_data {
	my($this, $sp, $gdata) = @_;
	if (ref($gdata) eq 'ARRAY') {
		$gdata = &MBGD::ClustTab::print_genename($gdata);
	}
	$this->printStrings( $this->gene_begin,
			"${sp}:${gdata}",
			$this->gene_end);
}

#########################################################################
# for test
package MBGD::ClustTab::Writer_HTMLSimple;
@ISA = qw( MBGD::ClustTab::Writer );
sub init {
	my($this) = @_;
	$this->SUPER::init;
	$this->{gene_begin} = '';
	$this->{gene_end} = ' ';
	$this->{field_begin} = "<TD>";
	$this->{field_end} = "</TD>";
	$this->{field_h_begin} = "<TH>";
	$this->{field_h_end} = "</TH>";
	$this->{field_rh_begin} = "<TH>";
	$this->{field_rh_end} = "</TH>\n";
	$this->{record_begin} = "<TR>";
	$this->{record_end} = "</TR>\n";
	$this->{table_begin} = "<TABLE BORDER=1>";
	$this->{table_end} = "</TABLE>\n";
	return $this;
}
#########################################################################
# experimental
package MBGD::ClustTab::Writer_XML;
@ISA = qw( MBGD::ClustTab::Writer );
sub init {
	my($this) = @_;
	$this->SUPER::init;
	$this->{gene_begin} = '<gene>';
	$this->{gene_end} = '</gene>';
	$this->{field_begin} = "<species_data>";
	$this->{field_end} = "</species_data>";
	$this->{field_h_begin} = "<species>";
	$this->{field_h_end} = "</species>";
	$this->{field_rh_begin} = "<id>";
	$this->{field_rh_end} = "</id>";
	$this->{record_begin} = "<cluster>";
	$this->{record_end} = "</cluster>\n";
	$this->{table_begin} = "<cluster_table>";
	$this->{table_end} = "</cluster_table>\n";
	return $this;
}
#########################################################################
## for MBGD WWW server
#########################################################################

package MBGD::ClustTab::Writer_HTML;
@ISA = qw( MBGD::ClustTab::Writer_HTMLSimple );

sub ValidArgs {
	[genes, qgenes, highlight_genes, showname, table_color, postscript,
		posclust, 'format', 'ref_spec', 'annotsp', 'addspec']
}

sub init {
	my($this, %opt) = @_;
	$this->SUPER::init;
}

sub field_rh_begin {
	my($this,$clustdata) = @_;
    my($funcCat) = 'FuncCat_' . $this->{'func_cat'}->getDbType();
	my $func_id = $clustdata->field($funcCat);
#	my $func_color = &MBGD::FuncCat::get_func_color_from_id($func_id);
    my($ftype, $fid) = split(':', $func_id);
    my($func_color) = $this->{'func_cat'}->getFunctionColorByLevel($fid);
	"<TH bgcolor=#$func_color>";
}
sub print_table_header_for_species {
	my($this) = @_;
    my($splist) = $this->get_splist();
    my(%specDefault);
    if ($this->{clusttab}->tabid() =~ /^all$/i) {
        %specDefault = map { $_ => '*' } @{$splist};
        my(@specDefault) = main::MBGD_SpecTableGetDefaultSpecies();
        foreach my$sp (@specDefault) {
            $specDefault{"$sp"} = '';
        }
    }

    # foreach Υ롼Ƥѹʤ褦
    # 򥳥ԡƤѤ롣
    my(@splist) = @{$splist};
	foreach my$spname (@splist) {
		my $bgcolor = '';
		if ($this->{ref_spec} && $this->{ref_spec} eq $spname) {
			$bgcolor = ' bgcolor=#ff8888';
		}

		if ($this->{showname} eq 'count') {
			$spname = join("<br>", split(//, $spname));
		}
		$this->printStrings("<TH${bgcolor}>$spname$specDefault{$spname}</TH>");
	}
}
sub print_row_header {
	my($this, $clustdata) = @_;
	print $this->field_rh_begin($clustdata),
		$this->print_header_HTML($clustdata),
		$this->field_rh_end();
}

sub print_species_gene_data {
	my($this, $sp,$data) = @_;
	my($disp, $ClustScore, $TotCount) = $this->print_data_HTML($sp,$data);
	print $disp;
}

sub print_field {
	my($this, $clustdata, $item) = @_;
	my($field_data) = $clustdata->field($item);

	if ($field = MBGD::ClustTab::Fields->getInstance($item)) {
		$field_data = $field->print_HTML($field_data);
	}
	$this->printStrings(
		$this->field_begin,
		$field_data,
		$this->field_end);
}

sub table_begin {
	qq{<TABLE BORDER="2">};
}

###############################################################################
sub print_data_HTML {
	my($this, $spec, $data) = @_;
	my($cnt) = (0);
	my($disp);
	my($d);
	my(%data);
	my($ClustScore, $TotCount);
	my($tabcolor,$repr_tabcolor);
	my($found_hit);
	my($HIT_GENE_COLOR) = "#ff0000";

	my($outMode) = 'html';

##	return if (! $mismatch && $exist{$spec} == -1);

	foreach $d (@{$data}) {
		$data{$d} = 1;
	}
	@data = sort keys %data;

    #
	my($objFuncCat) = $this->{'func_cat'};
	foreach $d (@data) {
		my($orf, $dom) = &MBGD::ClustTab::parse_genename($d);
		my $ent = "$spec:$orf";
		my $domStr = $dom ? "($dom)" : "";
		my($hit);
		$cnt++ if ($ent !~ /^\s*$/);

		if ($this->{table_color} eq 'category') {
			my($refList) = $objFuncCat->getFunctionListBySporf($ent);
			my($ref) = $refList->[0];
			my($lev) = $ref->{'LEVEL'};
			$tabcolor = $objFuncCat->getFunctionColorByLevel($lev);
		} elsif ($this->{table_color} eq 'posclust') {
			$tabcolor = $this->{posclust}->get_color($ent);
		}

		if ($this->{genes}->{$ent} || $this->{highlight_genes}->{$ent}) {
			$hit=1;
			$found_hit++;
		}

		if ($this->{showname} eq 'count') {
			if ($tabcolor && $tabcolor ne '#ffffff') {
				$repr_tabcolor = $tabcolor;
			}
		} elsif ($spec eq 'query') {
			$disp .= " $orf$domStr";
		} else {
			if ($this->{showname} eq 'genename') {
				my $GeneInfo = $this->{clusttab}->geneinfo;
				if ($GeneInfo->{$ent}->{gene}) {
					$outname = $GeneInfo->{$ent}->{gene};
				} else {
					$outname = $orf;
				}
			} else {
				$outname = $orf;
			}
			if ($this->{genes}->{$ent} || $this->{highlight_genes}->{$ent}) {
				$outname = "<FONT color=\"$HIT_GENE_COLOR\"><b>$outname</b></FONT></A>";
			}

			#
			$maxDom = $maxDomClust->{"$spec:$orf"};
#			$ORFname = qq{<A HREF="/htbin/MBGD_gene_info_frame.pl?spec=$spec&name=$orf">$outname</A>$domStr };
			$ORFname =
  qq{<A HREF="/htbin/MBGD_gene_info_frame.pl?name=$ent">$outname</A>$domStr };

			if ($this->{table_color} && $this->{table_color} ne 'off') {
###				if ($this->{table_color} eq 'category') {
###					$tabcolor = &MBGD::FuncCat::get_func_color(
###					   $spec, &MBGD::FuncCat::get_func_category($ent));
###				} elsif ($this->{table_color} eq 'posclust') {
###					$tabcolor = $this->{posclust}->get_color($ent);
###				}
				$disp .= "<TABLE BORDER=\"1\" CELLPADDING=\"1\" CELLSPACING=\"0\"><TR><TD bgcolor=\"$tabcolor\">$ORFname</TD></TR></TABLE> ";
			} else {
				$disp .= $ORFname;
			}
		}
	}
	if ($this->{showname} eq 'count') {
		if ($cnt) {
			if (! $this->{table_color} || $this->{table_color} eq 'off') {
				$repr_tabcolor = '#ffff00';	# yellow
				if ($cnt > 1) {
					$repr_tabcolor = '#00ffff'; # cyan
				}
			}
			$repr_tabcolor = '#ffffff' if (! $repr_tabcolor);
			my $output_cnt = $cnt;
			if ($found_hit) {
				$output_cnt =
			  "<FONT color=\"$HIT_GENE_COLOR\"><b>$cnt</b></FONT>";
			}
			$disp .= "<TABLE width=\"100%\"><TD bgcolor=$repr_tabcolor align=\"right\">$output_cnt</TD></TABLE>";
		}
	} else {
	}

##	$disp .= "</TD>\n";
#??	$disp =~ s#<TD></TD>#<TD><BR></TD>#;

	return($disp, $ClustScore, $TotCount);
}

###############################################################################
$AliButtonColor = '#a0ffff';
$MapButtonColor = '#ffffa0';
$HomButtonColor = '#ffa0ff';

sub print_header_HTML {
	my($this, $clustdata) = @_;
#	my($outMode, $name, $entnames, $button_off, $check_off, $qgene) = @_;
	my(@entnames, $entnames1, $sp0, $ent0, $ent);
	my(@qgenes) = keys %{$qgene};
	my($qent);
	my($disp);
	my($tabid);

	$clustname = "O" . $clustdata->clustid;
	$tabid = $this->{clusttab}->tabid;
	if (! $tabid) {
		warn "tabid is not defined\n";
	}

    #
    my($splist) = $this->get_splist();
    my(%splist_hash);
    foreach my$sp (@{$splist}) {
        $splist_hash{"$sp"} = 1;
    }

    #
    my(%default_spec_hash);
    if ($tabid =~ /^addspec$/i) {
        my(@default_spec_list) = main::MBGD_SpecTableGetDefaultSpecies();
        foreach my$sp (@default_spec_list) {
            $default_spec_hash{"$sp"} = 1;
        }
    }

    my(@all_genes) = $this->{clusttab}->allgenes($clustdata);
    my($addspec_spname) = '';
    foreach my$g (@all_genes) {
        my($sp, $name) = split(/\:/, $g);
        if (!exists($splist_hash{"$sp"})) {
            next;
        }
        if (($tabid =~ /^addspec$/i) && (!exists($default_spec_hash{"$sp"}))) {
            if ($addspec_spname ne '') {
                $addspec_spname .= ',';
            }
            $addspec_spname .= $g;
        }
	    push(@entnames, $g);
    }

	if (@qgenes) {
		$qent = '+' . join("+", @qgenes);
	} else {
		$qent = '';
	}
#	@entnames = split(/\+/, $entnames);
	($sp0, $entry0) = split(/:/, $entnames[0]);
	($ent0, $dom0) = ($entry0 =~ /([^(]+)\(*(\d*)\)*/);
	$entnames1 = join(',', @entnames);
	$entnamesNoDom = $entnames1;
	$entnamesNoDom =~ s/\(\d+\)//g;	# ɥᥤֹκ

	# åܥå饹ֹ桢'A', 'M', 'H'
	$disp .= "<TABLE BORDER=\"0\" width=\"100%\"><TR><TH align=\"left\"nowrap>";

	if (! $check_off && ! $this->{postscript}) {
		$disp .= "<INPUT TYPE=\"checkbox\" NAME=\"cluster\" VALUE=\"$clustname,$entnames1\">";
	}

    #
    my($tabid_orig) = $tabid;
    if ($tabid =~ /^addspec$/i) {
        $tabid_orig = 'default';
    }

	$disp .= "<A HREF=/htbin/hcluster?prog=hcluster&clustid=$clustname&tabid=$tabid_orig";
    $disp .= "&annotsp=".$this->{'addspec'} if ($this->{'addspec'});
    $disp .= ">$clustname</A>";
	$disp .= "</TH>\n";
#	if (@entnames > 1 && ! $button_off && ! $this->{postscript}) {
		# 'A' : Multiple sequence alignment
		$disp .= "<TD bgcolor=\"$AliButtonColor\" width=\"1\">\n";
		$disp .= "<A HREF=\"\" onclick=\"submitMalign('${entnamesNoDom}${qent}'); return false;\" ><FONT size=\"-1\"><b>A</b></FONT></A>";
		$disp .= "</TD>\n";
#	}

	# 'M' : Multiple map comparison
	$disp .= "<TD bgcolor=\"$MapButtonColor\" width=\"1\">";
	$disp .= "<A HREF=\"/htbin/MBGD_multi_region_html.pl?tabid=$tabid&clustid=$clustname&add_spname=$addspec_spname\">";
	$disp .= "<FONT size=\"-1\"><b>M</b></FONT></A>";
	$disp .= "</TD>\n";

	# 'H' : Find homologous clusters
	$disp .= "<TD bgcolor=\"$HomButtonColor\" width=\"1\">";
#	$disp .= "<A HREF=\"/htbin/hcluster?homclust=1\&allorfs=${entnamesNoDom}&tabid=$tabid\">";
	$disp .= "<A HREF=\"/htbin/hcluster?homclust=1\&clustid=${clustname}&tabid=$tabid&add_spname=$addspec_spname";
    $disp .= "&annotsp=".$this->{'addspec'} if ($this->{'addspec'});
    $disp .= "\">";
	$disp .= "<FONT size=\"-1\"><b>H</b></FONT></A>";
	$disp .= "</TD>\n";
	$disp .= "</TR></TABLE>";

	return($disp);
}
#########################################################################
package main;
if ($0 eq __FILE__) {
}
#########################################################################
1;
