#!/usr/local/bin/perl

=head1 NAME

MBGD::Object - Base class for MBGD object

=head1 SYNOPSIS

  use MBGD;

  $g = MBGD::Genome->new;	# create a new object, with schema setting
  $g->setValues({
	sp => $sp,
	fullname => $fullname,
	strain => $strain,
	chromosomes => [
		MBGD::Chromosome->new({
			sp => $sp,
			name => $chr_name1,
			seq => MBGD::DNASeq->new({
				seq => $seq1
			})
		}),
		MBGD::Chromosome->new({
			sp => $sp,
			name => $chr_name2,
			seq => MBGD::DNASeq->new({
				seq => $seq1
			})
		})
	]
  });

  $gtab = MBGD::FlatDB->new;
  $g->store($gtab);		# output data recursively into the flat tables

  $gdb = MBGD::DB->new("dbi:mysql:mbgd", {mode=>update});
  $g->store($gdb);		# store recursively into the database

  $g->show_schema;		# show schema of the object

=cut

package MBGD::Object;

use MBGD;
use MBGD::Schema;
use MBGD::FlatTab;
use Carp;

sub new {
	my($class, @args) = @_;
	my($this) = {};
	bless $this, $class;

#if (ref($args[0]) eq 'MBGD::DB') {

	if (UNIVERSAL::isa($args[0], 'MBGD::DB')) {
		$this->{db} = shift @args;
		$this->{dbname} = $this->{db}->{'dbname'};
	}
	$this->{schema} = MBGD::Schema->new($class, $this->{dbname});

	$this->setValues($args[0]);

	$this;
}

sub setValues {
	my($this, $values) = @_;
	foreach my $attr (keys %{$values}) {
		if ($this->{schema}->exists($attr)) {
			$this->{$attr} = $values->{$attr};
#		} else {
#			warn "$attr: does not exist in schema $this->{$schema}->{classname}\n";
		}
	}
}
sub getValues {
	my($this, @attrs) = @_;
	my(@retval);
	foreach my $attr (@attrs) {
		push(@retval, $this->{$attr});
	}
	@retval;
}

sub get_identifier {
	my($this) = @_;
	my $ident = $this->{schema}->identifier;
	if ($ident) {
		return $this->{$ident};
	}
}
sub get_schema_classname {
	my($this) = @_;
	return $this->{schema}->{classname};
}
sub primary_key_value {
	my($this) = @_;
	my($keyvalue);
	foreach $attr ($this->{schema}->primary_key) {
		$keyvalue .= ":" if ($keyvalue);
		$keyvalue .= $this->{$attr};
	}
	return $keyvalue;
}
sub print {
	$_[0]->print_tab;
}
sub print_tab {
	my($this, $opt) = @_;
	my($flag);
	my($sep);
	my(@fields);
	my($out);
	if ($opt->{sep}) {
		$sep = $opt->{sep};
	} else {
		$sep = "\t";
	}
	if ($opt->{fields}) {
		if (ref $opt->{fields} eq 'ARRAY') {
			@fields = @{$opt->{fields}};
		} else {
			@fields = split(/,/, $opt->{fields});
		}
	} else {
		@fields = $this->{schema}->get_attributes;
	}
	foreach my $attr (@fields) {
		next if (! defined $this->{$attr});
		print "$sep" if ($flag++);
		if (ref($this->{$attr}) && $this->{$attr}->isa(MBGD::Object)) {
			$out =  $this->{$attr}->primary_key_value;
		} else {
			 
			$out = $this->{$attr};
		}
		if ($opt->{remove_space}) {
			$out =~ s/^\s*//;
			$out =~ s/\s*$//;
		}
		print $out;
	}
	print "\n";
}
sub print_flattab {
	my($this, $flattab) = @_;
	if (! $flattab) {
		$flattab = MBGD::FlatTab->new($this->{schema}->{classname});
	}
	$flattab->setData($this);
	$flattab->writeData;
}
sub read_flattab {
	my($this, $flattab) = @_;
	if (! $flattab) {
		$flattab = MBGD::FlatTab->new($this->{schema}->{classname});
	}
	$flattab->readData;
	$this->setValues($flattab->{tab});
}

sub __process_args {
	my($db,@opt);
	if (UNIVERSAL::isa($_[0], 'MBGD::DB')) {
		($db, @opt) = @_;
	} else {
		$db = MBGD::DB->new;
		@opt = @_;
	}
#	if ($opt && ref($opt) ne 'HASH') {
#		warn "MBGD::DBget argument error\n";
#	}
	($db, @opt);
}
sub fetch {
	my($class) = shift;
	($db, $idlist, $opt) = &__process_args(@_);
	$idlist = [$idlist] if (! ref($idlist));
	my @ret = $db->fetch_objects($class, @{$idlist}, $opt);
	return wantarray ? @ret : $ret[0];
}
# Object->get($db, ["key:value",...], {options=>value,...})
#    options: fields, keys
sub get {
	my($class) = shift;
	($db, $keyvalues, $opt) = &__process_args(@_);
	$keyvalues = [$keyvalues] if (! ref($keyvalues));
	my @ret = $db->get_objects($class, $keyvalues, $opt);
	return wantarray ? @ret : $ret[0];
}
sub find {
	my($class) = shift;
	($db, $cond, $opt) = &__process_args(@_);
	my @ret = $db->find_objects($class, $cond, $opt);
	return wantarray ? @ret : $ret[0];
}
sub find_id {
	my($class) = shift;
	($db, $cond, $opt) = &__process_args(@_);
	my $schema = MBGD::Schema->new($class);
	my @primkey = $schema->primary_key;
	$opt->{fields} = join(',', @primkey);
	my @ret = $db->find_objects($class, $cond, $opt);
	return wantarray ? @ret : $ret[0];
}
sub store {
	## OBJ->store([DB], {opt=>value, ...});
	my($this, @args) = @_;
	my($occ_check) = {};
	my($DB);
	if (UNIVERSAL::isa($args[0], 'MBGD::DB')) {
		$DB = shift @args;
	} elsif ($this->{db}) {
		$DB = $this->{db};
	}
	$opt = $args[0];
	$this->{$this->{schema}->identifier} = '';
	return $this->_store($DB, $opt, $occ_check);
}
sub _store {
	my($this, $DB, $opt, $occ_check) = @_;
	my $ident = $this->{schema}->identifier;
	my $thisid;

	return $this->{$ident} if ($this->{$ident});

	return -1 if ($occ_check->{$this} >= 1);


	$occ_check->{$this}++;
	if (! $DB && $this->{db}) {
		$DB = $this->{db};
	}
	if ($opt->{force_to_insert}) {
		$thisid = $DB->store_object($this, $opt);
		print STDERR "$thisid\n" if ($main::DEBUG && $opt->{check});
		if ($thisid < 0 && ! $opt->{ignore}) {
			warn "store aborted: $this: ",
				$this->primary_key_value,"\n";
		}
		$this->{$ident} = $thisid;
	}
	foreach my $k (keys %{$this}) {
		my $val = $this->{$k};
		if (ref($val) && UNIVERSAL::isa($val, 'MBGD::Object')) {
			my $id = $val->_store($DB, $opt, $occ_check);
			# replace objectref with objectid
			$this->{$k} = $id if ($id > 0);
		}
	}
	$thisid = $DB->store_object($this, $opt);
	print STDERR "$thisid\n" if ($main::DEBUG && $opt->{check} && $thisid == 0);
	if ($thisid < 0 && ! $opt->{ignore}) {
		warn "store aborted: $this: ",
			$this->primary_key_value,"\n";
	}
	$this->{$ident} = $thisid;

	foreach my $k (keys %{$this}) {
		$val = $this->{$k};
		if (UNIVERSAL::isa($val, 'MBGD::Object')) {
			if (my $fkey = $val->{schema}->foreign_key(ref($this))){
				if ($thisid > 0 && $val->{$fkey} eq $this) {
					$val->{$fkey} = $thisid;
				}
			}
		} elsif (ref($val) =~ /ARRAY/) {
			foreach my $e (@{$val}) {
				if (my $fkey = $e->{schema}->foreign_key(ref($this))){
#					if ($thisid>0 && $e->{$fkey} eq $this) {
					if ($thisid>0 &&
			(! defined $e->{$fkey} || $e->{$fkey} eq $this) ) {
						$e->{$fkey} = $thisid;
					}
				}
				if (UNIVERSAL::isa($e, 'MBGD::Object')) {
					$e->_store($DB,$opt,$occ_check);
				}
			}
		}
	}
	return $thisid;
}
sub equivalent {
	my($this, $obj, $opt) = @_;
	my($ref1, $ref2);
	foreach $attr ($this->{schema}->get_attributes($opt)) {
		if ( ($ref1 = ref($obj->{$attr}))
				|| ($ref2 = ref($obj->{$attr})) ) {
			next if ($opt->{ignore_ref});
		}
		if ($obj->{$attr} ne $this->{$attr}) {
			if (! $opt->{for_insert} || $this->{$attr} ) {
			    ## ignore if empty in the inserted object (this)
				return 0;
			}
		}
	}
	return 1;
}
sub show_schema {
	my($this) = @_;
	$this->{schema}->print;
}

1;
