#!/usr/bin/perl -s
use strict;
use IO::File;

package NqsAxes;

###############################################################################
#
sub new {
    my($class) = shift;
    my($dir_bin) = shift;
    my($self) = {};

    bless($self, $class);
    $self->_init($dir_bin);

    return $self;
}

###############################################################################
#
sub _init {
    my($self) = shift;
    my($dir_bin) = shift;

    #
    $self->setup_command_path($dir_bin);
    $self->set_max_job(1);

    return;
}

###############################################################################
#
sub setup_command_path {
    my($self) = shift;
    my($dir_bin) = shift;

    my($dir) = '';
    if ($dir_bin) {
        if (-d $dir_bin) {
            $dir = $dir_bin . '/';
        }
        else {
            print STDERR "WARNING :: Can not found directory '$dir_bin'\n";
        }
    }

    #
    $self->{'CMD'}->{'qsub'}  = $dir . 'qsub';
    $self->{'CMD'}->{'qstat'} = $dir . 'qstat';
    $self->{'CMD'}->{'qdel'}  = $dir . 'qdel';
    $self->{'CMD'}->{'qmod'}  = $dir . 'qmod';

    #
    my($cmd) = $self->{'CMD'}->{'qstat'};
    my($ret) = system("$cmd > /dev/null");
    if ($ret) {
        print STDERR "ERROR :: Can not execute '$cmd'\n";
    }

    return;
}

###############################################################################
#
sub get_command_path {
    my($self) = shift;
    my($name) = shift;

    my($cmd) = $self->{'CMD'}->{"$name"};
    if (!$cmd) {
        print STDERR "ERROR :: Not defined '$name'\n";
    }

    return $cmd;
}

###############################################################################
#
sub set_max_job {
    my($self) = shift;
    my($n) = shift;

    $n = int($n);
    if ($n < 1) {
        $n = 1;
    }
    $self->{'MAX_JOB'} = $n;

    return;
}

###############################################################################
#
sub get_max_job {
    my($self) = shift;

    my($n) = $self->{'MAX_JOB'};

    return $n;
}

###############################################################################
#
sub update_qstat {
    my($self) = shift;
    my($cmd) = $self->get_command_path('qstat');

    my($ref) = $self->{'JOB'} = {};
    $ref->{'JOB_STATUS'} = {};
    $ref->{'JOB_USER'}   = {};


    my(@qstat_list) = `$cmd`;
    my($idx_jobid) = 0;
    my($idx_user)  = 3;
    my($idx_sta)   = 4;
    foreach my$qstat (@qstat_list) {
#print STDERR "QSTAT :: $qstat";
        $qstat =~ s#^\s*##;

        if ($qstat =~ /^job\-id/i) {
            # for SGE
            $idx_jobid = 0;
            $idx_user  = 3;
            $idx_sta   = 4;
            next;
        }
        elsif ($qstat =~ /^job\s+id/i) {
            # for PBS
            $idx_jobid = 0;
            $idx_user  = 2;
            $idx_sta   = 4;
            next;
        }

        my(@qstat_list) = split(/\s+/, $qstat);
        my($jobid)  = $qstat_list[$idx_jobid];
        if ($jobid !~ /^\d+$/) {
            print "DBG :: SKIP :: JOBID=$jobid\n" if ($main::DEBUG);
            next;
        }

        #
        $ref->{'JOB_USER'}->{"$jobid"}   = $qstat_list[$idx_user];
        $ref->{'JOB_STATUS'}->{"$jobid"} = $qstat_list[$idx_sta];
    }

    return;
}

###############################################################################
#
sub get_job_status {
    my($self) = shift;
    my($jobid) = shift;

    my($sta) = $self->{'JOB'}->{'JOB_STATUS'}->{"$jobid"};

    return $sta;
}

###############################################################################
#
sub wait_job {
    my($self) = shift;
    my($max_job) = shift;

    if (!defined($max_job) || ($max_job < 0)) {
        $max_job = $self->get_max_job();
    }

    #
    for (;;) {
        $self->update_qstat();

        my(@jobid_list) = $self->get_jobid_list();
        foreach my$jobid (@jobid_list) {
            my($sta) = $self->get_job_status($jobid);
            if (!$sta) {
                $self->del_jobid($jobid);
            }
            elsif ($sta =~ /^Eqw$/i) {
                my($cmd_qmod) = $self->get_command_path('qmod');
                my($cmd) = "$cmd_qmod -cj $jobid";
print STDERR "WARNING :: $cmd\n";
                system("$cmd");
            }
        }

        my(@jobid_list) = $self->get_jobid_list();
        my($n) = scalar(@jobid_list);
        if ($n <= $max_job) {
            last;
        }
#print STDERR "DBG :: Found $n jobs. Waiting... (max=$max_job)\n";
        sleep(5);
    }

    return;
}

###############################################################################
#
sub wait_all_job {
    my($self) = shift;

    $self->wait_job(0);

    return;
}

###############################################################################
#
sub set_jobid {
    my($self) = shift;
    my($jobid) = shift;

    $self->{'POST'}->{'JOBID'}->{"$jobid"} = 1;

    return;
}

###############################################################################
#
sub del_jobid {
    my($self) = shift;
    my($jobid) = shift;

    delete($self->{'POST'}->{'JOBID'}->{"$jobid"});

    return;
}

###############################################################################
#
sub get_jobid_list {
    my($self) = shift;

    my(@jobid_list) = sort {$a <=> $b} keys(%{$self->{'POST'}->{'JOBID'}});

    return @jobid_list;
}

###############################################################################
#
sub qsub_file {
    my($self) = shift;
    my($file_job) = shift;

    $self->wait_job();

    my($cmd_qsub) = $self->get_command_path('qsub');
    my($file_jobid) = "/tmp/nqs.jobid.$$";
    my($cmd) = "$cmd_qsub $file_job";
#print STDERR "CMD :: $cmd\n";
    system("$cmd > $file_jobid");
    my($jobid) = IO::File->new($file_jobid)->getline();
    ($jobid) = ($jobid =~ /(\d+)/);
#print STDERR "JOBID :: $jobid\n";
    $self->set_jobid($jobid);

    unlink($file_jobid);

    return $jobid;
}

###############################################################################
#
sub qsub_cmd {
    my($self) = shift;
    my(@cmd_list) = @_;

    my($file_job) = "/tmp/nqs_job.$$";
    my($fh) = IO::File->new(">$file_job") || die("Can not open $file_job($!)");
    foreach my$cmd (@cmd_list) {
        $fh->print($cmd, "\n");
    }
    $fh->close();

    chmod(0750, $file_job);
    $self->qsub_file($file_job);
    unlink($file_job);

    return;
}

###############################################################################
package main;
if ($0 eq __FILE__) {
    my($dir) = "/opt/sge/bin/lx24-amd64";
    my($obj) = NqsAxes->new($dir);
    $obj->set_max_job(3);
    foreach my$i (1 .. 20) {
        my(@cmd_list) = ("echo $i",
                         "sleep 2",
                         "echo 'Done.($i)'",
                        );
        $obj->qsub_cmd(@cmd_list);
    }
    $obj->wait_all_job();
}

###############################################################################
1;#
###############################################################################
