#!/usr/bin/perl -s
use strict;
use Digest::MD5;
use File::Path;
use IO::Dir;
use IO::File;
use MBGD::DB;
use MBGD::Taxonomy;
use RECOG::RecogProject;
require 'libMBGDaxes.pl';

package RecogProjectCommon;

###############################################################################
$RecogProjectCommon::TBL_project        = 'recog_project';
$RecogProjectCommon::TBL_user           = 'recog_user';
$RecogProjectCommon::TBL_proj_user      = 'recog_project_user';
$RecogProjectCommon::TBL_proj_domclust  = 'recog_project_domclust';
$RecogProjectCommon::TBL_proj_corealign = 'recog_project_corealign';

###############################################################################
$RecogProjectCommon::AUTH_SYS_ADMIN     = 'auth_sys_admin';
$RecogProjectCommon::AUTH_PRJ_ADMIN     = 'auth_prj_admin';
$RecogProjectCommon::AUTH_SYS_ADD_DATA  = 'auth_sys_add_data';
$RecogProjectCommon::AUTH_PRJ_ADD_DATA  = 'auth_prj_add_data';

###############################################################################
$RecogProjectCommon::ID_PRJ_SYSTEM      = 1;
$RecogProjectCommon::NAME_PRJ_SYSTEM    = 'system';
$RecogProjectCommon::NAME_USR_SYSADM    = 'recogadm';

###############################################################################
$RecogProjectCommon::REGEXP_NAME_VALID_CHAR = 'A-Za-z0-9\_\.\-\@';

###############################################################################
#
sub percent_encoding {
    my($val) = shift;

    #
    $val =~ s#([^0-9a-z])#sprintf("%%%02x",ord($1))#ige;

    return $val;
}

###############################################################################
#
sub get_system_project {
    my($ref) = {};

    $ref->{'ID'}   = $RecogProjectCommon::ID_PRJ_SYSTEM;
    $ref->{'NAME'} = $RecogProjectCommon::NAME_PRJ_SYSTEM;
    $ref->{'SPEC_LIST'} = [];
    $ref->{'SPID_LIST'} = [];
    $ref->{'SPID_LIST_ALL'} = [];

    return $ref;
}
###############################################################################
sub conv_proj_id {
	my($proj_id_str) = @_;
	my($proj_id) = ($proj_id_str =~ /_([0-9a-z]+)$/);
	if (! $proj_id) {
		$proj_id = $proj_id_str;
	}
	$proj_id;
}

###############################################################################
#
sub htdigest {
    my($name) = shift;
    my($realm) = shift;
    my($pass) = shift;

    my($digest_md5) = Digest::MD5->new();
    $digest_md5->add(join(':', $name, $realm, $pass));
    my($hexdigest) = $digest_md5->hexdigest();
    my($htdigest) = join(':', $name, $realm, $hexdigest);

    return $htdigest;
}

###############################################################################
#
sub del_htdigest {
    my($file_htdigest) = shift;
    my($name) = shift;

    my($fhr) = IO::File->new("$file_htdigest");
    my($fhw) = IO::File->new(">$file_htdigest.$$");
    if ($fhr) {
        while (my$line=$fhr->getline()) {
            if ($line =~ /^$name\:/) {
                next;
            }
            $fhw->print($line);
        }
        $fhr->close();
    }
    $fhw->close();

    #
    if (-e "$file_htdigest.bak") {
        unlink("$file_htdigest.bak");
    }
    rename("$file_htdigest",    "$file_htdigest.bak");
    rename("$file_htdigest.$$", "$file_htdigest");

    return;
}

###############################################################################
#
sub add_htdigest {
    my($file_htdigest) = shift;
    my($name) = shift;
    my($realm) = shift;
    my($pass) = shift;

    #
    del_htdigest($file_htdigest, $name);

    #
    my($htdigest) = htdigest($name, $realm, $pass);
    my($fh) = IO::File->new(">>$file_htdigest");
    $fh->print($htdigest, "\n");
    $fh->close();

    return;
}

###############################################################################
#
sub get_htdigest_path {
    my($path_htdigest) = "$ENV{'MBGD_HOME'}/etc/htdigest.mng";

    return $path_htdigest;
}

###############################################################################
#
sub passwd_user {
    my($name) = shift;
    my($passwd) = shift;

    #
    my($file_htdigest) = get_htdigest_path();
    my($key_digest) = $main::KEY_HTDIGEST;
    add_htdigest($file_htdigest, $name, $key_digest, $passwd);

    return;
}

###############################################################################
#
sub create_user {
    my($name) = shift;
    my($passwd) = shift;
    my($organization) = shift;

    #
    passwd_user($name, $passwd);

    #
    my($ref) = {};
    $ref->{'name'} = $name;
    $ref->{'organization'} = $organization;
    set_user_info($name, $ref);

    return $name;
}

###############################################################################
#
sub delete_user {
    my($user) = shift;
    my($name) = shift;

    #
    my($file_htdigest) = get_htdigest_path();
    del_htdigest($file_htdigest, $name);

    #
    my($auth) = RECOG::RecogProject->new();
    my(@id_list) = $auth->get_project_id_list();
    foreach my$proj_id (@id_list) {
        my($sta) = $auth->exists_project_user($user, $proj_id, $name);
        if (!$sta) {
           next;
        }
        $auth->del_project_user($user, $proj_id, $name);
    }

    #
    del_user_info($name);

    return $name;
}

###############################################################################
#
sub exists_user_name {
    my($name) = shift;

    #
    $name =~ s#^\s*##;
    $name =~ s#\s*$##;

    #
    my($file_htdigest) = get_htdigest_path();
    my($fh) = FileHandle->new("$file_htdigest") || return;
    while (my$line=$fh->getline()) {
        my($n, $k, $d) = split(/:/, $line);
        if ($n eq $name) {
            return $n;
        }
    }
    $fh->close();

    return;
}

###############################################################################
#
sub set_user_info {
    my($name) = shift;
    my($ref) = shift;

    #
    my($dir) = "$ENV{'MBGD_HOME'}/etc/users";
    File::Path::mkpath($dir, 0, 0750) if (! -e $dir);

    #
    my($file_user) = "$dir/$name";
    my($fh) = IO::File->new(">$file_user") || return;
    my(@key_list) = keys(%{$ref});
    foreach my$key (@key_list) {
        next if ($key =~ /^\s*$/);

        if ($key eq 'project') {
            foreach my$val (@{$ref->{"$key"}}) {
                $fh->print(join("\t", $key, $val), "\n");
            }
        }
        elsif ($key =~ /^auth/) {
            foreach my$val (keys(%{$ref->{"$key"}})) {
                $fh->print(join("\t", $key, $val), "\n");
            }
        }
        else {
            my($val) = $ref->{"$key"};
            $fh->print(join("\t", $key, $val), "\n");
        }
    }
    $fh->close();

    return;
}

###############################################################################
#
sub get_user_info {
    my($name) = shift;

    #
    my($dir) = "$ENV{'MBGD_HOME'}/etc/users";
    File::Path::mkpath($dir, 0, 0750) if (! -e $dir);

    #
    my($ref) = {};
    my($file_user) = "$dir/$name";
    my($fh) = IO::File->new("$file_user") || return $ref;
    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        $line =~ s#[\r\n]*$##;

        my($k, $v) = split(/\t/, $line);
        if ($k eq 'project') {
            if (!exists($ref->{"$k"})) {
                $ref->{"$k"} = [];
            }
            push(@{$ref->{"$k"}}, $v);
        }
        elsif ($k =~ /^auth_/) {
            if (!exists($ref->{"$k"})) {
                $ref->{"$k"} = {};
            }
            $ref->{"$k"}->{"$v"} = 1;
        }
        else {
            $ref->{"$k"} = $v;
        }
    }
    $fh->close();

    if ($ref->{'name'} =~ /^\s*$/) {
        $ref->{'name'} = $name;
        RecogProjectCommon::set_user_info($name, $ref);
    }

    return $ref;
}

###############################################################################
#
sub del_user_info {
    my($name) = shift;

    #
    my($dir) = "$ENV{'MBGD_HOME'}/etc/users";
    File::Path::mkpath($dir, 0, 0750) if (! -e $dir);

    #
    my($file_user) = "$dir/$name";
    unlink("$file_user");

    return;
}

###############################################################################
#
sub get_user_list_by_project {
    my($user_list_ref) = [];


    return $user_list_ref;
}

###############################################################################
#
sub get_user_list {
    my($user_list_ref) = [];

    my($dir) = "$ENV{'MBGD_HOME'}/etc/users";
    my($dh) = IO::Dir->new("$dir") || return $user_list_ref;
    while (my$file=$dh->read()) {
        next if ($file =~ /^\./);

        push(@{$user_list_ref}, $file);
    }

    return $user_list_ref;
}

###############################################################################
#
sub auth_user_proj {
    my($auth) = shift;
    my($name_user) = shift;
    my($name_proj) = shift;

    my($auth_ref) = RecogProjectCommon::get_user_info($name_user);
    if (!$auth_ref) {
        return;
    }

    #
    if ($auth eq $RecogProjectCommon::AUTH_SYS_ADMIN) {
        $name_proj = $RecogProjectCommon::NAME_PRJ_SYSTEM;
    }
    elsif ($auth eq $RecogProjectCommon::AUTH_SYS_ADD_DATA) {
        $name_proj = $RecogProjectCommon::NAME_PRJ_SYSTEM;
    }

    return $auth_ref->{"$auth"}->{"$name_proj"};
}

###############################################################################
#
sub get_project_domclust {
    my($db) = shift;
    my($project_name) = shift;
    my($user) = shift;

    #
    my(@bind_args) = ($project_name);
    my($tab) = "$RecogProjectCommon::TBL_proj_domclust pd";
    my($where) = "pd.recog_project=?";
    if ($user) {
        $tab .= ", $RecogProjectCommon::TBL_user u, $RecogProjectCommon::TBL_proj_user pu";
        $where .= " and p.id=pu.recog_project_id and pu.recog_user_id=u.id and u.name=?";
        push(@bind_args, $user);
    }
    my($sql) = "select pd.* from $tab where $where";
    my($sth) = $db->prepare($sql);
    $sth->execute(@bind_args);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
sub get_project_corealign {
    my($db) = shift;
    my($domclust_id) = shift;
    my($user) = shift;

    #
    my(@bind_args) = ($domclust_id);
    my($tab) = "$RecogProjectCommon::TBL_proj_domclust pd"
             . ", $RecogProjectCommon::TBL_proj_corealign pc";
    my($where) = "pd.id=pc.recog_project_domclust_id and pd.id=?";
    if ($user) {
        $tab .= ", $RecogProjectCommon::TBL_project p";
        $tab .= ", $RecogProjectCommon::TBL_user u";
        $tab .= ", $RecogProjectCommon::TBL_proj_user pu";
        $where .= " and p.id=pu.recog_project_id and pu.recog_user_id=u.id and u.name=?";
        push(@bind_args, $user);
    }
    my($sql) = "select pd.* from $tab where $where";
    my($sth) = $db->prepare($sql);
    $sth->execute(@bind_args);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
$RecogProjectCommon::CACHE_genome_info = {};
sub loadGenomeInfo {
    my($file) = shift;

#    my($fh) = FileHandle->new("$file") || die("Can not open $file($!)");
#
#    #
#    my($line) = $fh->getline();
#    $line =~ s#[\r\n]*$##;
#    my(@title_list) = split(/\t/, $line);
#
#    #
#    while (my$line=$fh->getline()) {
#        $line =~ s#[\r\n]*$##;
#        my(@g) = split(/\t/, $line);
#        my($ref) = {};
#        foreach my$key (@title_list) {
#            $ref->{"$key"} = shift(@g);
#        }
#        my($sp) = $ref->{'sp'};
#        $RecogProjectCommon::CACHE_genome_info->{"$sp"} = $ref;
#    }
#    $fh->close();

    #
    $RecogProjectCommon::CACHE_genome_info = read_tab_genome($file);

    return;
}
sub getGenomeInfo {
    my($sp) = shift;

#    my($ref) = $RecogProjectCommon::CACHE_genome_info->{"$sp"};
    my($ref) = $RecogProjectCommon::CACHE_genome_info->{'SPEC'}->{"$sp"};
    if (!$ref) {
        my(%null);
        return %null;
    }

    return %{$ref};
}

###############################################################################
#
$RecogProjectCommon::CACHE_gold_field_list = [];
$RecogProjectCommon::CACHE_gold_info = {};
$RecogProjectCommon::CACHE_gold_mbgd_info = {};
sub load_tab_gold {
    my($file_gold) = shift;

    #
    my($fh) = FileHandle->new("$file_gold") || die("Can not open $file_gold($!)");

    #
    my($line) = $fh->getline();
    $line =~ s#[\r\n]*$##;
    push(@{$RecogProjectCommon::CACHE_gold_field_list}, split(/\t/, $line));

    #
    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;
        my(@g) = split(/\t/, $line);
        my($ref) = {};
        foreach my$key (@{$RecogProjectCommon::CACHE_gold_field_list}) {
            $ref->{"$key"} = shift(@g);
        }
        my($goldstamp) = $ref->{'goldstamp'};

        $RecogProjectCommon::CACHE_gold_info->{"$goldstamp"} = $ref;
    }
    $fh->close();

    return;
}
sub load_tab_gold_mbgd {
    my($file_gold_mbgd) = shift;

    #
    my($fh) = FileHandle->new("$file_gold_mbgd") || die("Can not open $file_gold_mbgd($!)");

    #
    my($line) = $fh->getline();
    $line =~ s#[\r\n]*$##;
    my(@title_list) = split(/\t/, $line);

    #
    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;
        my(@g) = split(/\t/, $line);
        my($ref) = {};
        foreach my$key (@title_list) {
            $ref->{"$key"} = shift(@g);
        }
        my($sp) = $ref->{'sp'};
        $RecogProjectCommon::CACHE_gold_mbgd_info->{"$sp"} = $ref;
    }
    $fh->close();

    return;
}
sub loadGoldInfo {
    my($file_gold) = shift;
    my($file_gold_mbgd) = shift;

    load_tab_gold($file_gold);
    load_tab_gold_mbgd($file_gold_mbgd);

    return;
}
sub getGoldFieldList {
    my(@fieldList);

    @fieldList = @{$RecogProjectCommon::CACHE_gold_field_list};

    return @fieldList;
}
sub getGoldInfo {
    my($ref) = {};
    foreach my$sp (keys(%{$RecogProjectCommon::CACHE_gold_mbgd_info})) {
         my($goldstamp) = $RecogProjectCommon::CACHE_gold_mbgd_info->{"$sp"}->{'goldstamp'};
        $ref->{"$sp"} = $RecogProjectCommon::CACHE_gold_info->{"$goldstamp"};
    }

    return $ref;
}

###############################################################################
#
sub parse_gene_prop_head {
    my($line) = shift;

    my($sp, $name, $val_head) = split(/\t/, $line);
    my($name, $data_type) = ($val_head =~ /([^\(\)]+)\((.+)\)/);
    my($is_multi) = ($data_type =~ s#\s*\,\s*multi\s*$##i);
    my($elm_ext);
    if ($data_type =~ /(enum)\(([^\(\)]+)\)/i) {
        $data_type = $1;
        $elm_ext   = $2;
    }

    my($ent) = {};
    $ent->{'NAME'}     = $name;
    $ent->{'TYPE'}     = $data_type;
    $ent->{'ELM_EXT'}  = $elm_ext;
    $ent->{'IS_MULTI'} = $is_multi;

    return $ent;
}

###############################################################################
#
sub parse_gene_prop {
    my($val) = shift;

    my($val_head, @val_list) = split(/[\r\n]+/, $val);

    my($gene_prop_ref) = {};
    $gene_prop_ref->{'HEAD'} = parse_gene_prop_head($val_head);
    $gene_prop_ref->{'SPNAME'} = {};
    $gene_prop_ref->{'LIST'} = [];
    foreach my$line (@val_list) {
        if ($line =~ /^\s*#/) {
            next;
        }

        my($sp, $name, $val) = split(/\t/, $line);
        foreach my$v (split(/\;/, $val)) {
            my($ent) = {};
            $ent->{'sp'}   = $sp;
            $ent->{'name'} = $name;
            $ent->{'val'}  = $v;
            $gene_prop_ref->{'SPNAME'}->{"$sp"}->{"$name"} = 1;
            push(@{$gene_prop_ref->{'LIST'}}, $ent);
        }
    }

    return $gene_prop_ref;
}

###############################################################################
#
sub append_gene_prop {
    my($db) = shift;
    my($uid) = shift;
    my($gene_prop_ref) = shift;

    #
    my($head_ref)  = $gene_prop_ref->{'HEAD'};
    my($prop_name) = $head_ref->{'NAME'};
    my($type)      = $head_ref->{'TYPE'};

    #
    my($tab) = 'gene_prop';
    my($sql) = "select sp from $tab where sp=? and name=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    my(@spec_list) = sort(keys(%{$gene_prop_ref->{'SPNAME'}}));
    foreach my$spec (@spec_list) {
        my(@name_list) = sort(keys(%{$gene_prop_ref->{'SPNAME'}->{"$spec"}}));
        foreach my$name (@name_list) {
            $sth->execute($spec, $name, $prop_name);
            my($n) = $sth->rows();
            if ($n != 0) {
                delete($gene_prop_ref->{'SPNAME'}->{"$spec"}->{"$name"});
            }
        }
    }

    #
    my($n) = 0;
    my($filename) = "$ENV{'MBGD_HOME'}/work/tmp_gene_prop.$$";
    my($fh) = IO::File->new(">$filename") || die("Can not open $filename($!)");
    foreach my$ent (@{$gene_prop_ref->{'LIST'}}) {
        my($spec) = $ent->{'sp'};
        my($name) = $ent->{'name'};
        if (!exists($gene_prop_ref->{'SPNAME'}->{"$spec"}->{"$name"})) {
            next;
        }

        my($c_val) = '\N';
        my($n_val) = '\N';
        if ($type =~ /^(char|enum|hierarchy)/i) {
            $c_val = $ent->{'val'};
        }
        else {
            $n_val = $ent->{'val'};
        }
        $fh->print(join("\t", $uid, $spec, $name, $prop_name, $c_val, $n_val), "\n");
        $n++;
    }
    $fh->close();

    if ($n == 0) {
        # Can not found uniq data
        unlink($filename);
        return;
    }

    #
    my($tab) = 'gene_prop';
    my($cols) = 'recog_user_id, sp, name, prop_name, c_val, n_val';
    my($sql) = "load data local infile '$filename' into table $tab fields terminated by '\t' ($cols)";
    $db->execute($sql);

    #
#    unlink($filename);

    return;
}

###############################################################################
#
sub overwrite_gene_prop {
    my($db) = shift;
    my($uid) = shift;
    my($gene_prop_ref) = shift;

    my($head_ref) = $gene_prop_ref->{'HEAD'};
    my($prop_name)  = $head_ref->{'NAME'};
    my($type1)      = $head_ref->{'TYPE1'};

    my($tab) = 'gene_prop';
    my($sql) = "delete from $tab where sp=? and name=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    my(@spec_list) = sort(keys(%{$gene_prop_ref->{'SPNAME'}}));
    foreach my$spec (@spec_list) {
        my(@name_list) = sort(keys(%{$gene_prop_ref->{'SPNAME'}->{"$spec"}}));
        foreach my$name (@name_list) {
            $sth->execute($spec, $name, $prop_name);
        }
    }

    #
    append_gene_prop($db, $uid, $gene_prop_ref);

    return;
}

###############################################################################
#
sub replace_gene_prop {
    my($db) = shift;
    my($uid) = shift;
    my($gene_prop_ref) = shift;

    my($head_ref) = $gene_prop_ref->{'HEAD'};
    my($prop_name)  = $head_ref->{'NAME'};
    my($type1)      = $head_ref->{'TYPE1'};

    my($tab) = 'gene_prop';
    my($sql) = "delete from $tab where sp=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    my(@spec_list) = sort(keys(%{$gene_prop_ref->{'SPNAME'}}));
    foreach my$spec (@spec_list) {
        $sth->execute($spec, $prop_name);
    }

    #
    append_gene_prop($db, $uid, $gene_prop_ref);

    return;
}

###############################################################################
#
sub delete_gene_prop {
    my($db) = shift;
    my($prop_name) = shift;
    my(@spec_list) = @_;

    my($tab) = 'gene_prop';
    my($sql) = "delete from $tab where prop_name=?";
    if (scalar(@spec_list) != 0) {
        $sql .= " and sp=?";
        my($sth) = $db->prepare($sql);
        foreach my$spec (@spec_list) {
            $sth->execute($prop_name, $spec);
        }
    }
    else {
        my($sth) = $db->prepare($sql);
        $sth->execute($prop_name);

        my($tab) = 'gene_prop_info';
        my($sql) = "delete from $tab where prop_name=?";
        my($sth) = $db->prepare($sql);
        $sth->execute($prop_name);
    }

    return;
}

###############################################################################
#
sub get_gene_prop_info {
    my($db) = shift;
    my($prop_name) = shift;

    my($tab) = 'gene_prop_info';
    my($sql) = "select * from $tab where prop_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($prop_name);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
sub select_gene_prop_info {
    my($db) = shift;

    my(@prop_info_list);
    my($tab) = 'gene_prop_info gpi, gene_prop gp';
    my($col) = 'gpi.*, gp.sp as spec';
    my($where) = 'gpi.prop_name=gp.prop_name';
    my($group) = 'gpi.prop_name, sp';
    my($sql) = "select $col from $tab where $where group by $group";
    my($sth) = $db->execute($sql);
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@prop_info_list, $ref);
    }

    return @prop_info_list;
}

###############################################################################
#
sub insert_gene_prop_info {
    my($db) = shift;
    my($gene_prop_head_ref) = shift;

    my($prop_name) = $gene_prop_head_ref->{'NAME'};
    my($type)      = $gene_prop_head_ref->{'TYPE'};
    my($elm_ext)   = $gene_prop_head_ref->{'ELM_EXT'};
    my($is_multi)  = $gene_prop_head_ref->{'IS_MULTI'};

    my($tab) = 'gene_prop_info';
    my($sql) = "insert $tab (prop_name, type, elm_ext, is_multi) values(?, ?, ?, ?)";
    my($sth) = $db->prepare($sql);
    $sth->execute($prop_name, $type, $elm_ext, $is_multi);

    return;
}

###############################################################################
#
sub update_gene_prop_info {
    my($db) = shift;
    my($gene_prop_head_ref) = shift;

    my($prop_name) = $gene_prop_head_ref->{'NAME'};
    my($type)      = $gene_prop_head_ref->{'TYPE'};
    my($elm_ext)   = $gene_prop_head_ref->{'ELM_EXT'};
    my($is_multi)  = $gene_prop_head_ref->{'IS_MULTI'};

    my($tab) = 'gene_prop_info';
    my($sql) = "update $tab set type=?, elm_ext=?, is_multi=? where prop_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($type, $elm_ext, $is_multi, $prop_name);

    return;
}

###############################################################################
#
sub delete_gene_prop_info {
    my($db) = shift;
    my($prop_name) = shift;
    my(@spec_list) = @_;

    my($tab) = 'gene_prop_info';
    my($sql) = "delete from $tab where prop_name=?";
    if (scalar(@spec_list) != 0) {
        $sql .= " and sp=?";
        my($sth) = $db->prepare($sql);
        foreach my$spec (@spec_list) {
            $sth->execute($prop_name, $spec);
        }
    }
    else {
        my($sth) = $db->prepare($sql);
        $sth->execute($prop_name);
    }

    return;
}

###############################################################################
#
sub parse_cluster_prop_head {
    my($line) = shift;

    my($id, $val_head) = split(/\t/, $line);
    my($name, $data_type) = ($val_head =~ /([^\(\)]+)\((.+)\)/);
    my($is_multi) = ($data_type =~ s#\s*\,\s*multi\s*$##i);
    my($elm_ext);
    if ($data_type =~ /(enum)\(([^\(\)]+)\)/i) {
        $data_type = $1;
        $elm_ext   = $2;
    }

    my($ent) = {};
    $ent->{'NAME'}     = $name;
    $ent->{'TYPE'}     = $data_type;
    $ent->{'ELM_EXT'}  = $elm_ext;
    $ent->{'IS_MULTI'} = $is_multi;

    return $ent;
}

###############################################################################
#
sub parse_cluster_prop {
    my($val) = shift;

    my($val_head, @val_list) = split(/[\r\n]+/, $val);

    my($cluster_prop_ref) = {};
    $cluster_prop_ref->{'HEAD'} = parse_cluster_prop_head($val_head);
    $cluster_prop_ref->{'ID'} = {};
    $cluster_prop_ref->{'LIST'} = [];
    foreach my$line (@val_list) {
        if ($line =~ /^\s*#/) {
            next;
        }

        my($id, $val) = split(/\t/, $line);
        my($clustid, $subclustid) = split(/\./, $id);
        foreach my$v (split(/\;/, $val)) {
            my($ent) = {};
            $ent->{'clustid'}    = $clustid;
            $ent->{'subclustid'} = $subclustid;
            $ent->{'val'}        = $v;
            $cluster_prop_ref->{'ID'}->{"$clustid"}->{"$subclustid"} = 1;
            push(@{$cluster_prop_ref->{'LIST'}}, $ent);
        }
    }

    return $cluster_prop_ref;
}

###############################################################################
#
sub append_cluster_prop {
    my($db) = shift;
    my($uid) = shift;
    my($tabid) = shift;
    my($cluster_prop_ref) = shift;

    #
    my($head_ref)  = $cluster_prop_ref->{'HEAD'};
    my($prop_name) = $head_ref->{'NAME'};
    my($type)      = $head_ref->{'TYPE'};

    #
    my($tab) = 'cluster_prop';
    my($sql) = "select tabid, prop_name from $tab where tabid=? and prop_name=? and clustid=? and subclustid=?";
    my($sth) = $db->prepare($sql);
    my(@clustid_list) = sort(keys(%{$cluster_prop_ref->{'ID'}}));
    foreach my$clustid (@clustid_list) {
        my(@subclustid_list) = sort(keys(%{$cluster_prop_ref->{'ID'}->{"$clustid"}}));
        foreach my$subclustid (@subclustid_list) {
            $sth->execute($tabid, $prop_name, $clustid, $subclustid);
            my($n) = $sth->rows();
            if ($n != 0) {
                delete($cluster_prop_ref->{'ID'}->{"$clustid"}->{"$subclustid"});
            }
        }
    }

    #
    my($n) = 0;
    my($filename) = "$ENV{'MBGD_HOME'}/work/tmp_cluster_prop.$$";
    my($fh) = IO::File->new(">$filename") || die("Can not open $filename($!)");
    foreach my$ent (@{$cluster_prop_ref->{'LIST'}}) {
        my($clustid)    = $ent->{'clustid'};
        my($subclustid) = $ent->{'subclustid'};
        if (!exists($cluster_prop_ref->{'ID'}->{"$clustid"}->{"$subclustid"})) {
            next;
        }

        my($c_val) = '\N';
        my($n_val) = '\N';
        if ($type =~ /^(char|enum|hierarchy)/i) {
            $c_val = $ent->{'val'};
        }
        else {
            $n_val = $ent->{'val'};
        }
        $fh->print(join("\t", $uid, $tabid, $clustid, $subclustid, $prop_name, $c_val, $n_val), "\n");
        $n++;
    }
    $fh->close();

    if ($n == 0) {
        #
        print STDERR "Found no data\n";
#    unlink($filename);
#        return;
    }

    #
    my($tab) = 'cluster_prop';
    my($cols) = 'recog_user_id, tabid, clustid, subclustid, prop_name, c_val, n_val';
    my($sql) = "load data local infile '$filename' into table $tab fields terminated by '\t' ($cols)";
    $db->execute($sql);

    #
#    unlink($filename);

    return;
}

###############################################################################
#
sub overwrite_cluster_prop {
    my($db) = shift;
    my($uid) = shift;
    my($tabid) = shift;
    my($cluster_prop_ref) = shift;

    my($head_ref) = $cluster_prop_ref->{'HEAD'};
    my($prop_name)  = $head_ref->{'NAME'};
    my($type1)      = $head_ref->{'TYPE1'};

    my($tab) = 'cluster_prop';
    my($sql) = "delete from $tab where tabid=? and clustid=? and subclustid=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    my(@clustid_list) = sort(keys(%{$cluster_prop_ref->{'ID'}}));
    foreach my$clustid (@clustid_list) {
        my(@subclustid_list) = sort(keys(%{$cluster_prop_ref->{'ID'}->{"$clustid"}}));
        foreach my$subclustid (@subclustid_list) {
            $sth->execute($tabid, $clustid, $subclustid, $prop_name);
        }
    }

    #
    append_cluster_prop($db, $uid, $tabid, $cluster_prop_ref);

    return;
}

###############################################################################
#
sub replace_cluster_prop {
    my($db) = shift;
    my($uid) = shift;
    my($tabid) = shift;
    my($cluster_prop_ref) = shift;

    my($head_ref) = $cluster_prop_ref->{'HEAD'};
    my($prop_name)  = $head_ref->{'NAME'};
    my($type1)      = $head_ref->{'TYPE1'};

    my($tab) = 'cluster_prop';
    my($sql) = "delete from $tab where tabid=? and clustid=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    my(@clustid_list) = sort(keys(%{$cluster_prop_ref->{'ID'}}));
    foreach my$clustid (@clustid_list) {
        $sth->execute($tabid, $clustid, $prop_name);
    }

    #
    append_cluster_prop($db, $uid, $tabid, $cluster_prop_ref);

    return;
}

###############################################################################
#
sub delete_cluster_prop {
    my($db) = shift;
    my($tabid) = shift;
    my($prop_name) = shift;
    my(@clustid_list) = @_;

    my($tab) = 'cluster_prop';
    my($sql) = "delete from $tab where tabid=? and prop_name=?";
    if (scalar(@clustid_list) != 0) {
        $sql .= " and clustid=?";
        my($sth) = $db->prepare($sql);
        foreach my$clustid (@clustid_list) {
            $sth->execute($tabid, $prop_name, $clustid);
        }
    }
    else {
        my($sth) = $db->prepare($sql);
        $sth->execute($tabid, $prop_name);

        my($tab) = 'cluster_prop_info';
        my($sql) = "delete from $tab where tabid=? and prop_name=?";
        my($sth) = $db->prepare($sql);
        $sth->execute($tabid, $prop_name);
    }

    return;
}

###############################################################################
#
sub get_cluster_prop_info {
    my($db) = shift;
    my($tabid) = shift;
    my($prop_name) = shift;

    my($tab) = 'cluster_prop_info';
    my($sql) = "select * from $tab where tabid=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid, $prop_name);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
sub select_cluster_prop_info {
    my($db) = shift;
    my($tabid) = shift;

    my(@prop_info_list);
    my($tab) = 'cluster_prop_info';
    my($sql) = "select * from $tab where tabid=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid);
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@prop_info_list, $ref);
    }

    return @prop_info_list;
}

###############################################################################
#
sub insert_cluster_prop_info {
    my($db) = shift;
    my($tabid) = shift;
    my($cluster_prop_head_ref) = shift;

    my($prop_name) = $cluster_prop_head_ref->{'NAME'};
    my($type)      = $cluster_prop_head_ref->{'TYPE'};
    my($elm_ext)   = $cluster_prop_head_ref->{'ELM_EXT'};
    my($is_multi)  = $cluster_prop_head_ref->{'IS_MULTI'};

    my($tab) = 'cluster_prop_info';
    my($sql) = "insert $tab (tabid, prop_name, type, elm_ext, is_multi) values(?, ?, ?, ?, ?)";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid, $prop_name, $type, $elm_ext, $is_multi);

    return;
}

###############################################################################
#
sub update_cluster_prop_info {
    my($db) = shift;
    my($tabid) = shift;
    my($cluster_prop_head_ref) = shift;

    my($prop_name) = $cluster_prop_head_ref->{'NAME'};
    my($type)      = $cluster_prop_head_ref->{'TYPE'};
    my($elm_ext)   = $cluster_prop_head_ref->{'ELM_EXT'};
    my($is_multi)  = $cluster_prop_head_ref->{'IS_MULTI'};

    my($tab) = 'cluster_prop_info';
    my($sql) = "update $tab set type=?, elm_ext=?, is_multi=? where tabid=? and prop_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($type, $elm_ext, $is_multi, $tabid, $prop_name);

    return;
}

###############################################################################
#
sub delete_cluster_prop_info {
    my($db) = shift;
    my($tabid) = shift;
    my($prop_name) = shift;
    my(@clustid_list) = @_;

    my($tab) = 'cluster_prop_info';
    my($sql) = "delete from $tab where tabid=? and prop_name=?";
    if (scalar(@clustid_list) != 0) {
        $sql .= " and clustid=? and subclustid=?";
        my($sth) = $db->prepare($sql);
        foreach my$id (@clustid_list) {
            my($clustid, $subclustid) = split(/\./, $id);
            $sth->execute($tabid, $prop_name, $clustid, $subclustid);
        }
    }
    else {
        my($sth) = $db->prepare($sql);
        $sth->execute($tabid, $prop_name);
    }

    return;
}

###############################################################################
#
sub list_gene_set {
    my($db) = shift;

    my(@list_gene_set);
    my($tab) = 'gene_set';
    my($sql) = "select set_name, set_size, set_spec from $tab";
    my($sth) = $db->prepare($sql);
    $sth->execute();
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@list_gene_set, $ref);
    }

    return @list_gene_set;
}

###############################################################################
#
sub get_gene_set {
    my($db) = shift;
    my($set_name) = shift;

    my(@list_gene_set);
    my($tab) = 'gene_set';
    my($sql) = "select * from $tab where set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($set_name);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
sub insert_gene_set {
    my($db) = shift;
    my($uid) = shift;
    my($set_name) = shift;
    my($set_val) = shift;
    my($set_size) = shift;
    my($set_spec) = shift;

    my($tab) = 'gene_set';
    my($sql) = "insert $tab (recog_user_id, set_name, set_val, set_size, set_spec) values (?, ?, ?, ?, ?)";
    my($sth) = $db->prepare($sql);
    $sth->execute($uid, $set_name, $set_val, $set_size, $set_spec);

    return;
}

###############################################################################
#
sub update_gene_set {
    my($db) = shift;
    my($uid) = shift;
    my($set_name) = shift;
    my($set_val) = shift;
    my($set_size) = shift;
    my($set_spec) = shift;

    my($tab) = 'gene_set';
    my($sql) = "update $tab set recog_user_id=?, set_val=?, set_size=?, set_spec=? where set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($uid, $set_val, $set_size, $set_spec, $set_name);

    return;
}

###############################################################################
#
sub delete_gene_set {
    my($db) = shift;
    my($set_name) = shift;

    my($tab) = 'gene_set';
    my($sql) = "delete from $tab where set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($set_name);

    return;
}

###############################################################################
#
sub list_cluster_set {
    my($db) = shift;
    my($tabid) = shift;

    my(@list_cluster_set);
    my($tab) = 'cluster_set';
    my($sql) = "select set_name, set_size from $tab where tabid=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid);
    while (my$ref=$sth->fetchrow_hashref()) {
        push(@list_cluster_set, $ref);
    }

    return @list_cluster_set;
}

###############################################################################
#
sub get_cluster_set {
    my($db) = shift;
    my($tabid) = shift;
    my($set_name) = shift;

    my(@list_cluster_set);
    my($tab) = 'cluster_set';
    my($sql) = "select * from $tab where tabid=? and set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid, $set_name);
    if ($sth->rows() == 0) {
        return;
    }
    my($ref) = $sth->fetchrow_hashref();

    return $ref;
}

###############################################################################
#
sub insert_cluster_set {
    my($db) = shift;
    my($uid) = shift;
    my($tabid) = shift;
    my($set_name) = shift;
    my($set_val) = shift;
    my($set_size) = shift;

    my($tab) = 'cluster_set';
    my($sql) = "insert $tab (recog_user_id, tabid, set_name, set_val, set_size) values (?, ?, ?, ?, ?)";
    my($sth) = $db->prepare($sql);
    $sth->execute($uid, $tabid, $set_name, $set_val, $set_size);

    return;
}

###############################################################################
#
sub update_cluster_set {
    my($db) = shift;
    my($uid) = shift;
    my($tabid) = shift;
    my($set_name) = shift;
    my($set_val) = shift;
    my($set_size) = shift;

    my($tab) = 'cluster_set';
    my($sql) = "update $tab set recog_user_id=?, set_val=?, set_size=? where tabid=? and set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($uid, $set_val, $set_size, $tabid, $set_name);

    return;
}

###############################################################################
#
sub delete_cluster_set {
    my($db) = shift;
    my($tabid) = shift;
    my($set_name) = shift;

    my($tab) = 'cluster_set';
    my($sql) = "delete from $tab where tabid=? and set_name=?";
    my($sth) = $db->prepare($sql);
    $sth->execute($tabid, $set_name);

    return;
}

###############################################################################
#
sub print_select_project_species {
    my($proj_ref) = shift;
    my($opt) = shift;

    #
    my(%derive_spec);
    my($derive_project_id) = $opt->{'derive'};
    if ($derive_project_id) {
        my($auth) = RECOG::RecogProject->new();
        my($derive_proj_ref) = $auth->get_project(undef, $derive_project_id);
        foreach my$spec (@{$derive_proj_ref->{'SPEC_LIST'}}) {
            $derive_spec{"$spec"} = 1;
        }
    }

    #
    my(%sta_spid);
    if ($proj_ref) {
        foreach my$spid (@{$proj_ref->{'PROP'}->{'spid_list'}}) {
            $sta_spid{"$spid"} = 1;
        }
    }

    #
    my($file_tab_genome) = "$ENV{'MBGD_HOME'}/etc/'
                         . $main::NAME_PUBLIC_SERVER
                         . '/tab_genome.txt";
    my($tab_genome_ref) = read_tab_genome($file_tab_genome);
    my($tab_genome_local_ref) = read_tab_genome_local();

    #
    my(@spid_all) = keys(%{$tab_genome_ref->{'SPID'}});
    if (exists($proj_ref->{'SPID_LIST_ALL'})) {
        @spid_all = @{$proj_ref->{'SPID_LIST_ALL'}};
    }
    else {
        #
        my(@spid_local) = sort(keys(%{$tab_genome_local_ref->{'SPID'}}));
        push(@spid_all, @spid_local);
    }

    #
    my($dir_tax) = "$ENV{'MBGD_HOME'}/database.work";
    my($tax_ref) = MBGD::Taxonomy->new($dir_tax);
    @spid_all = $tax_ref->sortByTaxonomy(@spid_all);

    print qq{<select name="project_species" size="20" multiple="multiple">\n};
    foreach my$spid (@spid_all) {
        my($genome_ref) = $tab_genome_ref->{'SPID'}->{"$spid"};
        if (!$genome_ref) {
           $genome_ref = $tab_genome_local_ref->{'SPID'}->{"$spid"};
        }
        my($spec) = $genome_ref->{'sp'};

        my($sta) = '';
        if ($sta_spid{"$spid"}) {
            $sta = 'selected';
        }
        elsif ($derive_project_id) {
            next if (!$derive_spec{"$spec"});
        }

        my($orgname) = $genome_ref->{'orgname'};
        my($strain)  = $genome_ref->{'strain'};
        my($spec_name) = "[$spec] $orgname";
        if ($strain) {
            $spec_name .= "($strain)";
        }
        print qq{<option value="$spec" $sta>$spec_name</option>\n};
    }
    print qq{</select>\n};

    return;
}

###############################################################################
#
sub read_tab_genome_local {
    #
    my($ref) = {};
    $ref->{'SPEC'} = {};
    $ref->{'SPID'} = {};

    #
    my($dir) = "$ENV{'MBGD_HOME'}/species";
    my($dh) = IO::Dir->new("$dir") || return $ref;
    while (my$spid=$dh->read()) {
        next if ($spid !~ /^gu\d+$/);

        my($ent) = {};
        my($filename) = "$dir/$spid/gm/genome.txt";
        my($fh) = IO::File->new("$filename") || next;
        while (my$line=$fh->getline()) {
            $line =~ s#[\r\n]*$##;

            my($k, $v) = split(/\t/, $line);
            $ent->{"$k"} = $v;
        }
        $ent->{'spid'} = $spid;
        $fh->close();

        my($spec) = $ent->{'sp'};

        $ref->{'SPEC'}->{"$spec"} = $ent;
        $ref->{'SPID'}->{"$spid"} = $ent;
    }

    return $ref;
}

###############################################################################
#
sub read_tab_genome {
    my($file_genome) = shift;

    #
    my($ref) = {};
    $ref->{'SPEC'} = {};
    $ref->{'SPID'} = {};

    #
    my($fh) = IO::File->new("$file_genome") || return $ref;
    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;

        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        my(@d) = split(/\t/, $line);
        my($spid)    = $d[4];
        my($spec)    = $d[5];
        my($ent) = {};
        $ent->{'spid'}    = $spid;
        $ent->{'sp'}      = $spec;
        $ent->{'abbrev'}  = $d[6];
        $ent->{'orgname'} = $d[7];
        $ent->{'strain'}  = $d[8];
        $ent->{'taxid'}   = $d[9];

        $ref->{'SPEC'}->{"$spec"} = $ent;
        $ref->{'SPID'}->{"$spid"} = $ent;
    }
    $fh->close();

    return $ref;
}

###############################################################################
# read select-options for base clustering
sub read_bc_sel_conf {
    my($filename) = shift;

    my($ref) = {};
    my($fh) = IO::File->new("$filename") || return $ref;
    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;

        my($k, $v) = split(/\t/, $line);
        $ref->{"$k"} = $v;
    }
    $fh->close();

    return $ref;
}

###############################################################################
# save select-options for base clustering
sub save_bc_sel_conf {
    my($filename) = shift;
    my($ref) = shift;

    my(@key_list) = ( '-di',
                      '-SCORE',
                      '-EVAL',
                      '-PAM',
                      '-IDENT',
                      '-BESTHIT',
                      '-RATIOCUT',
                    );
    my($fh) = IO::File->new(">$filename") || return;
    foreach my$key (@key_list) {
        my($val) = $ref->{"$key"};
        $fh->print(join("\t", $key, $val), "\n");
    }
    $fh->close();

    return;
}

###############################################################################
# read domclust-options for base clustering
sub read_bc_dom_conf {
    my($filename) = shift;

    my($ref) = {};
    my($fh) = IO::File->new("$filename") || return $ref;
    while (my$line=$fh->getline()) {
        $line =~ s#[\r\n]*$##;

        my($k, $v) = split(/\t/, $line);
        $ref->{"$k"} = $v;
    }
    $fh->close();

    return $ref;
}

###############################################################################
# save domclust-options for base clustering
sub save_bc_dom_conf {
    my($filename) = shift;
    my($ref) = shift;

    my(@key_list) = ( '-S',
                      '-c',
                      '-d',
                      '-ci',
                      '-m',
                      '-mr',
                      '-C',
                      '-V',
                      '-n',
                      '-ne',
                      '-p',
                      '-HO',
                      '-H',
                      '-ai',
                      '-ao',
                      '-Ohorizweight',
                    );
    my($fh) = IO::File->new(">$filename") || return;
    foreach my$key (@key_list) {
        my($val) = $ref->{"$key"};
        $fh->print(join("\t", $key, $val), "\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub print_error_msg {
    my($msg) = shift;

    print "Content-type: text/html\n";
    print "\n";

    print <<EOB;
<html>
<head>
</head>

<body>
<h1>ERROR</h1>
$msg
</body>
</html>
EOB
    return;
}

###############################################################################
#
sub print_status {
    my($sta) = shift;
    my($msg) = shift;

    my($sec, $min, $hour, $mday, $mon, $year) = localtime(time());
    $year += 1900;
    $mon++;
    my($date) = sprintf("%04d-%02d-%02d %02d:%02d:%02d", $year, $mon, $mday,
                                                         $hour, $min, $sec);

    #
    print "Content-type: text/plain\n";
    print "\n";

    print '#STATUS='      . $sta . "\n";
    print '#STATUS2='     . $msg . "\n";
    print '#EXEC_SERVER=' . $ENV{'SERVER_NAME'} . "\n";
    print '#EXEC_PORT='   . $ENV{'SERVER_PORT'} . "\n";
    print '#EXEC_DATE='   . $date . "\n";

    return;
}

###############################################################################
#
sub load_base_cluster_conf {
    my($filename) = shift;

    my($ref) = {};
    my($fh) = IO::File->new("$filename");
    if (!$fh) {
        return;
    }

    while (my$line=$fh->getline()) {
        next if ($line =~ /^\s*$/);
        next if ($line =~ /^\s*#/);

        $line =~ s#[\r\n]*$##;

        my($k, $v) = split(/\t/, $line);
        $ref->{"$k"} = $v;
    }
    $fh->close();

    return $ref;
}

###############################################################################
#
sub save_base_cluster_conf {
    my($filename) = shift;
    my($ref) = shift;

    my($fh) = IO::File->new(">$filename");
    my(@key_list) = sort(keys(%{$ref}));
    foreach my$key (@key_list) {
        next if ($key =~ /^\s*$/);
        next if ($key =~ /^btn_/);
        next if ($key =~ /^project_id/);
        next if ($key =~ /^base_cluster_conf_id/);

        my($val) = $ref->{"$key"};
        $fh->print(join("\t", $key, $val), "\n");
    }
    $fh->close();

    return;
}

###############################################################################
#
sub get_dir_cluster_conf {
    my($project_id) = shift;
    my($dir_database) = shift;

    if (!$dir_database) {
        $dir_database = sprintf("%s/database", $ENV{'RECOG_HOME'});
    }
    my($dir_base_cluster) = sprintf("%s/base_cluster/%d.d", $dir_database,
                                                            $project_id);

    return $dir_base_cluster;
}

###############################################################################
#
sub get_file_cluster_conf {
    my($project_id) = shift;
    my($base_cluster_conf_id) = shift;
    my($dir_database) = shift;

    my($dir_base_cluster) = get_dir_cluster_conf($project_id, $dir_database);
    my($file_conf) = sprintf("%s/%d.conf", $dir_base_cluster,
                                           $base_cluster_conf_id);

    return $file_conf;
}

###############################################################################
#
sub get_cluster_conf_id_list {
    my($project_id) = shift;
    my($dir_database) = shift;

    #
    my(@id_list);
    my($dir_base_cluster) = get_dir_cluster_conf($project_id, $dir_database);
    my($dh) = IO::Dir->new($dir_base_cluster);
    if (!$dh) {
        print STDERR "Can not open $dir_base_cluster($!)\n";
        return;
    }
    while (my$file=$dh->read()) {
        if ($file =~ /^(\d+)\.conf$/) {
            my($id) = $1;
            push(@id_list, $id);
        }
    }

    return @id_list;
}

###############################################################################
#
sub get_public_species {
    my(@species);

    my($dbname) = $main::DBNAME_MBGD;
    my($db) = MBGD::DB->new($dbname);
    my($tab) = 'attribute';
    my($where) = "category='genome' and name='species_public'";
    my($sql) = "select * from $tab where $where";
    my($sth) = $db->execute($sql);
    if ($sth->rows() != 0) {
        my($attr_ref) = $sth->fetchrow_hashref();
        @species = split(/,/, $attr_ref->{'value'});
    }

    return @species;
}

###############################################################################
#
sub get_baseClusterID_by_clusterID {
    my($cluster_id) = shift;

    if (!$cluster_id) {
        return;
    }

    #
    my($base_cluster_id);
    my($dbname) = $main::DBNAME_RECOG;
    my($db) = MBGD::DB->new($dbname);
    my($tab) = 'cluster_tables_idx';
    my($where) = "clusterID='$cluster_id'";
    my($sql) = "select * from $tab where $where";
    my($sth) = $db->execute($sql);
    if ($sth->rows() != 0) {
        my($ref) = $sth->fetchrow_hashref();
        ($base_cluster_id) = ($ref->{'cmd'} =~ /baseClusterID=(\S+)/);
    }

    return $base_cluster_id;
}

###############################################################################
1;#
###############################################################################
