#!/usr/local/bin/perl

package Sequence;
use Sequence::CodonTable;
use FileHandle;

sub new {
	my($class) = @_;
	my($this) = {};
	$this->{dir} = 1;
	bless $this, $class;
	$this;
}
sub setseq {
	my($this, $seq) = @_;
	$this->{seq} = $seq;
}
sub setname {
	my($this, $name, $title) = @_;
	$this->{name} = $name;
	$this->{title} = $title;
}
sub settype {
	my($this, $type) = @_;
	if ($type) {
		$this->{type} = $type;
	} else {
		$this->{type} = $this->infer_type;
	}
}
sub gettype {
	my($this) = @_;
	if (! $this->{type}) {
		$this->settype;
	}
	return $this->{type};
}
sub infer_type {
	my($this) = @_;
	return if (! $this->{seq});
	my %comp = $this->composition;
	my($nt);
	foreach $c (keys %comp) {
		if ($c =~ /[atgcu]/i) {
			$nt += $comp{$c}
		}
	}
	if ($nt / $this->length >= 0.8) {
		'nt';
	}  else {
		'aa';
	}
}
sub catseq {
	my($this, $seq) = @_;
	if (! ref($seq)) {
		$this->{seq} .= $seq;
	} elsif (ref($seq) eq 'Sequence') {
		$this->{seq} .= $seq->getseq;
	}
}
sub getseq {
	my($this) = @_;
	if ($this->{dir} < 0) {
		return &calc_complement($this->{seq});
	} else {
		return $this->{seq};
	}
}
sub getname {
	my($this) = @_;
	$this->{name};
}
sub length {
	my($this) = @_;
	$this->{length} = length($this->{seq});
	return $this->{length};
}
sub complement {
	my($this) = @_;
	$this->{dir} *= -1;
}
sub get_subseq {
	my($this, $from, $to, $dir) = @_;
	my $subseq = substr($this->{seq}, $from, ($to-$from+1));
	if ($dir < 0) {
		$subseq = &get_complement($subseq)
	}
	$subseq;
}

sub get_complement {
	my($this) = @_;
	&calc_complement($this->{$seq});
}
sub calc_complement {
	my( $org_seq ) = @_;
	my( $new_seq ) = $org_seq;

	$new_seq =~ tr/ACGTURYMKWSBDHVNXacgturymkwsbdhvnx/TGCAAYRKMWSVHDBNXtgcaayrkmwsvhdbnx/;
	$new_seq =  reverse $new_seq;
	return $new_seq;
}

sub translate {
	my ($this, $transl_id) = @_;
	if ($this->gettype ne 'nt') {
		return;
	}
	return &translation($this->getseq, $transl_id);

}
sub translation {
	my($seq, $transl_id) = @_;
	my $trans_seq;
	$transl_id = 1 if (! $transl_id);
	my $codtab = Sequence::CodonTable->new($transl_id);
	my $length = length($seq);
	for (my $i= 0; $i < $length; $i+=3) {
		my $codon = substr($seq, $i, 3);
		$trans_seq .= $codtab->trans($codon);
	} 
	$trans_seq;
}
sub composition {
	my($this) = @_;
	my(%Count);
	foreach $s (split(//, $this->{seq})) {
		$Count{$s}++;
	}
	return %Count;
}

sub print_fasta {
	my($this, $comment, $opt) = @_;
	my $LINELEN = $opt->{linelen} ? $opt->{linelen} : 60;
	my $fh;
	my $seq = $this->getseq;
	if ($opt->{fh}) {
		$fh = $opt->{fh};
	} else {
		$fh = FileHandle->new(">&STDOUT");
	}
	$comment = join(" ", $this->{name}, $this->{title}) if (! $comment);
	if ($comment) {
		print $fh ">", $comment, "\n";
	}
	for (my $i = 0; $i < $this->length; $i+=$LINELEN) {
		print $fh substr($seq, $i, $LINELEN), "\n";
	}
}
sub read_from_fasta {
	my($class, $file) = @_;
	my($seqObj, $seq, @seqSet);
	$fh = FileHandle->new($file) || die "Can't open $file\n";
	while (<$fh>) {
		chomp;
		if (/^>\s*(\S.*)$/) {
			if ($seqObj && $seq) {
				$seqObj->setseq($seq);
			}
			my($comment) = $1;
			my($name, $title) = split(/\s+/, $comment, 2);
			$seqObj = Sequence->new;
			$seqObj->setname($name, $title);
			push(@seqSet, $seqObj);
			$seq = '';
		} else {
			$seq .= $_;
		}
	}
	if ($seqObj && $seq) {
		$seqObj->setseq($seq);
	}
	@seqSet;
}
package main;
if ($0 eq __FILE__) {
	@seq = Sequence->read_from_fasta("/home/uchiyama/test/seq/maliseq");
	foreach $sq (@seq) {
		$sq->print_fasta('', {linelen=>100});
	}
}
1;
